/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelSorter;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.fukurou.system.DateSet;						// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.security.HybsCryptography ;			// 5.7.4.3 (2014/03/28)
import org.opengion.fukurou.model.POIUtil;						// 6.2.2.0 (2015/03/27)
import org.opengion.fukurou.util.FileInfo;						// 6.2.2.0 (2015/03/27)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Arrays;
import java.io.File;
import java.io.FileFilter;
// import java.io.ObjectOutputStream;
// import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * ファイルを検索し、DBTableModel にセットするタグです。
 *
 * ファイルの検索結果は、[WRITABLE],LEVEL,FILE_TYPE,PARENT,NAME,LASTMODIFIED,FILE_LENGTH,RWH,[MD5],[TO_PARENT,TO_NAME],[････]
 * のカラムを持つ DBTableModel にセット されます。このカラムは、固定です。
 * 並び替えについては、このタグで指定しますが、ファイルの選別(where 条件)は、
 * BODY 部に記述する fileWhere タグで指定します。(複数指定可能))
 *
 * [カラム名]      検索するファイルの属性は、以下のカラム名で作成されます。
 *     [WRITABLE]       useWritable=trueで、先頭カラムに、WRITABLE カラムが挿入されます。
 *      LEVEL           ディレクトリを展開する場合のレベル。
 *      FILE_TYPE       ファイル(F)かディレクトリ(D)であるか判定。
 *      PARENT          この抽象パス名の親のパス名文字列を返します。
 *      NAME            この抽象パス名が示すファイルまたはディレクトリの名前を返します。
 *      LASTMODIFIED    最後に変更された時刻を返します。
 *      FILE_LENGTH     ファイルの長さを返します。
 *      RWH             読み込み、書き込み、隠し属性をそれぞれ、ｒ,w,h で表します。
 *     [MD5]            useMD5=trueで、MD5 というカラムを追加したうえで、ファイルのMD5計算を行います。
 *     [TEXT]           useText=trueで、ファイルの内容を文字列にして、TEXTというカラムに設定します。
 *     [TO_PARENT]      useUpdateClms=trueで、fileUpdateタグでCOPYやMOVEを行う時に使用する必須となるカラム(TO_PARENT,TO_NAME)を追加します。
 *     [TO_NAME]        同上
 *     [････]           addClms属性で指定されたカラムを追加します。
 *
 * @og.formSample
 * ●形式：&lt;og:fileQuery from="…" multi="true/false" &gt;
 *             &lt;og:fileWhere … /&gt;
 *              …
 *         &lt;/og:fileQuery&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:fileQuery
 *       from             ○【TAG】検索を開始するファイルまたはディレクトリを指定します(必須)。
 *       multi              【TAG】多段階展開するか、１レベル展開するかどうか[true:多段階/false:１レベル]を指定します(初期値:false:１レベル)。
 *       level              【TAG】多段階展開するレベルを指定します(初期値:100)。
 *       orderBy            【TAG】ソートするカラム名を指定します(一つのみ)。
 *       desc               【TAG】表示順を逆転するかどうか[true/false]を指定します(初期値:false)。
 *       useWritable        【TAG】先頭カラムに、WRITABLE カラムを追加するかどうか[true/false]を指定します(初期値:false)。
 *       useMD5             【TAG】MD5カラムを追加したうえで、MD5計算を行うかどうか[true/false]を指定します(初期値:false)。
 *       useText            【TAG】TEXTカラムを追加したうえで、ファイルの内容を読み込むかどうか[true/false]を指定します(初期値:false)。
 *       useUpdateClms      【TAG】TO_PARENT、TO_NAMEカラムを追加するかどうか[true/false]を指定します(初期値:false)。
 *       addClms            【TAG】検索結果のカラム列に追加するカラム名を、CSV形式で指定します。
 *       fileType           【TAG】選択対象[FILE/DIR]を指定します。下位展開は考慮(multi 属性準拠)されます。
 *       addFrom            【TAG】from属性で指定された基準ファイル/フォルダ自体をリストに追加するかどうか[true/false]を指定します(初期値:true)。
 *       command            【TAG】コマンド (NEW,RENEW)をセットします("NEW" と "RENEW" 時のみ実行する(初期値:NEW))。
 *       maxRowCount        【TAG】(通常は使いません)データの最大読み込み件数を指定します (初期値:DB_MAX_ROW_COUNT[=1000])(0:[無制限])。
 *       displayMsg         【TAG】検索結果を画面上に表示するメッセージリソースIDを指定します (初期値:VIEW_DISPLAY_MSG[=])
 *       overflowMsg        【TAG】検索データが最大検索数をオーバーした場合に表示するメッセージリソースIDを指定します (初期値:MSG0007[検索結果が、制限行数を超えましたので、残りはカットされました])。
 *       notfoundMsg        【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])。
 *       stopZero           【TAG】検索結果が０件のとき処理を続行するかどうか[true/false]を指定します(初期値:false[続行する])
 *       tableId            【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)。
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)。
 *   &gt;   ... Body ...
 *   &lt;/og:fileQuery&gt;
 *
 * ●使用例
 *    ・一般的な属性でファイルの検索を行います。
 *         &lt;og:fileQuery
 *                from    = "d:/webapps/dbdef/jsp/"
 *                multi   = "true"
 *                command = "{&#064;command}"  &gt;
 *            &lt;og:fileWhere endWith=".jsp" /&gt;
 *      &lt;/og:fileQuery&gt;
 *
 *    ・最終変更日で逆順ソートする。対象は、2002/10/01 以降に変更されたファイル。
 *        &lt;og:fileQuery
 *                from    = "d:/webapps/dbdef/jsp/"
 *                multi   = "true"
 *                orderBy = "LASTMODIFIED"
 *                desc    = "true"
 *                command = "{&#064;command}"  &gt;
 *            &lt;og:fileWhere lastModified="20021001000000" /&gt;
 *        &lt;/og:fileQuery&gt;
 *
 * @og.rev 4.0.0.0 (2005/01/31) 内部ロジック改定
 * @og.group その他入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileQueryTag extends QueryTag {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.3.4 (2016/03/11)" ;
	private static final long serialVersionUID = 643420160311L ;

//	private static final String[] SELECT ={ "LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH" };
	// 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	private static final ArraySet<String> SELECT_SET = new ArraySet<>( "LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH" );

	private static final String[] USE_UPDATE_CLM = { "TO_PARENT","TO_NAME" };	// 5.3.4.0 (2011/04/01)

	private transient FileFilter filter	;						// FileWhere で指定したフィルター

	private boolean		multi			;						// 下位層展開ﾌﾗｸﾞ
	private int			level			= 100;							// 展開ﾚﾍﾞﾙ
	private String      from			= HybsSystem.sys( "FILE_URL" );	// 検索起点ﾌｧｲﾙ
	private String		orderBy			;						// 5.3.4.0 (2011/04/01) ソートカラム
	private boolean		desc			;						// 5.3.4.0 (2011/04/01) ソートの方向(true:逆順)
	private String[]	addClms			= new String[0];		// 5.3.4.0 (2011/04/01) 追加カラム配列
	private String[]	defClms			;						// 5.7.4.3 (2014/03/28) 初期値のカラム配列
	private String		fileType		;						// 5.3.4.0 (2011/04/01) 選択対象を指定(FILE,DIR)
	private boolean		useWritable		;						// 5.7.4.3 (2014/03/28) 先頭カラムに、WRITABLE カラムを追加するかどうか[true/false](初期値:false)
	private boolean		useMD5			;						// 5.7.4.3 (2014/03/28) MD5カラムを追加したうえで、MD5計算を行うかどうか[true/false](初期値:false)
	private boolean		useText			;						// 6.2.2.0 (2015/03/27) TEXTカラムを追加したうえで、ファイルの内容を読み込むかどうか[true/false]を指定します(初期値:false)。
	private boolean		useUpdateClms	;						// 5.3.4.0 (2011/04/01) TO_PARENT、TO_NAMEカラムを追加(true:追加)
	private boolean		addFrom			= true;					// 5.3.9.0 (2011/09/01) from属性で指定された基準ファイル/フォルダ自体をリストに追加するかどうか

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public FileQueryTag() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 指定カラムのソート処理機能追加
	 * @og.rev 5.3.5.0 (2011/05/01) 最初のファイルが存在する場合のみ、実行する。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		executeCount = 0;

		table = initDBTable();
		if( maxRowCount < 0 ) {
			maxRowCount	= sysInt( "DB_MAX_ROW_COUNT" ) ;
		}

		// 5.3.5.0 (2011/05/01) 最初のファイルが存在する場合のみ、実行する。
		final File fin = new File( from );
		if( fin.exists() ) {
			execute( fin,0 ) ;

			// 5.3.4.0 (2011/04/01) 指定カラムのソート処理
			if( orderBy != null ) {
				final int clmNo = table.getColumnNo( orderBy );
				final DBTableModelSorter temp = new DBTableModelSorter();
				temp.setModel( table );
				temp.sortByColumn( clmNo,!desc );	// 注意 desc の値と ソート正逆は、反対です。
				table = temp;
			}
		}

		return SKIP_BODY ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 指定カラムのソート処理機能、カラム追加機能、fileType追加
	 * @og.rev 5.3.9.0 (2011/09/01) addFrom属性追加
	 * @og.rev 5.7.4.3 (2014/03/28) useWritable,useMD5属性追加。valClms を defClms に置き換え。
	 * @og.rev 6.2.2.0 (2015/03/27) useText属性追加。
	 */
	@Override
	protected void release2() {
		super.release2();
		multi			= false;
		level			= 100;
		from			= HybsSystem.sys( "FILE_URL" );
		filter			= null;
		orderBy			= null;				// 5.3.4.0 (2011/04/01) ソートカラム
		desc			= false;			// 5.3.4.0 (2011/04/01) 降順ﾌﾗｸﾞ
		addClms			= new String[0];	// 5.3.4.0 (2011/04/01) 追加カラム配列
		defClms			= null;				// 5.7.4.3 (2014/03/28) 初期値のカラム配列
		fileType		= null;				// 5.3.4.0 (2011/04/01) 選択対象を指定(FILE,DIR,ALL)
		useWritable		= false;			// 5.7.4.3 (2014/03/28) 先頭カラムに、WRITABLE カラムを追加するかどうか[true/false](初期値:false)
		useMD5			= false;			// 5.7.4.3 (2014/03/28) MD5カラムを追加したうえで、MD5計算を行うかどうか[true/false](初期値:false)
		useText			= false;			// 6.2.2.0 (2015/03/27) TEXTカラムを追加したうえで、ファイルの内容を読み込むかどうか[true/false]を指定します(初期値:false)。
		useUpdateClms	= false;			// 5.3.4.0 (2011/04/01) TO_PARENT、TO_NAMEカラムを追加(true:追加)
		addFrom			= true;				// 5.3.9.0 (2011/09/01) addFrom属性追加
	}

	/**
	 * FileQuery を実行します。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) fileType の条件に合致する場合だけ、データを作成する。
	 * @og.rev 5.3.7.0 (2011/07/01) フォルダにアクセスできない場合は、null となるのでその対応
	 * @og.rev 5.3.9.0 (2011/09/01) addFrom属性追加
	 *
	 * @param	fin	検索を開始するファイル/ディレクトリ
	 * @param	lvl	階層展開レベル
	 */
	protected void execute( final File fin,final int lvl ) {
		if( ( !multi && lvl > 1 ) || lvl > level ) { return; }	// 階層展開する、しない
		if( executeCount > maxRowCount ) { table.setOverflow( true ); return; }

		final boolean isDIR = fin.isDirectory();

		if( fileType == null ||
			(  isDIR &&  "DIR".equalsIgnoreCase( fileType ) ) ||
			( !isDIR && "FILE".equalsIgnoreCase( fileType ) ) ) {
				// 6.0.2.4 (2014/10/17) RpC:条件テストを繰り返しています。 
				if( addFrom || lvl > 0 ) {
					addFileData( executeCount++,lvl,fin );
				}
		}
		if( isDIR ) {
			final File[] list = fin.listFiles( filter );
			// 5.3.7.0 (2011/07/01) フォルダにアクセスできない場合は、null となる。
			if( list != null ) {
				for( int i=0; i<list.length; i++ ) {
					execute( list[i],lvl+1 );
				}
			}
		}
	}

	/**
	 * 初期化された DBTableModel を返します。
	 *
	 * ここでは、useWritable、useMD5、useUpdateClms、addClms を加味した
	 * DBTableModel と初期値データ(defClms)を作成します。
	 * 以前は、TO_PARENT、TO_NAMEと、addClms 分のみ初期値を持っていましたが、
	 * 5.7.4.3 (2014/03/28)で、先頭カラムのWRITABLEの初期値を、DBColumn の初期値ではなく
	 * 手動設定する必要がある為、すべてのカラム列の初期値を持っておきます。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 指定カラム追加機能追加
	 * @og.rev 5.7.4.3 (2014/03/28) useWritable,useMD5属性追加
	 * @og.rev 6.2.2.0 (2015/03/27) TEXTカラムを追加したうえで、ファイルの内容を読み込むかどうか[true/false]を指定します(初期値:false)。
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @return	 テーブルモデル
	 */
	private DBTableModel initDBTable() {
		final DBTableModel tbl = DBTableModelUtil.newDBTable();

		// 5.7.4.3 (2014/03/28) 以下の処理は、ほぼ全面見直し
//		int size = SELECT.length ;								// 基本カラムの数
		int size = SELECT_SET.size() ;							// 6.4.3.4 (2016/03/11) 基本カラムの数
		if( useWritable   ) { size++ ; }						// WRITABLE カラムを追加
		if( useMD5        ) { size++ ; }						// MD5 カラムを追加
		if( useText       ) { size++ ; }						// 6.2.2.0 (2015/03/27) TEXT カラムを追加
		if( useUpdateClms ) { size += USE_UPDATE_CLM.length; }	// TO_PARENT、TO_NAMEカラムを追加
		size += addClms.length ;								// addClms(追加カラム)数を追加

		// DBTableModel の初期化と、初期値配列の確保
		tbl.init( size );
		defClms = new String[size];

		int ad=0;
		// 先頭は、WRITABLE
		if( useWritable ) {
			final DBColumn dbColumn = getDBColumn( "WRITABLE" );
			defClms[ad] = "1";									// WRITABLE を設定するときは、とりあえず 書き込み許可
			tbl.setDBColumn( ad++,dbColumn );
		}

		// SELECT の 基本カラムの設定。(初期値は不要)
		// 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
		// オリジナルの forEach。カウンタ初期値とラムダ式を与えると、ラムダ式の引数に、カウンタと値が設定される。
		SELECT_SET.forEach( ad , (i,v) -> {
			final DBColumn dbColumn = getDBColumn( v );
			tbl.setDBColumn( i,dbColumn );
		} );


//		// SELECT の 基本カラムの設定。(初期値は不要)
//		for( int i=0; i<SELECT.length; i++ ) {
//			final DBColumn dbColumn = getDBColumn( SELECT[i] );
//			tbl.setDBColumn( ad++,dbColumn );
//		}

		// MD5 カラムを追加。
		if( useMD5 ) {
			final DBColumn dbColumn = getDBColumn( "MD5" );
			defClms[ad] = "";									// ディレクトリの場合は、MD5計算しません。
			tbl.setDBColumn( ad++,dbColumn );
		}

		// 6.2.2.0 (2015/03/27) TEXT カラムを追加
		if( useText ) {
			final DBColumn dbColumn = getDBColumn( "TEXT" );
			defClms[ad] = "";									// ディレクトリの場合は、TEXT計算しません。
			tbl.setDBColumn( ad++,dbColumn );
		}

		// TO_PARENT、TO_NAMEカラムを追加
		if( useUpdateClms ) {
			for( int i=0; i<USE_UPDATE_CLM.length; i++ ) {
				final DBColumn dbColumn = getDBColumn( USE_UPDATE_CLM[i] );
				defClms[ad] = dbColumn.getDefault();			// 初期値を指定しておく
				tbl.setDBColumn( ad++,dbColumn );
			}
		}

		// 追加カラムのaddClmsカラムを追加
		for( int i=0; i<addClms.length; i++ ) {
			final DBColumn dbColumn = getDBColumn( addClms[i] );
			defClms[ad] = dbColumn.getDefault();					// 初期値を指定しておく
			tbl.setDBColumn( ad++,dbColumn );
		}

		return tbl ;
	}

	/**
	 * DBTableModel に、ファイル情報をセットします。
	 * ファイル情報は、[WRITABLE],LEVEL,FILE_TYPE,PARENT,NAME,LASTMODIFIED,FILE_LENGTH,RWH,[MD5],[TO_PARENT,TO_NAME],[････] です。
	 *
	 * useWritable=true の場合、先頭カラムに、WRITABLE カラムを追加します。
	 * useMD5=true の場合、MD5カラムを追加したうえで、MD5計算を行います(ファイルのみ計算します)。
	 * useUpdateClms=true の場合TO_PARENT、TO_NAMEカラムを追加します。
	 * addClms で指定されたカラムをその後ろに追加します。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 指定カラム追加機能追加
	 * @og.rev 5.7.4.3 (2014/03/28) useWritable,useMD5属性追加
	 * @og.rev 6.2.2.0 (2015/03/27) TEXTカラムを追加したうえで、ファイルの内容を読み込むかどうか[true/false]を指定します(初期値:false)。
	 * @og.rev 6.2.3.0 (2015/05/01) POIUtil のメソッド名変更に伴う対応
	 * @og.rev 6.2.4.2 (2015/05/29) POIUtil#extractor の判定方法変更
	 * @og.rev 6.4.2.0 (2016/01/29) DateSet.getDate( String ) を利用するように修正します。
	 *
	 * @param	rowNo	セットする行番号
	 * @param	lvl  	セットするレベル
	 * @param	fin		ファイル情報の元となるファイルオブジェクト
	 */
	private void addFileData( final int rowNo,final int lvl,final File fin ) {
		try {
			final File file = fin.getCanonicalFile();

			final String rwh = (    file.canRead()  ? "r" : "-" )
						+	( file.canWrite() ? "w" : "-" )
						+	( file.isHidden() ? "h" : "-" );

//			final String lastModified = HybsSystem.getDate( file.lastModified(),"yyyyMMddHHmmss" );
			final String lastModified = DateSet.getDate( file.lastModified(),"yyyyMMddHHmmss" );				// 6.4.2.0 (2016/01/29)

			final boolean isF = file.isFile();								// File=true,それ以外=false

			final int size = table.getColumnCount() ;
			String[] data = Arrays.copyOf( defClms,size );				// JDK1.6

			int ad=0;
			if( useWritable ) { ad++ ; }		// 単にひとつ進める。初期値はセット済み。

			// SELECT の 基本カラムの設定
			data[ad++] = String.valueOf( lvl ) ;						// LEVEL
			data[ad++] = isF ? "F" : "D" ;								// FILE_TYPE
			data[ad++] = file.getParent() ;								// PARENT
			data[ad++] = file.getName() ;								// NAME
			data[ad++] = lastModified ;									// LASTMODIFIED
			data[ad++] = isF ? String.valueOf( file.length() ) : "" ;	// FILE_LENGTH
			data[ad++] = rwh ;											// RWH

			// MD5 カラムを追加(ファイルの場合のみ計算します)
			if( useMD5 && isF ) {
				data[ad++] = HybsCryptography.getMD5( file );
			}

			// 6.2.2.0 (2015/03/27) TEXT カラムを追加(ファイルの場合のみ取り込みます)
			if( useText && isF ) {
				final String sufix = FileInfo.getSUFIX( file ) ;
				String val = "";
				try {
					if( POIUtil.isPOI( file ) ) {						// 6.2.4.2 (2015/05/29)
						val = POIUtil.extractor( file );
					}
					else if( "txt,jsp,java,xml,css,js".contains( sufix ) ) {
						val = POIUtil.extractor( file , "UTF-8");				// 6.2.3.0 (2015/05/01)
					}
					else {
						val = POIUtil.extractor( file , "Windows-31J");			// 6.2.3.0 (2015/05/01)
					}
				}
				catch( RuntimeException ex ) {
					// 変換に失敗しても、処理は継続する。
					final String errMsg = "ファイルのテキスト変換に失敗しました。[" + fin + "]"
								+ " ROW=[" + rowNo + "]"
								+ CR + ex.getMessage();
					System.err.println( errMsg );
				}
				data[ad++] = val;
			}

			// useUpdateClms=true 時の TO_PARENT、TO_NAMEカラムや、addClmsの追加カラムは、初期値のみセット
			// 初期値セットは、Arrays.copyOf で、defClms のコピーで完了。

			table.addColumnValues( data );
		}
		catch( IOException ex ) {
			final String errMsg = "正式なファイル名の取得に失敗しました。[" + fin + "]"
						+ " ROW=[" + rowNo + "]"
						+ CR + ex.getMessage();
			throw new HybsSystemException( errMsg,ex );
		}
	}

	/**
	 * 【TAG】ファイルの検索元となるディレクトリを指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.tag ファイルの検索元となるディレクトリを指定します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に、"/" を追加する処理を廃止。
	 * @og.rev 6.4.2.1 (2016/02/05) HybsSystem.url2dir に引数追加。
	 *
	 * @param	url ファイルの検索元となるディレクトリ
	 */
	public void setFrom( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
//		if( furl != null ) {
//			final char ch = furl.charAt( furl.length()-1 );
//			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
//		}
//		furl = StringUtil.urlAppend( from,furl );
//		furl = StringUtil.urlAppend( furl,"." );
//
//		from = HybsSystem.url2dir( furl );
//
		from = HybsSystem.url2dir( from,furl,"." );			// 6.4.2.1 (2016/02/05)
	}

	/**
	 * 【TAG】多段階展開するか、１レベル展開するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 初期値は、false (１レベル) です。
	 *
	 * @param	mlti 多段階展開するか [true:する/false:１レベル]
	 */
	public void setMulti( final String mlti ) {
		multi = nval( getRequestParameter( mlti ),multi );
	}

	/**
	 * 【TAG】多段階展開するレベルを指定します(初期値:100)。
	 *
	 * @og.tag
	 *
	 * @param	lvl 多段階展開するレベル
	 */
	public void setLevel( final String lvl ) {
		level = nval( getRequestParameter( lvl ),level );
	}

	/**
	 * 【TAG】ソートするカラム名を指定します(一つのみ)。
	 *
	 * @og.tag
	 * ソートするカラム名を、"LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH"
	 * から一つ選びます。
	 * これは、複数カラムでのソートはできません。
	 * 逆順にソートする場合は、desc属性を true にセットください。
	 * + をつけても、無効(カラム名がないということ)でエラーになります。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @param	clm ソートするカラム名 (一つのみ、逆順はマイナスを付ける)
	 * @see		#setDesc( String )
	 */
	public void setOrderBy( final String clm ) {
		orderBy = nval( getRequestParameter( clm ),orderBy );

//		if( orderBy != null && ! check( orderBy, SELECT ) ) {
		if( orderBy != null && ! check( orderBy, SELECT_SET ) ) {
			final String errMsg = "指定の orderBy は、指定できません。" + CR
							+ "orderBy=[" + orderBy + "] "				+ CR
//							+ StringUtil.array2csv( SELECT ) + CR ;
//							+ "orderByList=" + String.join( ", " , SELECT ) ;
							+ "orderByList=" + String.join( ", " , SELECT_SET ) ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】表示順を逆転するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * orderBy 属性で指定した表示順を、逆順にするかどうかを指定できます。
	 * 初期値は、false (昇順) です。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	flag 表示順逆順 [逆順:true/正順:false]
	 * @see		#setOrderBy( String )
	 */
	public void setDesc( final String flag ) {
		desc = nval( getRequestParameter( flag ),desc );
	}

	/**
	 * 【TAG】先頭カラムに、WRITABLE カラムを追加するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * ファイル検索結果の１レコード単位に、書き込み許可/禁止属性を付けるには、
	 * カラム列の先頭に、WRITABLE カラムを追加する必要があります。
	 * 初期値は、false (追加しない) です。
	 *
	 * @og.rev 5.7.4.3 (2014/03/28) 新規追加
	 *
	 * @param	flag WRITABLEカラム追加 [true:する/false:しない]
	 */
	public void setUseWritable( final String flag ) {
		useWritable = nval( getRequestParameter( flag ),useWritable );
	}

	/**
	 * 【TAG】MD5カラムを追加したうえで、MD5計算を行うかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * ファイルの改変等をチェックするには、ファイルのハッシュ値を拾う必要があります。
	 * タイムスタンプとサイズ（LASTMODIFIED,FILE_LENGTH）でも、類似の処理は可能ですが、
	 * より、厳密な一致をみるなら、MD5でハッシュした結果を突き合わせるのがベストです。
	 * useMD5=true に設定すると、MD5 というカラムを追加したうえで、MD5計算結果をセットします。
	 * 初期値は、false (追加しない) です。
	 *
	 * @og.rev 5.7.4.3 (2014/03/28) 新規追加
	 *
	 * @param	flag MD5カラム追加 [true:する/false:しない]
	 */
	public void setUseMD5( final String flag ) {
		useMD5 = nval( getRequestParameter( flag ),useMD5 );
	}

	/**
	 * 【TAG】TEXTカラムを追加したうえで、ファイルの内容を読み込むかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * ファイルの内容を取得する場合に、true に設定します。
	 * 初期値は、false (追加しない) です。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) TEXTカラムを追加したうえで、ファイルの内容を読み込むかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @param	flag TEXTカラム追加 [true:する/false:しない]
	 */
	public void setUseText( final String flag ) {
		useText = nval( getRequestParameter( flag ),useText );
	}

	/**
	 * 【TAG】TO_PARENT、TO_NAMEカラムを追加するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * fileUpdateタグでは、ファイルのCOPYやMOVEが出来ますが、そのコピー先、移動先の
	 * ファイルを行ごとに指定する場合、TO_PARENT、TO_NAMEカラムという固定名のカラムが
	 * 必要です。
	 * これを、addClms 属性で指定する代わりに、この属性で、true をセットすることで、
	 * 自動的に追加されます。
	 * 初期値は、false (追加しない) です。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	flag TO_PARENT、TO_NAMEカラム追加 [true:追加する/false:追加しない]
	 * @see		#setAddClms( String )
	 * @see		org.opengion.hayabusa.taglib.FileUpdateTag
	 */
	public void setUseUpdateClms( final String flag ) {
		useUpdateClms = nval( getRequestParameter( flag ),useUpdateClms );
	}

	/**
	 * 【TAG】検索結果のカラム列に追加するカラム名を、CSV形式で指定します。
	 *
	 * @og.tag
	 * デフォルトのカラム名、[WRITABLE],LEVEL,FILE_TYPE,PARENT,NAME,LASTMODIFIED,FILE_LENGTH,RWH,[MD5],[TO_PARENT,TO_NAME]
	 * 以外に、指定のカラム名を追加することが可能です。
	 * これは、ファイル検索結果以外の項目を追加して、データベースに書き込む場合に、利用できます。
	 * 並び順は、デフォルトカラムの後ろに、指定のカラムの順番で付きます。
	 * ここで追加したカラムには、カラムリソースの初期値がセットされます。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	clms 追加するカラム名(CSV形式)
	 * @see		#setUseUpdateClms( String )
	 */
	public void setAddClms( final String clms ) {
		final String tmpClms = nval( getRequestParameter( clms ),null );

		if( tmpClms != null && tmpClms.length() > 0 ) {
			addClms = StringUtil.csv2Array( tmpClms );
		}
	}

	/**
	 * 【TAG】ファイル名が、指定されたファイルタイプ[DIR/FILE]と一致した場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 大文字小文字は区別しません。
	 * ファイルタイプ は、DIR,FILE が指定できます。
	 * DIR は、ディレクトリのみ検索します。(階層下がりも行います)
	 * FILEは、ファイルのみ検索します。(階層下がりも行います)
	 * 引数が null の場合は、追加しません。(つまり、すべてスルーされます。)
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) fileType メソッドで選択対象指定の追加
	 *
	 * @param    str ファイルタイプ [null:スルー/DIR:ディレクトリのみ検索/FILE:ファイルのみ検索]
	 */
	public void setFileType( final String str ) {
		final String tmp = nval( getRequestParameter( str ),fileType );
		if( tmp == null						||
			"DIR".equalsIgnoreCase( tmp )	||
			"FILE".equalsIgnoreCase( tmp ) ) {
				fileType = tmp;
		}
		else {
			// ファイルタイプに不正な値が設定された場合は、エラーになる。
			final String errMsg = "この、fileType 属性には、DIR,FILE 以外は指定できません。["
						+ tmp + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】from属性で指定された基準ファイル/フォルダ自体をリストに追加するかどうか[true/false]を指定します(初期値:true)。
	 * @og.tag
	 * 初期値はtrue(追加する)です。
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) 新規作成
	 *
	 * @param    flag 基準をリストに追加するかどうか [true:追加する/false:追加しない]
	 */
	public void setAddFrom( final String flag ) {
		addFrom = nval( getRequestParameter( flag ),addFrom );
	}

	/**
	 * FileFilterオブジェクトをセットします。
	 * これは、BODY 部に登録した、FileWhereタグによって設定された
	 * ファイルフィルターです。
	 *
	 * @param	filter	オブジェクト
	 */
	protected void setFileFilter( final FileFilter filter ) {
		this.filter = filter;
	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
//	 *
//	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectOutputStreamオブジェクト
//	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
//	 */
//	private void writeObject( final ObjectOutputStream strm ) throws IOException {
//		strm.defaultWriteObject();
//	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
//	 *
//	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
//	 *
//	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectInputStreamオブジェクト
//	 * @see #release2()
//	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
//	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
//	 */
//	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
//		strm.defaultReadObject();
//	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "multi"		,multi	)
				.println( "level"		,level	)
				.println( "from"		,from	)
				.fixForm().toString()
			+ CR
			+ super.toString() ;
	}
}
