/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.TagBuffer;

import static org.opengion.fukurou.util.StringUtil.isNull;				// 6.1.1.0 (2015/01/17)

/**
 * HM エディターは、カラムのデータを時分編集する場合に使用するクラスです。
 * 選択用ピッカーが付属します。
 * 
 * 編集パラメータは、starthour , endhour , minutedivisions , showminutes の順に、
 * CSV形式で与えます。
 * 変数の数が少ない場合は、初期値が設定されます。
 * 初期値は、starthour="6" endhour="20" minutedivisions="4" showminutes="true" です。
 * 
 * ex) 6,19,4,true
 * starthour		：ピッカーの開始時間(0-23)
 * endhour			：ピッカーの終了時間(0-23)
 * minutedivisions	：単位「分」の分割数。4なら15分単位。(2-60)
 * showminutes		：分ピッカーの表示(true/false)
 * 
 * このエディタはeventColumnに対応していません。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 5.6.5.2 (2013/06/21) 新規作成
 * @og.group データ編集
 *
 * @version  5.0
 * @author   Takahashi Masakazu
 * @since    JDK6.0,
 */
public class Editor_HM extends AbstractEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.5.2 (2013/06/21)" ;

	private static final String PIC1 = "<img src=\"../image/clock.png\" class=\"clockpick\"  valuefield=\""; // altは付けない
	private static final String PIC2 = " />";

	private static final String OPT1 = "starthour=\"";
	private static final String OPT2 = "endhour=\"";
	private static final String OPT3 = "minutedivisions=\"";
	private static final String OPT4 = "showminutes=\"";
	private static final String END_ATTR = "\" ";

	private final String options;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Editor_HM() {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		// 4.3.4.4 (2009/01/01)
		options = null;
	}

	/**
	 * コンストラクター。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_HM( final DBColumn clm ) {
		super( clm );
		options = createOptions(clm.getEditorParam());
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_HM( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String tag = new TagBuffer( "input" )
						.add( "name"	, name )
						.add( "id"		, name , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
						.add( "value"	, value )
						.add( "size"	, size1 )
						.add( tagBuffer.makeTag() )
						.makeTag();

		return tag + PIC1 + name + END_ATTR + options + PIC2 ;

	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final String name2 =  name + HybsSystem.JOINT_STRING + row ;

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String tag = new TagBuffer( "input" )
						.add( "name"	, name2 )
						.add( "id"		, name2 , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
						.add( "value"	, value )
						.add( "size"	, size2 )
						.add( tagBuffer.makeTag() )
						.makeTag( row,value );

		return tag + PIC1 + name2 + END_ATTR + options + PIC2 ;

	}

	/**
	 * 編集パラメータから、オプション文字列を作成します。
	 * 編集パラメータは、starthour , endhour , minutedivisions , showminutes の順に、
	 * CSV形式で与えます。
	 * 変数の数が少ない場合は、初期値が設定されます。
	 * 初期値は、starthour="6" endhour="20" minutedivisions="4" showminutes="true" です。
	 *
	 * @param	editPrm  編集パラメータ
	 *
	 * @return	パラメータのオプション文字列
	 * @og.rtnNotNull
	 */
	private String createOptions( final String editPrm ){
		if( editPrm == null ) { return "" ;}
		// 6.0.2.5 (2014/10/31) null でないことがわかっている値の冗長な null チェックがあります。
		final String[] param = editPrm.split( "," ) ;
		final String start = param.length > 0 ? param[0].trim() : "6" ;
		final String end   = param.length > 1 ? param[1].trim() : "20" ;
		final String step  = param.length > 2 ? param[2].trim() : "4" ;
		final String min   = param.length > 3 ? param[3].trim() : "true" ;

		return OPT1 + start + END_ATTR + OPT2 + end + END_ATTR + OPT3 + step + END_ATTR + OPT4 + min + END_ATTR;
	}
}
