/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report;

import org.opengion.hayabusa.common.HybsSystem;

import static org.opengion.fukurou.system.HybsConst.CR ;				// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring
import org.opengion.fukurou.system.LogWriter;
// import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.system.ThrowUtil ;						// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.db.ApplicationInfo;
import org.opengion.fukurou.db.DBUtil;

/**
 * 【DB登録】EXCEL取込機能の最終処理である、テンポラリテーブルから各種アプリ側の
 * 本番テーブルへのデータ取込処理のための、PL/SQL をコールします。
 * 実際の呼び出しは、{ call 帳票IDP.帳票ID( 結果(STATUS),内容(ERR_CODE),PGID,要求番号 ) }
 * という PL/SQL を Call します。
 * 第１引数、第２引数は、OUT属性で、結果(STATUS)とエラー時の内容(ERR_CODE)を返します。
 * 第３引数は、起動元PGIDです。 第４引数は、処理を行う要求番号です。
 * 結果(STATUS)は、正常ならば、0 を返してください。
 *
 * @og.rev 3.8.0.0 (2005/06/07) 新規追加
 * @og.group 帳票システム
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ProgramRun {
	private final StringBuilder errMsg	;

	private final String	SYSTEM_ID	;
	private final String	YKNO		;
	private final String	LISTID		;
	private final boolean	DEBUG		;	// 3.8.5.0 (2006/03/06) デバッグ用のフラグを追加

	private String	sqlCmd	;

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	public static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	private final ApplicationInfo appInfo;

	/**
	 * コンストラクター
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 *
	 * @param system_id システムID
	 * @param ykno      要求番号
	 * @param listId    帳票ID
	 * @param debug デバッグフラグ言語
	 */
	public ProgramRun( final String system_id, final String ykno, final String listId, final boolean debug ) {
		SYSTEM_ID	= system_id;
		YKNO		= ykno;
		LISTID		= listId;
		DEBUG		= debug;
		errMsg		= new StringBuilder( BUFFER_MIDDLE );

		// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
		if( USE_DB_APPLICATION_INFO ) {
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			appInfo.setClientInfo( SYSTEM_ID,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "ProgramRun",YKNO,LISTID );
		}
		else {
			appInfo = null;
		}
	}

	/**
	 * レポート出力処理を実行します。
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) StringUtil#stringStackTrace(Throwable) を、ThrowUtil#ogStackTrace(Throwable) に置き換え。
	 *
	 * @return 結果 [true:正常/false:異常]
	 */
	public boolean execute() {
		System.out.print( "ProgramRun [" + SYSTEM_ID + "]... " );
		boolean flag;

		try {
			flag = makeSQLCommand();
			if( flag ) { System.out.print( " MK SQL," ); }

			if( flag ) {
				flag = programRun();
				if( flag ) { System.out.print( " PG RUN," ); }
			}
			System.out.println( " End." );
		}
		catch( Throwable ex ) {
			errMsg.append( "ProgramRun Execute Error! " ).append( CR )
				.append( "==============================" ).append( CR )
//				.append( StringUtil.stringStackTrace( ex ) )
				.append( ThrowUtil.ogStackTrace( ex ) ).append( CR ) ;				// 6.4.2.0 (2016/01/29)
			flag = false;
		}

		return flag ;
	}

	/**
	 * PLSQL の call コマンドの文字列を作成します。
	 * { call 帳票ID+P.帳票ID( 結果(STATUS),内容(ERR_CODE),PGID,要求番号 ) } に対応する
	 * { call 帳票IDP.帳票ID( ?,?,?,? ) } 文字列を作成します。
	 *
	 * @return 結果 [true:正常/false:異常]
	 */
	private boolean makeSQLCommand() {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( "{ call " ).append( LISTID )
			.append( "P." ).append( LISTID )
			.append( "( ?,?,?,? ) }" );

		sqlCmd = buf.toString();
		if( DEBUG ) {
			System.out.println();
			System.out.println( sqlCmd );
		}

		return true;
	}

	/**
	 * 実際のPL/SQL コール処理を行います。
	 * { call 帳票IDP.帳票ID( 結果(STATUS),内容(ERR_CODE),PGID,要求番号 ) }
	 * という PL/SQL を Call します。
	 * 第１引数、第２引数は、OUT属性で、結果(STATUS)とエラー時の内容(ERR_CODE)を
	 * 返してください。第３引数は、起動元PGIDです。
	 * 結果(STATUS)は、正常ならば、0 を返してください。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 *
	 * @return 結果 [true:正常/false:異常]
	 */
	private boolean programRun() {

		final String[] args = new String[] { "ProgRUN",YKNO };
		final String[] rtn = DBUtil.dbCallExecute( sqlCmd,args,appInfo );	// 3.8.7.0 (2006/12/15)

		boolean flag = false;
		if( rtn != null && rtn.length == 2 ) {
			final String rtnCode = rtn[0];
			final String rtnMsg  = rtn[1];
			if( "0".equals( rtnCode ) ) {		// 正常
				flag = true;
			}
			else {
				errMsg.append( "PL/SQL=[" ).append( sqlCmd )
					.append( "] YKNO=[" ).append( YKNO )
					.append( "] LISTID=[" ).append( LISTID )
					.append( "] " ).append( rtnCode ).append( ':' ).append( rtnMsg )		// 6.0.2.5 (2014/10/31) char を append する。
					.append( CR );
				LogWriter.log( errMsg.toString() );
			}
		}
		return flag;
	}

	/**
	 * エラーが存在した場合に、エラーメッセージを返します。
	 *
	 * @return エラーメッセージ String
	 * @og.rtnNotNull
	 */
	public String getErrMsg() {
		return errMsg.toString();
	}
}
