/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.util.Attributes;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.resource.CodeData;
import org.opengion.hayabusa.resource.ColumnData;
import org.opengion.hayabusa.resource.LabelData;

/**
 * DBColumnオブジェクトを構築する為の 引数を持っているデータクラスです。
 * このクラス自身は、書き換え可能ですが、DBColumnオブジェクトは、読込み専用の
 * オブジェクトのため、別のオブジェクトを作成したい場合は,このクラスを
 * DBColumnオブジェクトより取得し、内容を書き換えて、再度 DBColumnオブジェクトを
 * 構築することになります。
 *
 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
 * @og.group テーブル管理
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBColumnConfig {
	private String[]  clmData	;	// カラム配列データ
	private LabelData labelData	;	// ラベルデータ
	private CodeData  codeData	;	// コードデータ

	// 6.4.5.2 (2016/05/06) rendAttri,editAttri は、not null にします。
	private final Attributes	rendAttri = new Attributes();			// 表示用レンデラー追加用属性
	private final Attributes	editAttri = new Attributes();			// 編集用エディター追加用属性

	private String		lang 		;			// 言語
	private String		writable	= "true";	// カラムが書き込み可能かどうか
//	private Attributes	rendAttri	;			// 表示用レンデラー追加用属性
//	private Attributes	editAttri	;			// 編集用エディター追加用属性
	private boolean		addNoValue	;			// メニューに空の選択リストを追加するかどうか	// 3.5.5.7 (2004/05/10)
	private String		addKeyLabel	;			// 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか
	private boolean		writeKeyLabel	;		// 6.2.3.0 (2015/05/01)
	private String		dbid		;			// データベース接続先ID
	private boolean		official ;				// カラムリソースから作成されたかどうか(再設定可能)		// 5.4.3.4 (2012/01/12)

	private String eventColumn		;			// 4.3.6.0 (2009/04/01) イベントカラム
	private String eventValue		;			// 6.3.3.0 (2015/07/25) eventValue 追加
	private String rawEditParameter	;			// 4.3.6.0 (2009/04/01) {@XXXX}を変換していないパラメータ
	private String rawRendParameter	;			// 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	private String eventURL			;			// 5.1.9.0 (2010/08/01) 動的プルダウン不具合対応

	private String	useSLabel		= "auto";	// 5.5.1.0 (2012/04/03)  MENU系でSLABELを利用するかどうか
	private String	noDisplayVal	;			// 5.6.2.3 (2013/03/22) 非表示文字の設定

	// 6.0.4.0 (2014/11/28) codeKeyVal は、KBMENU として、param にセットする為、廃止

	private boolean	stringOutput	= HybsSystem.sysBool( "USE_STRING_EXCEL_OUTPUT" );	// 5.7.6.3 (2013/05/23) ファイルレンデラ出力時のフラグ

	/**
	 * コンストラクター
	 * 引数は、すべて文字列型で渡します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
	 * @og.rev 3.5.5.7 (2004/05/10) CodeSelection を、codeName を元に構築できるように設定します。
	 * @og.rev 3.6.0.3 (2004/10/05) dbTypePrm は、editorPrm と無関係に設定します。
	 * @og.rev 3.6.0.7 (2004/11/06) カラムリソースから作成されたかどうか(official属性追加)
	 * @og.rev 5.1.6.0 (2010/05/01) DBTypeの初期値を "XK" とします。
	 * @og.rev 5.7.3.0 (2014/02/07) labelData の null 対応
	 * @og.rev 6.2.0.0 (2015/02/27) フィールドサイズ 追加(VIEW_LENGTHと分離して、役割を明確にする)
	 *
	 * @param	lang		言語
	 * @param	name		カラム名
	 * @param	labelData	カラムのラベルデータオブジェクト
	 * @param	className	カラムのクラスを文字列にした名称
	 * @param	fieldSize	入力枠サイズ							// 6.2.0.0 (2015/02/27)
	 * @param	viewLength	カラムの表示文字数						// 6.2.0.0 (2015/02/27)
	 * @param	maxlength	カラムの最大桁数
	 * @param	writable	カラムが書き込み可能かどうか
	 * @param	renderer	データの表示用レンデラー
	 * @param	editor		データの編集用エディター
	 * @param	codeData 	メニューの項目オブジェクト
	 * @param	dbType		データのタイプ
	 * @param	dataDef		データのデフォルト値
	 * @param	rendererPrm	表示用レンデラーのパラメータ
	 * @param	editorPrm	編集用エディターのパラメータ
	 * @param	dbTypePrm	データのタイプのパラメータ
	 * @param	roles		カラムロール
	 * @param	official	正規カラムかどうか
	 * @param	dbid		データベース接続先ID
	 */
	public DBColumnConfig(	final String	lang		,
							final String	name		,
							final LabelData	labelData	,		// 4.0.0 (2005/01/31)
							final String	className	,
							final String	fieldSize	,		// 6.2.0.0 (2015/02/27)
							final String	viewLength	,		// 6.2.0.0 (2015/02/27)
							final String	maxlength	,		// 3.5.5.5 (2004/04/23)
							final String	writable	,
							final String	renderer	,
							final String	editor		,
							final CodeData	codeData	,		// 4.0.0 (2005/01/31)
							final String	dbType		,
							final String	dataDef		,
							final String	rendererPrm	,
							final String	editorPrm	,
							final String	dbTypePrm	,
							final String	roles		,		// 4.0.0 (2005/11/30)
							final boolean	official	,		// 3.6.0.7 (2004/11/06)
							final String	dbid ) {
		clmData = new String[ColumnData.DATA_SIZE];
		clmData[ColumnData.CLM           ] = name		;
		clmData[ColumnData.CLS_NAME      ] = className	;
		clmData[ColumnData.FIELD_SIZE    ] = fieldSize	;		// 6.2.0.0 (2015/02/27)
		clmData[ColumnData.VIEW_LENGTH   ] = viewLength	;		// 6.2.0.0 (2015/02/27)
		clmData[ColumnData.USE_LENGTH    ] = maxlength	;
		clmData[ColumnData.RENDERER      ] = renderer	;
		clmData[ColumnData.EDITOR        ] = editor		;
		clmData[ColumnData.DBTYPE        ] = dbType		;
		clmData[ColumnData.DATA_DEFAULT  ] = dataDef	;
		clmData[ColumnData.RENDERER_PARAM] = rendererPrm;
		clmData[ColumnData.TYPE_PARAM    ] = dbTypePrm	;
		clmData[ColumnData.ROLES         ] = roles		;		// 4.0.0 (2005/11/30)

		this.lang		= lang		;
		// 5.7.3.0 (2014/02/07) labelData の null 対応
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		this.labelData	= labelData == null ? new LabelData( name ) : labelData ;
//		this.labelData	= (labelData != null) ? labelData : new LabelData( name ) ;
		this.writable	= writable	;
		this.codeData	= codeData	;		// 4.0.0 (2005/01/31)
		this.dbid		= dbid		;		// 4.0.0 (2005/01/31)

		if( "=".equals(editorPrm) ) {
			clmData[ColumnData.EDITOR_PARAM  ] = rendererPrm	;
		}
		else {
			clmData[ColumnData.EDITOR_PARAM  ] = editorPrm	;
		}

		// 5.1.6.0 (2010/05/01) DBTypeの初期値を "XK" とします。
		if( dbType == null ) {
			clmData[ColumnData.DBTYPE        ] = DBType.DEF_TYPE;
		}

		this.official = official;		// 3.6.0.7 (2004/11/06)
	}

	/**
	 * 簡易コンストラクター
	 * キーのみを設定して作成する、簡易的な DBColumnConfig オブジェクトを
	 * 作成します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
	 * @og.rev 3.6.0.7 (2004/11/06) カラムリソースから作成されたかどうか(official属性追加)
	 * @og.rev 5.1.6.0 (2010/05/01) DBTypeの初期値を "XK" とします。
	 * @og.rev 5.7.3.0 (2014/02/07) labelData の null 対応
	 * @og.rev 6.2.0.0 (2015/02/27) フィールドサイズ 追加(VIEW_LENGTHと分離して、役割を明確にする)
	 * @og.rev 6.2.3.0 (2015/05/01) もう一つのコンストラクタを呼びように変更。
	 *
	 * @param	name		 	カラム名
	 */
	public DBColumnConfig( final String name ) {
		this(	"ja"					,	// lang
				name					,	// name
				new LabelData( name )	,	// labelData
				"VARCHAR2"				,	// className
				null					,	// fieldSize
				null					,	// viewLength
				"60"					,	// maxlength
				"true"					,	// writable
				null					,	// renderer
				null					,	// editor
				null					,	// codeData
				DBType.DEF_TYPE			,	// dbType
				""						,	// dataDef
				""						,	// rendererPrm
				""						,	// editorPrm
				""						,	// dbTypePrm
				null					,	// roles
				false					,	// official
				null						// dbid
			);

	}

	/**
	 * 言語を返します。
	 *
	 * @return	言語
	 */
	public String getLang() {
		return lang;
	}

	/**
	 * 言語を設定します。
	 *
	 * @param	lang 言語
	 */
	public void setLang( final String lang ) {
		this.lang = lang;
	}

	/**
	 * カラム名を返します。
	 *
	 * @return	カラム名
	 */
	public String getName() {
		return clmData[ColumnData.CLM];
	}

	/**
	 * カラム名を設定します。
	 *
	 * @param	name カラム名
	 */
	public void setName( final String name ) {
		clmData[ColumnData.CLM] = name;
	}

	/**
	 * ラベル名を返します。
	 *
	 * @return	ラベル名
	 */
	public String getLabel() {
		return ( labelData == null ) ? clmData[ColumnData.CLM] : labelData.getLabel() ;
	}

	/**
	 * カラムのクラスを文字列にした名称を返します。
	 *
	 * @return	カラムのクラスを文字列にした名称
	 */
	public String getClassName() {
		return clmData[ColumnData.CLS_NAME];
	}

	/**
	 * 現在の値と、新しい値をチェックして、条件に応じて新しい値を返します。
	 *
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 * 上位で、val が null チェックを入れていますが、汎用的に、こちらでも入れておきます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	now 現在の値
	 * @param	val 新しい値（"既存値:設定値" と指定可能)
	 * @return	新しい値
	 */
	private String oldNewSplit( final String now , final String val ) {
		final String rtn ;

		if( val == null || val.length() == 0 ) {
			rtn = now;
		}
		else {
			final int ad = val.indexOf( ':' );
			if( ad < 0 ) {							// コロンがない。通常処理
				rtn = val;
			}
			else if( ad == 0 ) {					// コロンが先頭。コロンをはずして、セット
				rtn = val.substring(1);
			}
			else {									// コロンがある。OLD:NEW 形式なので、コロンで分割
				final String old = val.substring( 0,ad );
				if( old.equals( now ) ) {			// 一致した。
					rtn = val.substring( ad+1 );
				}
				else {
					rtn = now ;
				}
			}
		}

		return rtn ;
	}

	/**
	 * カラムのクラスを文字列にした名称を設定します。
	 *
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	className カラムのクラス名
	 */
	public void setClassName( final String className ) {
//		clmData[ColumnData.CLS_NAME] = className;
		clmData[ColumnData.CLS_NAME] = oldNewSplit( clmData[ColumnData.CLS_NAME] , className );		// 6.4.1.1 (2016/01/16)
	}

	/**
	 * カラムの入力枠サイズ(横幅)を文字数で取得します。
	 * fieldSize は、設定した場合のみ、使用できます。通常は、null が返ります。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) fieldSize 属性の追加
	 *
	 * @return	カラムの入力枠サイズ
	 */
	public String getFieldSize() {
		return clmData[ColumnData.FIELD_SIZE];
	}

	/**
	 * カラムの入力枠サイズ(横幅)を文字数で設定します。
	 *
	 * 従来の #setViewLength(String) で指定していた値です。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) fieldSize 属性の追加
	 *
	 * @param	fieldSize カラムの入力枠サイズ
	 */
	public void setFieldSize( final String fieldSize ) {
		clmData[ColumnData.FIELD_SIZE] = fieldSize;
	}

	/**
	 * カラムの表示桁数を返します。
	 * viewLength は、設定した場合のみ、使用できます。通常は、null が返ります。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) メソッド名変更 getSize() → getViewLength()
	 *
	 * @return	カラムの表示桁数
	 */
	public String getViewLength() {
		return clmData[ColumnData.VIEW_LENGTH];
	}

	/**
	 * カラムの表示桁数を設定します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) メソッド名変更 setSize() → setViewLength()
	 *
	 * @param	viewLength カラムの表示桁数
	 */
	public void setViewLength( final String viewLength ) {
		clmData[ColumnData.VIEW_LENGTH] = viewLength;
	}

	/**
	 * カラムの文字桁数を返します。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規追加
	 *
	 * @return	カラムの文字桁数
	 */
	public String getMaxlength() {
		return clmData[ColumnData.USE_LENGTH];
	}

	/**
	 * カラムの文字桁数を設定します。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規追加
	 *
	 * @param	maxlength カラムの文字桁数
	 */
	public void setMaxlength( final String maxlength ) {
		clmData[ColumnData.USE_LENGTH] = maxlength;
	}

	/**
	 * カラムが書き込み可能かどうか[true/false]を返します。
	 *
	 * @return	カラムが書き込み可能かどうか[true/false]
	 */
	public boolean isWritable() {
		return "true".equalsIgnoreCase( writable ) ;
	}

	/**
	 * カラムが書き込み可能かどうか[true/false]を設定します。
	 *
	 * @param	writable 書き込み可能かどうか[true/false]
	 */
	public void setWritable( final String writable ) {
		this.writable = writable;
	}

	/**
	 * データの表示用レンデラーを返します。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) Renderer をキーのみ返すように変更
	 *
	 * @return	データの表示用レンデラー
	 */
	public String getRenderer() {
		return clmData[ColumnData.RENDERER];
	}

	/**
	 * データの表示用レンデラーを設定します。
	 *
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	renderer 表示用レンデラー
	 */
	public void setRenderer( final String renderer ) {
//		clmData[ColumnData.RENDERER] = renderer;
		clmData[ColumnData.RENDERER] = oldNewSplit( clmData[ColumnData.RENDERER] , renderer );		// 6.4.1.1 (2016/01/16)
	}

	/**
	 * データの編集用エディターを返します。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) Editor をキーのみ返すように変更
	 *
	 * @return	データの編集用エディター
	 */
	public String getEditor() {
		return clmData[ColumnData.EDITOR];
	}

	/**
	 * データの編集用エディターを設定します。
	 *
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	editor 編集用エディター
	 */
	public void setEditor( final String editor ) {
//		clmData[ColumnData.EDITOR] = editor;
		clmData[ColumnData.EDITOR] = oldNewSplit( clmData[ColumnData.EDITOR] , editor );		// 6.4.1.1 (2016/01/16)
	}

	/**
	 * 文字種別名を返します。
	 *
	 * @return  データの文字種別
	 */
	public String getDbType() {
		return clmData[ColumnData.DBTYPE];
	}

	/**
	 * データのタイプを設定します。
	 *
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) DBTypeの初期値を "XK" とします。
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	dbType 文字種別
	 */
	public void setDbType( final String dbType ) {
//		clmData[ColumnData.DBTYPE] = (dbType == null) ? DBType.DEF_TYPE : dbType ;
		clmData[ColumnData.DBTYPE] = oldNewSplit( clmData[ColumnData.DBTYPE] , (dbType == null) ? DBType.DEF_TYPE : dbType );		// 6.4.1.1 (2016/01/16)
	}

	/**
	 * カラムロールを返します。
	 *
	 * @return  カラムロール
	 */
	public String getRoles() {
		return clmData[ColumnData.ROLES];
	}

	/**
	 * カラムロールを設定します。
	 *
	 * @param	roles カラムロール
	 */
	public void setRoles( final String roles ) {
		clmData[ColumnData.ROLES] = roles;
	}

	/**
	 * データのデフォルト値を返します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) メソッド名変更(getDefValue ⇒ getDefault)
	 *
	 * @return	データのデフォルト値
	 */
	public String getDefault() {
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		return ( clmData[ColumnData.DATA_DEFAULT] == null ) ? "" : clmData[ColumnData.DATA_DEFAULT];
//		return ( clmData[ColumnData.DATA_DEFAULT] != null ) ? clmData[ColumnData.DATA_DEFAULT] : "";
	}

	/**
	 * データのデフォルト値を設定します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) メソッド名変更(setDefValue ⇒ setDefault)
	 *
	 * @param	dataDef デフォルト値
	 */
	public void setDefault( final String dataDef ) {
		clmData[ColumnData.DATA_DEFAULT] = dataDef;
	}

	/**
	 * 表示用レンデラーの追加属性を返します。
	 *
	 * @return   属性リスト
	 * @og.rtnNotNull
	 */
	public Attributes getRendererAttributes() {
		return rendAttri ;
	}

	/**
	 * 表示用レンデラーの追加属性を設定します。
	 *
	 * 元のrendAttriが、何も設定されていない場合は、引数のAttributesに
	 * 置き換えます。
	 * すでに、設定済みの場合は、putAll で、すべての属性を追加します。
	 * 既存の属性は、そのまま残りますので、クリアしたい場合は、
	 * 先に、getRendererAttributes() で取り出した Attributes を、
	 * clear() してから、セットしてください。
	 *
	 * @og.rev 6.4.5.2 (2016/05/06) Attributesは、既存のAttributesに、ADDする。
	 *
	 * @param   attri	属性リスト
	 */
	public void setRendererAttributes( final Attributes attri ) {
//		rendAttri = attri;
		rendAttri.set( attri );
	}

	/**
	 * 編集用エディター用の追加属性を返します。
	 *
	 * @return	 追加属性
	 * @og.rtnNotNull
	 */
	public Attributes getEditorAttributes() {
		return editAttri;
	}

	/**
	 * 編集用エディター用の追加属性を設定します。
	 *
	 * 元のrendAttriが、何も設定されていない場合は、引数のAttributesに
	 * 置き換えます。
	 * すでに、設定済みの場合は、putAll で、すべての属性を追加します。
	 * 既存の属性は、そのまま残りますので、クリアしたい場合は、
	 * 先に、getRendererAttributes() で取り出した Attributes を、
	 * clear() してから、セットしてください。
	 *
	 * @og.rev 6.4.5.2 (2016/05/06) Attributesは、既存のAttributesに、ADDする。
	 *
	 * @param   attri	属性リスト
	 */
	public void setEditorAttributes( final Attributes attri ) {
//		editAttri = attri;
		editAttri.set( attri );
	}

	/**
	 * パラメータを設定します。
	 * ここで設定された値は、すべて(表示、編集、文字)のパラメータにセットされます。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
	 *
	 * @param	inParam	入力パラメータ
	 */
	public void setParameter( final String inParam ) {
		clmData[ColumnData.RENDERER_PARAM]	= inParam ;
		clmData[ColumnData.EDITOR_PARAM]	= inParam ;
		clmData[ColumnData.TYPE_PARAM]		= inParam ;
	}

	/**
	 * 表示用レンデラーのパラメータを取得します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @return	表示用レンデラーのパラメータ
	 */
	public String getRendererParam() {
		return clmData[ColumnData.RENDERER_PARAM];
	}

	/**
	 * 表示用レンデラーのパラメータを設定します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @param	inParam	表示用レンデラーのパラメータ
	 */
	public void setRendererParam( final String inParam ) {
		clmData[ColumnData.RENDERER_PARAM] = inParam;
	}

	/**
	 * 編集用エディターのパラメータを取得します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @return	編集用エディターのパラメータ
	 */
	public String getEditorParam() {
		return clmData[ColumnData.EDITOR_PARAM];
	}

	/**
	 * 編集用エディターのパラメータを設定します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @param	inParam	編集用エディターのパラメータ
	 */
	public void setEditorParam( final String inParam ) {
		clmData[ColumnData.EDITOR_PARAM] = inParam	;
	}

	/**
	 * データタイプのパラメータを取得します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @return	データタイプのパラメータ
	 */
	public String getDbTypeParam() {
		return clmData[ColumnData.TYPE_PARAM];
	}

	/**
	 * データタイプのパラメータをします。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @param	inParam データタイプのパラメータ
	 */
	public void setDbTypeParam( final String inParam ) {
		clmData[ColumnData.TYPE_PARAM] = inParam ;
	}

	/**
	 * メニューに空の選択リストを追加するかどうかを取得します。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 新規追加
	 *
	 * @return	選択リストを追加するかどうか(true:追加する/false:追加しない)
	 */
	public boolean isAddNoValue() {
		return addNoValue ;
	}

	/**
	 * メニューに空の選択リストを追加するかどうかを設定します。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 新規追加
	 *
	 * @param	add	[true:追加する/false:追加しない]
	 */
	public void setAddNoValue( final boolean add ) {
		this.addNoValue	= add	;
	}

	/**
	 * セレクトメニューの場合、キー:ラベル形式で表示するかどうか[true/false/null]を指定します(初期値:null)。
	 *
	 * カラムがプルダウンメニュー形式の場合、キー:ラベル形式で表示するか、ラベルだけで表示するかを
	 * 選択できます。
	 * true を指定すると、「強制的に」キー:ラベル形式で表示します。
	 * false の場合は、「強制的に」ラベルのみで表示されます。
	 * 初期値の null の場合、コードリソースや、SQL文に準拠した指定通りの形式で表示します。
	 *
	 * &lt;og:column name="CDC" addKeyLabel="true" /&gt;
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか。新規追加
	 *
	 * @param   addKeyLbl  [true:キー:ラベル形式/false:ラベルのみ/null:指定通り]
	 * @see		#getAddKeyLabel()
	 */
	public void setAddKeyLabel( final String addKeyLbl ) {
		addKeyLabel = addKeyLbl;
	}

	/**
	 * セレクトメニューの場合、キー:ラベル形式で表示するかどうか[true/false/null]を返します。
	 *
	 * これは、上位入れ子のタグの OptionTag で、addKeyLabel を取り出して、
	 * true であれば、キー:ラベル形式 のオプションを、#addOption( String ) で
	 * 登録させます。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか。新規追加
	 *
	 * @return	true:キー:ラベル形式/false:ラベルのみ/null:指定通り
	 * @see		#setAddKeyLabel( String )
	 */
	public String getAddKeyLabel() {
		return addKeyLabel;
	}

	/**
	 * セレクトメニューの場合、キーとラベルを分けて出力するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * カラムがプルダウンメニュー形式の場合、ファイル出力時に、キーとラベルを分離して出力する事ができます。
	 * この属性と、addKeyLabel とは、微妙に関連していて、addKeyLabel="true" のみの場合は、
	 * キー:ラベル形式で、出力されますが、writeKeyLabel="true" も指定されている場合は、こちらが優先されます。
	 * なお、この指定が有効なのは、セレクトメニューのカラムのみです。
	 * 初期値は、false です。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) キー／ラベルで分離して出力するかどうか。
	 *
	 * @param   flag  キー／ラベル分離出力するかどうか [true:分離出力/false:通常]
	 */
	public void setWriteKeyLabel( final boolean flag ) {
		writeKeyLabel = flag;
	}

	/**
	 * セレクトメニューの場合、キーとラベルを分けて出力するかどうか[true/false]を取得します。
	 *
	 * カラムがプルダウンメニュー形式の場合、ファイル出力時に、キーとラベルを分離して出力する事ができます。
	 * この属性と、addKeyLabel とは、微妙に関連していて、addKeyLabel="true" のみの場合は、
	 * キー:ラベル形式で、出力されますが、writeKeyLabel="true" も指定されている場合は、こちらが優先されます。
	 * なお、この指定が有効なのは、セレクトメニューのカラムのみです。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) キー／ラベルで分離して出力するかどうか。
	 *
	 * @return	キー／ラベル分離出力するかどうか[true:分離出力/false:通常]
	 */
	public boolean isWriteKeyLabel() {
		return writeKeyLabel ;
	}

	/**
	 * このカラムオブジェクトがカラムリソースより正式に作られたかどうかを取得します。
	 * カラムリソースが無い場合は、仮オブジェクトかデータベースメタデータより作成されます。
	 * その場合は、チェック機能が緩くなるため、正式なカラムオブジェクトと区別する為に
	 * この属性を持ちます。
	 *
	 * @og.rev 3.6.0.7 (2004/11/06) 新規追加
	 *
	 * @return	カラムリソースより正式に作られたかどうか(true:正式/false:暫定)
	 */
	public boolean isOfficial() {
		return official;
	}

	/**
	 * このカラムオブジェクトがカラムリソースより正式に作られたかどうかを設定します。
	 * 従来は、自動判定で、再セット不可能でしたが、動的カラムを実現する場合に、
	 * エラーチェックで引っかかるため、再設定可能にします。
	 *
	 * @og.rev 5.4.3.4 (2012/01/12) 新規追加
	 *
	 * @param	official 正式かどうか [true:正式/false:暫定]
	 */
	public void setOfficial( final boolean official ) {
		this.official = official ;
	}

	/**
	 * 接続先IDを設定します。
	 *
	 * @param dbid 接続先ID
	 */
	public void setDbid( final String dbid ) {
		this.dbid = dbid;
	}

	/**
	 * 接続先IDを返します。
	 *
	 * @return	接続先ID
	 */
	public String getDbid() {
		return dbid;
	}

	/**
	 * カラムデータオブジェクトを取得します。
	 *
	 * @return	カラムデータオブジェクト
	 * @og.rtnNotNull
	 */
	public ColumnData getColumnData() {
		return new ColumnData( clmData );
	}

	/**
	 * カラムデータオブジェクトを設定します。
	 * ここでは、オブジェクトその物を持つのではなく、設定された値のみ持ちます。
	 *
	 * @param	clmDt カラムデータオブジェクト
	 */
	public void setColumnData( final ColumnData clmDt ) {
		clmData = clmDt.getColumnData();
	}

	/**
	 * コードデータオブジェクトを取得します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31)  新規作成
	 *
	 * @return	コードデータオブジェクト
	 */
	public CodeData getCodeData() {
		return codeData;
	}

	/**
	 * コードデータオブジェクトを設定します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31)  新規作成
	 *
	 * @param	codeData コードデータオブジェクト
	 */
	public void setCodeData( final CodeData codeData ) {
		this.codeData = codeData;
	}

	/**
	 * ラベルデータオブジェクトを取得します。
	 *
	 * @return	ラベルデータオブジェクト
	 */
	public LabelData getLabelData() {
		return labelData;
	}

	/**
	 * ラベルデータオブジェクトを設定します。
	 *
	 * @param	labelData ラベルデータオブジェクト
	 */
	public void setLabelData( final LabelData labelData ) {
		this.labelData = labelData;
	}

	/**
	 * イベントカラム(親カラム)のセット
	 *
	 * イベントカラムをセットします。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 * @param	col	親カラムID
	 */
	public void setEventColumn( final String col ) {
		this.eventColumn = col;
	}

	/**
	 * イベントカラム(親カラム)の取得
	 *
	 * イベントカラムを取得します。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @return  イベントカラム
	 */
	public String getEventColumn() {
		return eventColumn;
	}

	/**
	 * イベントカラムの子カラムの値を出力するためのSQL文のセット
	 *
	 * イベントカラムの子カラムの値を出力するためのSQL文をセットします。
	 *
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加
	 *
	 * @param	val	イベントカラムの値SQL文
	 */
	public void setEventValue( final String val ) {
		this.eventValue = val;
	}

	/**
	 * イベントカラムの子カラムの値を出力するためのSQL文の取得
	 *
	 * イベントカラムの子カラムの値を出力するためのSQL文を取得します。
	 *
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加
	 *
	 * @return  イベントカラムの値SQL文
	 */
	public String getEventValue() {
		return eventValue;
	}

	/**
	 * {&#064;XXXX}を変換していない状態の編集パラメータ
	 *
	 * 生のパラメータを取得します。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @return  生編集パラメータ
	 */
	public String getRawEditParameter() {
		return rawEditParameter;
	}

	/**
	 * {&#064;XXXX}を変換していない状態の表示パラメータ
	 *
	 * 生のパラメータを取得します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @return  生表示パラメータ
	 */
	public String getRawRendParameter() {
		return rawRendParameter;
	}

	/**
	 * {&#064;XXXX}を変換していない状態のparameterセット用
	 *
	 * 編集種別・表示種別
	 * 生のパラメータをセットします。(編集種別・表示種別の両方に適用されます。)
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param param パラメータ
	 */
	public void setRawParameter( final String param ) {
		rawEditParameter = param;
		rawRendParameter = param;
	}

	/**
	 * {&#064;XXXX}を変換していない状態の編集種別 parameterセット用
	 *
	 * DBColumn とのデータ変換のため、内部変数として、編集種別・表示種別を分けて管理します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規追加
	 *
	 * @param param パラメータ
	 */
	public void setRawEditParameter( final String param ) {
		rawEditParameter = param;
	}

	/**
	 * {&#064;XXXX}を変換していない状態の表示種別 parameterセット用
	 *
	 * DBColumn とのデータ変換のため、内部変数として、編集種別・表示種別を分けて管理します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規追加
	 *
	 * @param param パラメータ
	 */
	public void setRawRendParameter( final String param ) {
		rawRendParameter = param;
	}

	/**
	 * イベントURLのセット。
	 *
	 * イベントカラム利用時にコールされるURLのセット
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 * @param url イベントURL
	 */
	public void setEventURL( final String url ) {
		this.eventURL = url;
	}

	/**
	 * イベントURLの取得。
	 *
	 * eventColumn利用時のJSから呼び出すためのURL取得
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @return  イベントURL
	 */
	public String getEventURL() {
		return eventURL;
	}

	/**
	 * MENU時にラベル短ベースで表示するかどうか[auto/true/その他]のコントロール
	 *
	 * auto/trueが有効。それ以外はfalse扱いとする。標準auto
	 *
	 * @og.rev 5.5.1.0 (2012/04/03)
	 * @param use 表示するかどうか[auto/true/その他]
	 */
	public void setUseSLabel( final String use ) {
		this.useSLabel = use;
	}

	/**
	 * MENU時にラベル短ベースで表示するかどうかのコントロール。
	 *
	 * @og.rev 5.5.1.0 (2012/04/03)
	 *
	 * @return  イベントURL
	 */
	public String getUseSLabel() {
		return useSLabel;
	}

	/**
	 * 非表示文字列を設定します。
	 *
	 * これは、例えば、数字型の場合は、0 がデフォルトなどの場合、
	 * あえて表示したくないなどのケースに使います。
	 * そのような状況が設定されていない場合は、null を設定してください。
	 * 初期値は、null です。
	 *
	 * @og.rev 5.6.2.3 (2013/03/22) 新規追加
	 *
	 * @param val 非表示にする文字列
	 */
	public void setNoDisplayVal( final String val ) {
		this.noDisplayVal = val;
	}

	/**
	 * 非表示文字列を返します。
	 *
	 * これは、例えば、数字型の場合は、0 がデフォルトなどの場合、
	 * あえて表示したくないなどのケースに使います。
	 * そのような状況が設定されていない場合は、null が返されます。
	 * 初期値は、null です。
	 *
	 * @og.rev 5.6.2.3 (2013/03/22) 新規追加
	 *
	 * @return 非表示文字
	 */
	public String getNoDisplayVal() {
		return noDisplayVal;
	}

	/**
	 * レンデラー利用ファイル出力時に数値等も文字タイプ固定にするかどうかを設定します
	 *		(初期値:USE_STRING_EXCEL_OUTPUT[={@og.value SystemData#USE_STRING_EXCEL_OUTPUT}])。
	 *
	 * これは、レンデラーでのExcel出力時のセルタイプに影響します。
	 * trueの場合は数値も全て文字として出力します。
	 * falseの場合はDBTYPEに従った出力になりるため、数値型の DBTYPE では、doubleに一度変換されます。
	 * (初期値:USE_STRING_EXCEL_OUTPUT[={@og.value SystemData#USE_STRING_EXCEL_OUTPUT}])。
	 *
	 * @og.rev 5.7.6.3 (2014/05/23) 新規追加
	 *
	 * @param flag レンデラー利用時の出力方法 (true:String出力/false:DBTYPEに従う)
	 */
	public void setStringOutput( final boolean flag ) {
		stringOutput = flag;
	}

	/**
	 * レンデラー利用ファイル出力時に数値等も文字タイプ固定にするかどうか
	 *		(初期値:USE_STRING_EXCEL_OUTPUT[={@og.value SystemData#USE_STRING_EXCEL_OUTPUT}])。
	 *
	 * これは、レンデラーでのExcel出力時のセルタイプに影響します。
	 * trueの場合は数値も全て文字として出力します。
	 * falseの場合はDBTYPEに従った出力になりるため、数値型の DBTYPE では、doubleに一度変換されます。
	 * (初期値:USE_STRING_EXCEL_OUTPUT[={@og.value SystemData#USE_STRING_EXCEL_OUTPUT}])。
	 *
	 * @og.rev 5.7.6.3 (2014/05/23) 新規追加
	 *
	 * @return レンデラー利用時の出力方法 (true:String出力/false:DBTYPEに従う)
	 */
	public boolean isStringOutput() {
		return stringOutput;
	}
}
