/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.util.List;
import java.util.ArrayList;
import java.util.Collections;										// 6.3.9.0 (2015/11/06)

/**
 * HybsTimerTaskManager.java は、HybsTimerTask オブジェクトを管理しているクラスです。
 *
 * HybsTimerTask は、タスクとして一定間隔で実行されます。オブジェクトが生成されてから、
 * 実行中のタスクを、このクラスで管理することにより、個別に削除、生成、全件終了等の制御が
 * 可能になります。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class HybsTimerTaskManager implements Cleanable {
//	private final List<HybsTimerTask> list = new ArrayList<>();
	private final List<HybsTimerTask> list = Collections.synchronizedList( new ArrayList<>() );		// 6.3.9.0 (2015/11/06)

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public HybsTimerTaskManager() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * HybsTimerTask クラスの 実オブジェクトを管理のため、登録します。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
	 * @param task HybsTimerTaskオブジェクト
	 */
//	public synchronized void addTask( final HybsTimerTask task ) {
	public void addTask( final HybsTimerTask task ) {
		if( task != null ) {				// 6.3.9.0 (2015/11/06) task の nullチェックを行います。
			list.add( task );
		}
//		list.add( task );
	}

	/**
	 * このタイマータスクオブジェクトを削除します。
	 * このとき、すでにタイマータスク自信でキャンセルされているオブジェクトの
	 * 削除も行います。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
	 * @param key  HybsTimerTask のユニークキー
	 *
	 * @return 結果(削除成功 true/ 削除できなかった false )
	 */
//	public synchronized boolean cancelTask( final int key ) {
	public boolean cancelTask( final int key ) {
		synchronized( list ) {
			list.stream()
				.filter(  task -> task != null && task.isAlive() && key == task.getUniqKey() )
				.forEach( task -> task.cancel() );
		}
		taskCleaning();			// 6.3.9.0 (2015/11/06)

		return true;			// なくても true を返す。

//		boolean flag = false;

//		final int size = list.size();
//		for( int i=size-1; i>=0; i-- ) {
//			final HybsTimerTask task = list.get( i );
//			if( task != null ) {
//				if( !task.isAlive() || task.getUniqKey() == key ) {
//					list.remove( i );
//					task.cancel();
//					flag = true;
//				}
//			}
//			else {
//				list.remove( i );
//			}
//		}
//
//		return flag;
	}

	/**
	 * このタイマータスクオブジェクトを削除します。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
	 * @param  name  HybsTimerTask の名前
	 *
	 * @return 結果(削除成功 true/ 削除できなかった false )
	 */
//	public synchronized boolean cancelTask( final String name ) {
	public boolean cancelTask( final String name ) {
		if( name == null ) { return false; }

		synchronized( list ) {
			list.stream()
				.filter(  task -> task != null && task.isAlive() && name.equals( task.getName() ) )
				.forEach( task -> task.cancel() );
		}
		taskCleaning();			// 6.3.9.0 (2015/11/06)

		return true;			// なくても true を返す。

//		boolean flag = false;
//		if( name == null ) { return flag; }

//		final int size = list.size();
//		for( int i=size-1; i>=0; i-- ) {
//			final HybsTimerTask task = list.get( i );
//			if( task != null ) {
//				if( !task.isAlive() || name.equals( task.getName() ) ) {
//					list.remove( i );
//					task.cancel();
//					flag = true;
//				}
//			}
//			else {
//				list.remove( i );
//			}
//		}
//
//		return flag;
	}

	/**
	 * 現在実行中の全てのタイマータスクオブジェクトを削除します。
	 * (org.opengion.fukurou.util.Cleanable の実装)。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
	 */
//	public synchronized void clear() {
	public void clear() {
		synchronized( list ) {
			list.stream()
				.filter(  task -> task != null )
				.forEach( task -> task.cancel() );

			list.clear();
		}

//		HybsTimerTask[] tasks = list.toArray( new HybsTimerTask[list.size()] );
//
//		for( int i=0; i<tasks.length; i++ ) {
//			if( tasks[i] != null ) {
//				tasks[i].cancel();
//				tasks[i] = null;
//			}
//		}
//		list.clear();
	}

	/**
	 * 現在実行中の、タイマータスクを、すべて返します。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
	 * @return タイマータスクの配列
	 */
//	public synchronized HybsTimerTask[] toArray() {
	public HybsTimerTask[] toArray() {
		taskCleaning();			// 6.3.9.0 (2015/11/06)

		synchronized( list ) {
			return list.toArray( new HybsTimerTask[list.size()] );
		}

//		final int size = list.size();
//		for( int i=size-1; i>=0; i-- ) {
//			final HybsTimerTask task = list.get( i );
//			if( task != null ) {
//				if( !task.isAlive() ) {
//					list.remove( i );
//				}
//			}
//			else {
//				list.remove( i );
//			}
//		}
//
//		final HybsTimerTask[] tasks = list.toArray( new HybsTimerTask[list.size()] );
//
//		return tasks;
	}

	/**
	 * 現在実行中の、タイマータスクの件数を返します。
	 *
	 * タスクがすでに 終了していたり、null 化されているかをチェックし、
	 * 不要なタスクは、削除します。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
	 * @return タイマータスクの件数
	 */
//	public synchronized int size() {
	public int size() {
		taskCleaning();			// 6.3.9.0 (2015/11/06)

//		final int size = list.size();
//		for( int i=size-1; i>=0; i-- ) {
//			final HybsTimerTask task = list.get( i );
//			if( task != null ) {
//				if( !task.isAlive() ) {
//					list.remove( i );
//				}
//			}
//			else {
//				list.remove( i );
//			}
//		}

		return list.size();
	}

	/**
	 * 現在実行中の、タイマータスクに、同一の Name 属性を持ったタスクが存在するかどうか。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
	 * @param  nm タイマータスクの、Name 属性
	 *
	 * @return 存在する：true / 存在しない：false
	 */
//	public synchronized boolean contains( final String nm ) {
	public boolean contains( final String nm ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		// 条件組み入れ注意
		return nm != null &&
				list.stream()
					.filter( task -> task != null && nm.equals( task.getName() ) )
					.findFirst()					// 1件目をピックアップ
					.isPresent();					// あれば true, なければ false

//		if( nm == null ) { return false; }

////		final HybsTimerTask[] tasks = toArray() ;
////
////		for( int i=0; i<tasks.length; i++ ) {
////			if( tasks[i] != null && nm.equals( tasks[i].getName() ) ) { return true; }
////		}
////		return false;
//
//		return list.stream()
//					.filter( task -> task != null && nm.equals( task.getName() ) )
//					.findFirst()					// 1件目をピックアップ
//					.isPresent();					// あれば true, なければ false
	}

	/**
	 * 現在実行中の、タイマータスクから、活きているタスクのみに List をクリーニングします。
	 *
	 * この処理を実行すると、とりあえず現時点で生きているタスクのみに再編成します。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) 新規追加
	 */
	private void taskCleaning() {
		list.removeIf( task -> task == null || !task.isAlive() );

//		synchronized( list ) {
//			final int size = list.size();
//			for( int i=size-1; i>=0; i-- ) {
//				final HybsTimerTask task = list.get( i );
//				if( task != null ) {
//					if( !task.isAlive() ) {
//						list.remove( i );
//					}
//				}
//				else {
//					list.remove( i );
//				}
//			}
//		}
	}
}
