/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

// import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.SystemManager;
import org.opengion.fukurou.util.Cleanable;

// import java.util.Map;
// import java.util.HashMap;
import java.util.concurrent.ConcurrentMap;							// 6.4.3.3 (2016/03/04)
import java.util.concurrent.ConcurrentHashMap;						// 6.4.3.1 (2016/02/12) refactoring
import java.util.Locale ;
// import java.util.Collections;

/**
 * DBType オブジェクトを取得する為に使用する，ファクトリクラスです。
 *
 *  DBType オブジェクト の識別ID を元に、DBTypeFactory.newInstance( String id )
 * メソッドで，DBType オブジェクトを取得します。
 * このオブジェクトは、内部的にすべてキャッシュしておき、Webアプリケーション内で
 * 同時アクセスされますが、このオブジェクトは読み取り専用の為，マルチスレッド対応
 * していません。
 * よって、DBTypeFactory.close() メソッドで，オブジェクトを返す必要も
 * ありません。
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBTypeFactory {
	/** 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。  */
//	private static Map<String,DBType> map = Collections.synchronizedMap( new HashMap<>() );
	private static final ConcurrentMap<String,DBType> TYPE_MAP = new ConcurrentHashMap<>();

	// 4.0.0 (2005/01/31) Cleanable インターフェースによる初期化処理
	static {
		final Cleanable clr = new Cleanable() {
			/**
			 * 初期化(クリア)します。
			 * 主に、キャッシュクリアで利用します。
			 */
			public void clear() {
				DBTypeFactory.clear();
			}
		};

		SystemManager.addCleanable( clr );
	}

	/**
	 *  デフォルトコンストラクターをprivateにして、
	 *  オブジェクトの生成をさせないようにする。
	 *
	 */
	private DBTypeFactory() {}

	/**
	 * 識別id に応じた DBType オブジェクトを取得します。
	 * DBType オブジェクト はすべてのWebアプリケーション中で
	 * 共有して使用されます。
	 *
	 * @og.rev 3.4.0.2 (2003/09/05) DBType のデフォルト値を、'X' から 'XK' に変更します。
	 * @og.rev 3.5.6.0 (2004/06/18) 各種プラグイン関連付け設定を、システムパラメータ に記述します。
	 * @og.rev 4.0.0.0 (2005/01/31) キーの指定を、DBType. から、DBType_ に変更します。
	 * @og.rev 5.1.6.0 (2010/05/01) 初期タイプを DBType.DEF_TYPE を使用するように変更します(設定値は、XK のままです。)
	 * @og.rev 6.0.4.0 (2014/11/28) NullPointerException が発生するので、事前にチェックします。
	 * @og.rev 6.4.3.3 (2016/03/04) HybsSystem.newInstance(String,String) への置き換え。
	 *
	 * @param	id DBTypeインターフェースを実装したサブクラスの識別id
	 *
	 * @return	DBTypeオブジェクト
	 */
	public static DBType newInstance( final String id ) {
		final String type = ( id == null ) ? DBType.DEF_TYPE : id.toUpperCase(Locale.JAPAN);

		// Map#computeIfAbsent ： 戻り値は、既存の、または計算された値。追加有り、置換なし、削除なし
		return TYPE_MAP.computeIfAbsent( type , k -> HybsSystem.newInstance( "DBType_" , k ) );

//		DBType dbType = typeMap.get( type );
//		if( dbType == null ) {
//			final String cls = HybsSystem.sys( "DBType_" + type ) ;	// 4.0.0 (2005/01/31)
//
//			// NullPointerException が発生するので、事前にチェックします。
//			if( cls == null ) {
//				final String errMsg = "DBType クラスが見つかりません。class=" + "DBType_" + type ;
//				throw new OgRuntimeException( errMsg );
//			}
//
//			dbType = (DBType)HybsSystem.newInstance( cls );	// 3.5.5.3 (2004/04/09)
//			typeMap.put( type,dbType );
//		}
//		return dbType;
	}

	/**
	 * 内部キャッシュのすべての DBType オブジェクトを削除します。
	 */
	public static void clear() {
		TYPE_MAP.clear() ;
	}
}
