/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.html;

import org.opengion.fukurou.util.StringUtil;

/**
 * 【廃止】タブ表示を行う場合の各タブに対応するデータを管理します。
 *
 * タブ表示には、text , id , body の項目を持っています。
 * このタブ表示には、tabstrip.htc と multipage.htc の２つの JavaScript が必要です。
 * text は、tabstrip の tab に表示する文字列を指定します。
 * id は、multipage の pageview の id を指定します。
 * body は、multipage の pageview の BODY 部に記述する タブの内容です。
 * タブとタブの間には、tabseparator が挿入されます。これは、タブ間の大きさを指定します。
 * 一番最後の tabseparator は、タブの配置方法(縦か横)に応じて変更されます。
 * horizontal の場合は、widt を 100% に、vertical の場合は、height を 100% に設定します。
 * 設定方法は、tabseparator の defaultstyle 属性に style 属性の形式(width:100%)で指定します。
 *
 * @og.rev 3.5.6.5 (2004/08/09) 新規作成
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TabData {
	private final String text ;
	private final String name  ;		// 3.5.6.6 (2004/08/23) id から name へ変更
	private final String body ;
	private final String  style ;		// 3.8.6.1 (2006/10/24)
	private final boolean openFlag ;

	/**
	 * コンストラクター
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) action属性を追加
	 *
	 * @param	text	タブのテキスト
	 * @param	name	multipage の pageview の id を指定します。
	 * @param	body	multipage の pageview の BODY 部に記述するタブの内容を指定します。
	 * @param	openFlag	タブが選択されているかどうか
	 * @param	style	タブに指定するスタイルシート属性を設定します。
	 */
	public TabData( final String text,final String name,final String body,
						final boolean openFlag,final String style ) {
		this.text		= text;
		this.name		= name;
		this.body		= body;
		this.openFlag	= openFlag;
		this.style		= style;
	}

	// 5.9.1.3 (2015/10/30) 削除
//	/**
//	 * tab のタグを作成して返します。
//	 *
//	 * 引数の style が、null でなければ、defaultStyle と selectedStyle に設定します。
//	 * また、タブ単独に直接指定されている場合は、そちらが優先されます。
//	 *
//	 * @param	inStyle	外部より指定されるスタイル
//	 *
//	 * @return	tabのタグ
//	 */
//	public String getTab( final String inStyle ) {
//		return "<ts:tab " + getStyleString( style,inStyle ) + " text=\"" + text + "\" />" ;
//	}

	/**
	 * tab用 のタグを作成して返します。
	 *
	 * 引数の style が、null でなければ、defaultStyle と selectedStyle に設定します。
	 * また、タブ単独に直接指定されている場合は、そちらが優先されます。
	 * 
	 * @og.rev 5.9.1.3 (2015/10/30) 引数追加で新規作成
	 *
	 * @param	inStyle	外部より指定されるスタイル
	 * @param 	i		ID用数値
	 *
	 * @return	tab要のリストタグ
	 */
	public String getTab( final String inStyle, int i ) {
		String id = StringUtil.nval( name,"ogtablist_" + Integer.toString(i) );
		return "<li><a href=\"#" + id + "\"><span " + getStyleString( style,inStyle ) + ">" 
					+ text + "</span></a></li>" ;
	}
	
	// 5.9.1.3 (2015/10/30) 削除
//	/**
//	 * pageview のタグを作成して返します。
//	 * タブの内容を表示するタグを作成します。
//	 *
//	 * @return	pageviewのタグ
//	 *
//	 */
//	public String getTabBody() {
//		return "<mp:pageview id=\"" + name + "\">" + body + "</mp:pageview>" ;
//	}
	
	/**
	 * pageview のタグを作成して返します。
	 * タブの内容を表示するタグを作成します。
	 *
	 * @return	pageviewのタグ
	 * 
	 * @param		i id用数値
	 *
	 */
	public String getTabBody( int i ) {
		String id = StringUtil.nval( name,"ogtablist_" + Integer.toString(i) );
		return "<div id=\"" + id + "\">" + body + "</div>" ;
	}

	/**
	 * タブが選択されているかどうか(true:選択/false:通常)を取得します。
	 *
	 * タブが選択されるかどうかは、tabTag の term,termList が成立するか、
	 * tabTableTag で、selectedIndex 指定されるかです。
	 *
	 * @og.rev 3.8.6.1 (2006/10/24) 新規追加
	 *
	 * @return	タブが選択されているかどうか(true:選択/false:通常)
	 */
	public boolean isOpen() {
		return openFlag ;
	}

	/**
	 * defaultStyle と selectedStyle を指定した style属性を作成します。
	 *
	 * style属性 は、このタブ構築時に指定されたスタイル(defStyle)が優先されます。
	 * これが null の場合は、外部より指定されるスタイル(inStyle)を適用します。
	 * それも null の場合は、ゼロ文字列を返します。
	 *
	 * @og.rev 5.9.1.3 (2015/10/30) 変更
	 *
	 * @param	defStyle	このタブ構築時に指定されたスタイル(優先)
	 * @param	inStyle		外部より指定されるスタイル
	 *
	 * @return	style属性
	 */
	private String getStyleString( final String defStyle, final String inStyle ) {
		String tmp = ( defStyle != null ) ? defStyle : inStyle ;

		String rtn = "";
		if( tmp != null ) {
//			rtn = "defaultStyle=\"" + tmp + "\" selectedStyle=\"" + tmp + "\"";
			rtn = " style=\"" + tmp + "\" ";
		}

		return rtn ;
	}
}
