/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractQuery;
import org.opengion.hayabusa.db.DBErrMsg;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
// import org.opengion.fukurou.system.Closer;

import java.util.Map;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Types;
import java.sql.Array;								// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ) 対応。oracle.sql.ARRAY の置き換え

/**
 * Callableのエラー配列対応版です。バッチ系標準のPL/SQL をコールする Query クラスです。
 *
 * java.sql.CallableStatement を用いて、データベース検索処理を行います。
 * 引数は、従来のPL/SQLの実行が可能なように、第一引数はエラーコード、第二引数は、
 * エラーメッセージを返してきます。第三引数以降は、自由に指定できます。
 * 内部変数の受け渡しのデフォルト実装は、AbstractQuery クラスを継承している
 * ため,ここでは、execute() メソッドを実装しています。
 *
 * @og.formSample
 * 例：
 *     第一引数、第二引数は、通常のPL/SQLと同じ、結果(STATUS)と
 *     内容(ERR_CODE)を返します。
 *     それ以降の引数については、入力(IN)のみですが、自由に設定できます。
 *     引数に変数を使用する場合は、? 記号を当てはめます。
 *     第一引数、第二引数は、予約済みですが、それ以降は、好きな位置に割り当てられます。
 *     names 属性の順番に、値だけがセットされていきます。
 *     下記の例は、変数の引数は、使用していません。
 *
 * &lt;og:query
 *     command=&quot;NEW&quot;
 *     queryType=&quot;JDBCArrayCallable&quot;
 *     displayMsg=&quot;&quot; &gt;
 *         { call GEP00002.GEP00002( ?,?,&#39;{&#064;GUI.KEY}&#39;,&#39;{&#064;USER.ID}&#39; ) }
 * &lt;/og:query&gt;
 *
 *    CREATE OR REPLACE PACKAGE GEP00002 AS
 *        PROCEDURE GEP00002(
 *            P_STATUS    OUT    NUMBER,
 *            P_ERR_CODE  OUT    ERR_MSG_ARRAY,
 *            P_MIDDB     IN     VARCHAR2,
 *            P_USRUPD    IN     VARCHAR2  );
 *    END;
 *
 * @og.group データ表示
 * @og.group データ編集
 *
 * @version  4.0
 * @author   高橋正和
 * @since    JDK5.0,
 */
public class Query_JDBCArrayCallable extends AbstractQuery {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.1 (2016/02/05)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public Query_JDBCArrayCallable() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * ? 部分の引数を
	 * 順番にセットしていきます。
	 *
	 * @og.rev 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応
	 * @og.rev 6.3.6.1 (2015/08/28) close(),realClose() 廃止。Queryはキャッシュしません。
	 * @og.rev 6.4.2.1 (2016/02/05) try-with-resources 文で記述。
	 *
	 * @param   args オブジェクトの引数配列(可変長引数)
	 */
	@Override
	public void execute( final String... args ) {			// 6.1.1.0 (2015/01/17) refactoring
//		CallableStatement callStmt = null ;
//		try {
			final Connection conn = getConnection();
//			callStmt  = getConnection().prepareCall( getStatement() );
		// 6.4.2.1 (2016/02/05) try-with-resources 文
		try( final CallableStatement callStmt = conn.prepareCall( getStatement() ) ) {
			callStmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
			final Map<String,Class<?>> map = conn.getTypeMap();
			map.put( ERR_MSG,DBErrMsg.class );

			// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応
			callStmt.registerOutParameter(1, Types.INTEGER);
			callStmt.registerOutParameter(2, Types.ARRAY,ERR_MSG_ARRAY);
			// 6.1.1.0 (2015/01/17) refactoring. 可変引数にしたため、null は来ない。
				for( int i=0; i<args.length; i++ ) {
					callStmt.setObject( i+3,StringUtil.rTrim( args[i] ) );
				}
			callStmt.execute();

			final int rtnCode = callStmt.getInt(1);
			setErrorCode( rtnCode );

			if( rtnCode > ErrorMessage.OK ) {		// 正常以外の場合
				// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ) 対応
				final Array rtn3 = callStmt.getArray(2);
				final Object[] rtnval3 = (Object[])rtn3.getArray();
				final ErrorMessage errMessage = new ErrorMessage( "Query_JDBCArrayCallable Error!!" );
				for( int i=0; i<rtnval3.length; i++ ) {
					final DBErrMsg er = (DBErrMsg)rtnval3[i];
					if( er == null ) { break; }
					errMessage.addMessage( er.getErrMsg() );
				}
				setErrorMessage( errMessage );
			}
//			commit();										// 6.3.6.1 (2015/08/28)
		}
		catch( final SQLException ex ) {		// catch は、close() されてから呼ばれます。
			setErrorCode( ErrorMessage.EXCEPTION );
			final String errMsg = ex.getMessage() + ":" + ex.getSQLState() + CR
						+ getStatement() + CR;
//			rollback();
//			realClose();									// 6.3.6.1 (2015/08/28) close(),realClose() 廃止。
			throw new HybsSystemException( errMsg,ex );
		}
//		finally {
//			Closer.stmtClose( callStmt );
//		}
	}
}
