/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil;

import java.sql.SQLData;
import java.sql.SQLInput;
import java.sql.SQLOutput;
import java.sql.SQLException;

/**
 * QLData インターフェースを継承した ユーザー変数の受け渡し用オブジェクトです。
 * 登録されている属性情報は、セットメソッドを通して、順番に設定されます。
 *
 * @og.group ＤＢ/Shell制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBUserArg implements SQLData {
	private String sqlType ;

	private final String[] names;
	private String[] values;

	/**
	 *  すべての属性情報を指定して、新しい DBUserArg オブジェクトを作成します。
	 *
	 * @og.rev 3.3.3.1 (2003/07/18) ＤＢ登録時の後ろスペースを削除する。
	 * @og.rev 3.5.6.0 (2004/06/18) 内部に取り込み時に、キー配列は、 arraycopy を行う。
	 *
	 * @param    type データベースタイプ文字列
	 * @param    nms キー配列
	 * @param    vals 属性配列
	 */
	public DBUserArg( final String type,final String[] nms,final String[] vals ) {
		if( nms == null ) {
			final String errMsg = "引数のキー配列が null です。" ;
			throw new HybsSystemException( errMsg );
		}

		final int size = nms.length;
		names = new String[size];
		System.arraycopy( nms,0,names,0,size );

		sqlType = type;
		values   = StringUtil.rTrims( vals );
	}

	/**
	 *  属性配列情報を取得します。
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) 取り出し時に内部配列を clone して返します。
	 * @og.rev 3.6.0.0 (2004/09/22) 属性配列が null の場合は、エラー
	 *
	 * @return    属性配列
	 * @og.rtnNotNull
	 */
	public String[] getValues() {
		if( values != null ) {
			return values.clone();
		}

		final String errMsg = "属性配列が、初期化されていません。";
		throw new HybsSystemException( errMsg );
	}

	// ============================================================
	// implements SQLData
	// ============================================================

	/**
	 *  ＳＱＬタイプの文字列を返します。
	 *
	 * @return    ＳＱＬタイプの文字列
	 * @throws SQLException ※ この実装からは SQLException は、throw されません。
	 */
	public String getSQLTypeName() throws SQLException {
		return sqlType;
	}

	/**
	 *  データベース内部より内部属性を取得し、オブジェクトを構築します。
	 *
	 * @param	stream	ストリーム
	 * @param    typeName ＳＱＬタイプの文字列
	 * @throws SQLException データベースアクセスエラー
	 */
	public void readSQL( final SQLInput stream, final String typeName ) throws SQLException {
		sqlType = typeName;

		values = new String[names.length];
		for( int i=0; i<names.length; i++ ) {
			values[i] = stream.readString();
		}
	}

	/**
	 *  データベース内部に内部属性を設定します。
	 *
	 * @param	stream	ストリーム
	 * @throws SQLException データベースアクセスエラー
	 */
	public void writeSQL( final SQLOutput stream ) throws SQLException {
		for( int i=0; i<names.length; i++ ) {
			stream.writeString( values[i] );
		}
	}
}
