/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.resource;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.SystemManager;
import org.opengion.fukurou.util.Cleanable;

// import java.util.Set;
// import java.util.Map;
// import java.util.HashMap;
// import java.util.Collections ;
import java.util.concurrent.ConcurrentMap;							// 6.4.3.3 (2016/03/04)
import java.util.concurrent.ConcurrentHashMap;						// 6.4.3.3 (2016/03/04)

/**
 * java.util.ResourceBundle クラスを複数管理するResourceManager をリソース毎に作成します。
 * ResourceFactory#newInstance( String lang ) により，ResourceManager の要求毎に
 * 新しくオブジェクトを作成するのではなく,ロケール毎に ResourceManager を作成します。
 * ResourceManagerは,ロケール毎に 内部のプールに保存されています。
 *
 * リソース作成時に指定するロケールは,ISO 言語コード(ISO-639 で定義される 2 桁の小文字)
 * <a href ="http://www.ics.uci.edu/pub/ietf/http/related/iso639.txt">
 * http://www.ics.uci.edu/pub/ietf/http/related/iso639.txt</a>を使用して下さい。
 * ただし，内部的に Locale を構築していますが,その正しさは,チェックされていませんので,
 * 指定するロケールに応じた properties ファイルを用意しておいて下さい。
 *
 * @og.group リソース管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class ResourceFactory {
	private static final String SYSTEM_ID = HybsSystem.sys( "SYSTEM_ID" );

	// デフォルトシステムＩＤの日本語(ja)は、特別扱いする。
	private static final ResourceManager JA_MANAGER = new ResourceManager( SYSTEM_ID,"ja",true );				// 6.4.1.1 (2016/01/16) ja_Manager → JA_MANAGER  refactoring

	/** 6.4.3.1 (2016/02/12) Collections.synchronizedMap で同期処理を行います。  */
//	private static final Map<String,ResourceManager> POOL = Collections.synchronizedMap( new HashMap<>() );		// 6.4.1.1 (2016/01/16) pool → POOL  refactoring
	private static final ConcurrentMap<String,ResourceManager> POOL = new ConcurrentHashMap<>();				// 6.4.3.3 (2016/03/04)

	// 4.0.0 (2005/01/31) Cleanable インターフェースによる初期化処理
	static {
		final Cleanable clr = new Cleanable() {
			/**
			 * 初期化(クリア)します。
			 * 主に、キャッシュクリアで利用します。
			 */
			public void clear() {
				ResourceFactory.clear();
			}
		};

		SystemManager.addCleanable( clr );
	}

	/**
	 *  デフォルトコンストラクターをprivateにして、
	 *  オブジェクトの生成をさせないようにする。
	 *
	 */
	private ResourceFactory() {
	}

	/**
	 * ResourceManager オブジェクトを取得します。
	 * 引数の言語コードに応じたリソースを１度だけ作成します。
	 * 作成したリソースオブジェクトは，内部にプールしておき，同じリソース要求が
	 * あったときは，プールのリソースを返します。
	 *
	 * @param	lang	言語コード(null の場合は、"ja" とします。)
	 *
	 * @return	ResourceManagerオブジェクト
	 * @og.rtnNotNull
	 */
	public static ResourceManager newInstance( final String lang ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return lang == null || "ja".equalsIgnoreCase( lang ) ? JA_MANAGER : newInstance( SYSTEM_ID,lang,true ) ;

//		if( lang == null || "ja".equalsIgnoreCase( lang ) ) {
//			return JA_MANAGER ;
//		}
//		return newInstance( SYSTEM_ID,lang,true );
	}

	/**
	 * ResourceManager オブジェクトを取得します。
	 * 引数の言語コードに応じたリソースを１度だけ作成します。
	 * 作成したリソースオブジェクトは，内部にプールしておき，同じリソース要求が
	 * あったときは，プールのリソースを返します。
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) Map#computeIfAbsent で対応する。
	 *
	 * @param	systemId	システムＩＤ(null の場合は、HybsSystem の SYSTEM_ID パラメータ)
	 * @param	lang		言語コード(null の場合は、"ja" とします。)
	 * @param	initLoad	リソースデータの先読み可否(true:先読みする)
	 *
	 * @return	ResourceManagerオブジェクト
	 */
	public static ResourceManager newInstance( final String systemId,final String lang,final boolean initLoad ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		final String sys = (systemId == null ) ? SYSTEM_ID : systemId ;
		final String lg  = (lang     == null ) ? "ja"      : lang;
//		final String sys = (systemId != null ) ? systemId : SYSTEM_ID ;
//		final String lg  = (lang     != null ) ? lang     : "ja" ;

		if( SYSTEM_ID.equalsIgnoreCase( sys ) && "ja".equalsIgnoreCase( lg ) ) {
			return JA_MANAGER ;
		}

		final String key = sys + lg ;

		// Map#computeIfAbsent ： 戻り値は、新しい値。追加有り、置換有り、削除有り
		return POOL.computeIfAbsent( key , k -> new ResourceManager( sys,lg,initLoad ) );

//		ResourceManager resource = POOL.get( key );
//
//		if( resource == null ) {
//			resource = new ResourceManager( sys,lg,initLoad );
//			POOL.put( key,resource );
//		}
//		return resource;
	}

	/**
	 * キャッシュ(プール)から、すべてのオブジェクトをクリアします。
	 * この時、POOLされているオブジェクトは、ResourceManager#clear() メソッドを
	 * 呼び出します。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) CodeSelectionFactoryをクリアします。
	 * @og.rev 6.4.3.3 (2016/03/04) Map#forEach で対応する。
	 */
	public static void clear() {
		JA_MANAGER.clear();

	//	POOL.forEach( (k,v) -> v.clear() );			// ConcurrentMap なのでnullチェック不要
		POOL.values().forEach( v -> v.clear() );	// ConcurrentMap なのでnullチェック不要
		POOL.clear();

//		final Set<String> keyset = POOL.keySet();
//		final String[] keys = keyset.toArray( new String[keyset.size()] ) ;
//
//		for( int i=0; i<keys.length; i++ ) {
//			final ResourceManager resource = POOL.remove( keys[i] );
//			resource.clear();
//		}
//		POOL.clear();
	}

	/**
	 * キャッシュ(プール)から、すべてのGUI情報オブジェクトをクリアします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 * @og.rev 6.4.3.3 (2016/03/04) Map#forEach で対応する。
	 */
	public static void guiClear() {
		JA_MANAGER.guiClear();

	//	POOL.forEach( (k,v) -> v.guiClear() );			// ConcurrentMap なのでnullチェック不要
		POOL.values().forEach( v -> v.guiClear() );		// ConcurrentMap なのでnullチェック不要

//		final Set<String> keyset = POOL.keySet();
//		final String[] keys = keyset.toArray( new String[keyset.size()] ) ;
//
//		ResourceManager resource ;
//		for( int j=0; j<keys.length; j++ ) {
//			resource = POOL.get( keys[j] );
//			resource.guiClear();
//		}
	}
}
