/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import java.util.stream.Stream;									// 6.4.3.4 (2016/03/11)
import java.util.stream.Collectors;								// 6.4.3.4 (2016/03/11)

/**
 * プルダウンメニューの選択項目を作成するHTML拡張タグです。
 *
 * name 属性は、ラベルリソース のキーを与えることで、使用する上位のタグの
 * ロケールにあわせたリソースを使用して、画面に表示します。
 * 従って、このタグでは ロケールは指定できません。
 * selected属性は、そのタグが選ばれている場合を、"true" で指定します。 初期値は、"false" です。
 *
 * @og.formSample
 * ●形式：&lt;og:option value="…" lbl ="…" selected="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:option
 *       value              【TAG】値を指定します
 *       selected           【TAG】オプションを選択済みの状態(selected)にセットします(初期値:未選択)
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       lbls               【TAG】ラベルをCSV形式で複数指定します
 *       label              【TAG】optionタグのラベルを指定します
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       id                 【HTML】要素に対して固有の名前(id)をつける場合に設定します
 *       lang               【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       dir                【HTML】文字表記の方向(dir)を指定します
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるスタイルシート(style)を設定します
 *       disabled           【TAG】その部品に対して、選択や変更が出来ないように(disabled)指定します(サーバーに送信されない)
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       roles              【TAG】ロールをセットします
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     プルダウンで選択する 値(value) に対して ラベル(lbl) を付けられます。
 *     &lt;og:select name="ORDER_BY" lbl="ORDER_BY"&gt;
 *         &lt;og:option value="SYSTEM_ID,CLM,LANG" lbl ="CLM" selected="selected" /&gt;
 *     &lt;/og:select&gt;
 *
 *     ラベルが複数ある場合は、lbls 属性を使用します。
 *     &lt;og:select name="ORDER_BY" lbl="ORDER_BY"&gt;
 *         &lt;og:option value="SYSTEM_ID,CLM,LANG" lbls="LANG,NAME_JA" /&gt;
 *     &lt;/og:select&gt;
 *
 *     MessageResource プロパティの値を使用したいとききはlbl属性を使います。
 *     &lt;og:select name="CDC"&gt;
 *         &lt;og:option lbl="MSG0001" /&gt;
 *     &lt;/og:select&gt;
 *
 *     LabelResource プロパティの値を使用したいとききはlbl属性を使います。
 *     &lt;og:select name="CDC"&gt;
 *         &lt;og:option lbl="CDC1" /&gt;
 *     &lt;/og:select&gt;
 *
 *     選択肢の中から複数選択できるようにするときはmultiple属性を使います。
 *     &lt;og:select name="CDC" multiple="multiple"&gt;
 *         &lt;og:option value="AAA" /&gt;
 *     &lt;/og:select&gt;
 *
 *     選択不可にするときはdisabled属性を使います。
 *     &lt;og:select name="CDC" disabled="disabled"&gt;
 *         &lt;og:option value="AAA" /&gt;
 *     &lt;/og:select&gt;
 *
 *     選択肢をSELECT文の結果から作成したいときはqueryOptionタグと組み合わせて使います。
 *     &lt;og:select name="CDC"&gt;
 *         &lt;og:queryOption&gt;
 *                     select NOSYN,NOSYN,':',NMSYN from DB01 ORDER BY 1
 *         &lt;/og:queryOption&gt;
 *     &lt;/og:select&gt;
 *
 * @og.rev 5.7.1.0 (2013/12/06) DatalistTag 対応で、大幅に見直し
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class OptionTag extends HTMLTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.5.0.0 (2016/09/30)" ;
	private static final long serialVersionUID = 650020160930L ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public OptionTag() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * ラベルを作成します。
	 *
	 * lbl 属性でセットされた場合は，そちらを優先します。
	 * セットされていない場合は，value 属性をキーに、
	 * LabelResource プロパティの値をセットします。
	 * value 属性に、カンマ(,)で区切られた複数の Label を
	 * セットできます。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) selVal 属性を追加。
	 * @og.rev 3.5.5.7 (2004/05/10) DBColumn.getOption( String ) メソッド廃止
	 * @og.rev 3.8.0.9 (2005/10/17) 複数選択可能時に全選択を設定する。
	 * @og.rev 5.0.2.0 (2009/11/01) 複数パラメーターの選択に対応
	 * @og.rev 5.7.1.0 (2013/12/06) findAncestorWithClass を移動
	 * @og.rev 5.7.1.0 (2013/12/06) SelectTag ⇒ OptionAncestorIF に変更して、DatalistTag にも対応。
	 * @og.rev 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうかの、addKeyLabel 属性対応
	 *
	 * @return  null固定（null を返せば、doEndTag() では、何もしない）
	 */
	@Override
	protected String makeTag() {
		// 3.5.4.0 (2003/11/25) selVal 属性を追加。
		// 5.7.1.0 (2013/12/06) findAncestorWithClass を doEndTag() から移動
		final OptionAncestorIF select = (OptionAncestorIF)findAncestorWithClass( this,OptionAncestorIF.class );
		if( select == null ) {
			final String errMsg = "<b>" + getTagName() + "タグは、SelectTag または、DatalistTag のBODY に記述する必要があります。</b>";
			throw new HybsSystemException( errMsg );
		}

		final String selVal = "|" + select.getValue() + "|";	// 5.0.2.0 (2009/11/01)
		final boolean multipleAll	= select.isMultipleAll();	// 3.8.0.9 (2005/10/17) 複数選択可能時に全選択を設定する。

		if( multipleAll || selVal.indexOf( "|" + get( "value" ) + "|" ) >= 0 ) {
			set( "selected","selected" );
		}

		// 5.7.1.0 (2013/12/06) 上位に上げるのではなく、BODY部に出力する。
		final String msglbl = getMsglbl();
		if( msglbl != null ) {
			set( "body", msglbl );
		}

		// 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか[true:キー:ラベル形式/false:ラベルのみ/null:指定通り]
		select.addOption( XHTMLTag.option( getAttributes() , select.getAddKeyLabel() ) );

		return null;
	}

	/**
	 * 【TAG】値を指定します。
	 *
	 * @og.tag
	 * ここで指定した値がプルダウンメニュー中に存在する場合、選択状態になります。
	 *
	 * @param   val 値
	 */
	public void setValue( final String val ) {
		set( "value",getRequestParameter( val ) );
	}

	/**
	 * 【TAG】optionタグのラベルを指定します。
	 *
	 * @og.tag
	 * ここでのラベルは、optionタグのラベルです。(lbl属性は、異なります。)
	 * これは、optgroup とともに使用される階層化メニュー時に使用されます。
	 *
	 * @param   label ラベル
	 */
	public void setLabel( final String label ) {
		set( "label",getRequestParameter( label ) );
	}

	/**
	 * 【TAG】オプションを選択済みの状態(selected)にセットします(初期値:未選択)。
	 *
	 * @og.tag
	 * selected="selected" または selected="true" 以外の値はセットできないように
	 * 制限をかけます。
	 * 初期値は、未選択 です。
	 *
	 * @param   sel 選択済み状態 [selected:選択済み/それ以外:未選択]
	 */
	public void setSelected( final String sel ) {
		final String select = getRequestParameter( sel );
		if( "selected".equalsIgnoreCase( select ) || "true".equalsIgnoreCase( select ) ) {
			set( "selected","selected" );
		}
	}

	/**
	 * 【TAG】ラベルをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * シングルラベルの lbl 属性との違いは,ここではラベルを複数 CSV形式で
	 * 渡すことが可能であることです。これにより、"A,B,C" という値に対して、
	 * "Aのラベル表示,Bのラベル表示,Cのラベル表示" という具合に文字列を
	 * 再合成します。
	 * これは、SQL文のOrdr By 句で、ソート順を指定する場合などに便利です。
	 * &lt;og:option lbls="MKNMJP,MKCD,MKNMEN" /&gt;
	 *
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 5.2.2.0 (2010/11/01) setMsglbl 廃止 ⇒ setLbl に置換え
	 * @og.rev 6.4.3.4 (2016/03/11) CSV形式の文字連結を、stream 経由で行います。
	 * @og.rev 6.5.0.0 (2016/09/30) " DESC" 文字列を見つけた場合、DESCだけリソース変換します。
	 *
	 * @param   lbls 複数ラベルID(CSV形式)
	 */
	public void setLbls( final String lbls ) {

		final String[] keys = getCSVParameter( lbls );
		if( keys == null || keys.length == 0 ) { return ; }

		final String lblCSV = Stream.of( keys )
//								.map( key -> getLabel( key ) )
								.map( key -> gwtDescLabel( key ) )			// 6.5.0.0 (2016/09/30)
								.collect( Collectors.joining( "," ) );

		setLbl( lblCSV );		// 5.2.2.0 (2010/11/01) setMsglbl 廃止 ⇒ setLbl に置換え

//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
//		buf.append( getLabel( keys[0] ) );
//		for( int i=1; i<keys.length; i++ ) {
//			buf.append( ',' );			// 6.0.2.5 (2014/10/31) char を append する。
//			buf.append( getLabel( keys[i] ) );
//		}
//		setLbl( buf.toString() );		// 5.2.2.0 (2010/11/01) setMsglbl 廃止 ⇒ setLbl に置換え
	}

	/**
	 * ラベルの文字列表現に、DESC 文字列解析を加えました。
	 *
	 * キーに、" DESC" 文字列を見つけた場合、DESC部分をリソース変換して、元のキーの
	 * ラベル文字列に合成します。
	 * 主に、並び順を処理する optionタグで、使用することを想定しています。
	 *
	 * @og.rev 6.5.0.0 (2016/09/30) " DESC" 文字列を見つけた場合、DESCだけリソース変換します。
	 *
	 * @param   key ラベルID
	 * @return DESC文字列解析付きラベルの文字列表現
	 */
	private String gwtDescLabel( final String key ) {
		final int sp = key.indexOf( " DESC" );
		if( sp < 0 ) {
			return getLabel( key );
		}
		else {
			return getLabel( key.substring( 0,sp ) ) + getLabel( "DESC" );
		}
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @og.rev 5.7.1.0 (2013/12/06) selVal と、multipleAll をローカル変数化する。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
