/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.system.LogWriter;
import org.opengion.hayabusa.resource.CodeData;
import org.opengion.fukurou.util.StringUtil ;						// 6.2.2.0 (2015/03/27)

import static org.opengion.fukurou.system.HybsConst.CR ;				// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * データのコード情報を取り扱うクラスです。
 *
 * コードのキーとラベルの情報から、HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したり
 * します。
 *
 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 基本実装とします。
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Selection_CODE extends Selection_NULL {
	private final CodeData codeData ;

	private final int[]    ADRS  ;
	private final String   CACHE ;
	private final int      LEN   ;

	private final int[]    LADRS  ;		// 5.1.3.0 (2010/02/01)
	private final String   LCACHE ;		// 5.1.3.0 (2010/02/01)
	private final int      LLEN   ;		// 5.1.3.0 (2010/02/01)

	private final String addKeyLabel ;	// 6.2.0.0 (2015/02/27) キー:ラベル形式

	/**
	 * コンストラクター
	 *
	 * @og.rev 4.0.0.0 (2007/11/07) コードリソースの有無はDBColumnでチェックする。
	 * @og.rev 4.3.8.0 (2009/08/01) ツールチップ表示機能追加
	 * @og.rev 5.1.3.0 (2010/02/01) 一覧表示以外は、ツールチップ表示しない
	 * @og.rev 5.1.9.0 (2010/08/01) グループ機能とパラメータ機能の追加
	 * @og.rev 5.6.8.2 (2013/09/20) ツールチップは未加工のlongLabelを利用に修正
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、super( null ); を追加します。
	 * @og.rev 6.2.0.0 (2015/02/27) コードリソースのパラメータの指定方法を変更します。
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 * @og.rev 6.2.2.0 (2015/03/27) BRと\nを相互に変換する処理を追加
	 * @og.rev 6.2.2.3 (2015/04/10) htmlフィルターに、BR→改行処理機能を追加。
	 *
	 * @param	cdData	コードデータオブジェクト
	 * @param	addKeyLabel キー:ラベル形式で表示するかどうか[true/false/null]
	 */
	public Selection_CODE( final CodeData cdData,final String addKeyLabel ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		codeData = cdData ;
		this.addKeyLabel = addKeyLabel;			// 6.2.0.0 (2015/02/27) キー:ラベル形式
		final int size = codeData.getSize();

		// 5.1.9.0 (2010/08/01) グループ機能とパラメータの判定
		final boolean isLbl = codeData.useLevel();
		final boolean isCls = codeData.useParam();
		boolean isLblSet = false;				// optgroup の設定に使う。

		ADRS  = new int[size];
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		boolean useTitle = false;								// 4.3.8.0 (2009/08/01) title属性付加フラグ
		for( int i=0; i<size; i++ ) {
			if( ! codeData.isUse(i) ) { continue; }

			// 6.2.0.0 (2015/02/27) キー:ラベル形式
			final String cdkey = codeData.getCodeKey(i);
			final String kv = "true".equalsIgnoreCase( addKeyLabel ) ? cdkey + ':' : "" ;
			final String rslbl = codeData.getRawShortLabel(i);

			// 5.1.9.0 (2010/08/01) コードレベルの判定
			if( isLbl && "0".equals( codeData.getCodeLevel(i) ) ) {
				if( isLblSet ) { buf.append( "</optgroup>" ); }

				buf.append( "<optgroup label=\"" + rslbl + "\">" );
				isLblSet = true;
				continue;
			}

			// 6.0.2.5 (2014/10/31) char を append する。
			// 6.2.0.0 (2015/02/27) キー:ラベル形式
			buf.append( "<option value=\"" ).append( cdkey ).append( '"' );
			ADRS[i] = buf.length() ;

			// 5.1.9.0 (2010/08/01) パラメータの利用
			if( isCls ) {
				// 6.2.0.0 (2015/02/27) コードリソースのパラメータの指定方法を変更します。
				final String paramKey = codeData.getCodeParam(i);
				setCodeParam( buf,paramKey );
			}

			// 6.2.0.0 (2015/02/27) Description があれば、優先して title 属性に設定します。
			final String desc = codeData.getDescription(i);
			if( desc.isEmpty() ) {
				// 4.3.8.0 (2009/08/01) LongLabelはRawShortLabelと異なるときにツールチップとして表示させる
				final String lblb  = codeData.getRawLongLabel(i); // 5.6.8.2 (2013/09/20) ここでは生の名称長を取る
				if( ! rslbl.equals( lblb ) ){
					// 6.2.2.0 (2015/03/27) BRと\nを相互に変換する処理を追加
					buf.append(" title=\"").append( StringUtil.htmlFilter( lblb,true ) ).append( '"' );
					useTitle = true;
				}
			}
			else {
				// 6.2.2.0 (2015/03/27) BRと\nを相互に変換する処理を追加
				buf.append(" title=\"").append( StringUtil.htmlFilter( desc,true ) ).append( '"' );
				useTitle = true;
			}

			// ShortLabelをBODYにする。
			// 6.2.0.0 (2015/02/27) キー:ラベル形式
			buf.append( '>' ).append( kv ).append( rslbl ).append( "</option>" );
		}
		if( isLbl ) {
			buf.append( "</optgroup>" );
		}

		CACHE = buf.toString();
		LEN   = CACHE.length() + 30;

		// 5.1.3.0 (2010/02/01) ツールチップ表示が適用されている場合のみ、ツールチップなしの状態のoptionをキャッシュする。
		if( useTitle ) {
			isLblSet = false;

			LADRS  = new int[size];
			final StringBuilder lbuf = new StringBuilder( BUFFER_MIDDLE );
			for( int i=0; i<size; i++ ) {
				// 6.2.0.0 (2015/02/27) キー:ラベル形式
				final String cdkey = codeData.getCodeKey(i);
				final String kv = "true".equalsIgnoreCase( addKeyLabel ) ? cdkey + ':' : "" ;
				final String lblb  = codeData.getLongLabel(i);
				// 5.1.9.0 (2010/08/01) グループ機能の判定
				if( isLbl && "0".equals( codeData.getCodeLevel(i) ) ) {
					if( isLblSet ) { lbuf.append( "</optgroup>" ); }

					lbuf.append( "<optgroup label=\"" + lblb + "\">" );
					isLblSet = true;
					continue;
				}

				// 6.0.2.5 (2014/10/31) char を append する。
				// 6.2.0.0 (2015/02/27) キー:ラベル形式
				lbuf.append( "<option value=\"" ).append( cdkey ).append( '"' );
				LADRS[i] = lbuf.length() ;

				// 5.1.9.0 (2010/08/01) パラメータの利用
				if( isCls ) {
					// 6.2.0.0 (2015/02/27) コードリソースのパラメータの指定方法を変更します。
					final String paramKey = codeData.getCodeParam(i);
					setCodeParam( lbuf,paramKey );
				}

				// 6.2.0.0 (2015/02/27) キー:ラベル形式
				lbuf.append( '>' ).append( kv ).append( lblb ).append( "</option>" );
			}
			if( isLbl ) {
				buf.append( "</optgroup>" );
			}

			LCACHE = lbuf.toString();
			LLEN   = LCACHE.length() + 30;
		}
		else {
			LADRS  = ADRS;
			LCACHE = CACHE;
			LLEN   = LEN;
		}
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このメソッドでは、引数のuseShortLabelがtrueに指定された場合に、ラベル(短)をベースとした
	 * ツールチップ表示を行います。
	 *
	 * @og.rev 5.1.3.0 (2010/02/01) 追加
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 * @param   useShortLabel ラベル(短)をベースとしたオプション表示を行うかどうか。
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		int[] adrs = null;
		String cache = null;
		int len = 0;
		if( useShortLabel ) {
			adrs  = ADRS;
			cache = CACHE;
			len   = LEN;
		}
		else {
			adrs  = LADRS;
			cache = LCACHE;
			len   = LLEN;
		}

		// マッチするアドレスを探す。
		final int selected = codeData.getAddress( selectValue );

		if( selected < 0 ) {
			// 4.0.0 (2005/01/31)
			if( selectValue != null && selectValue.length() > 0 ) {
				final String errMsg = "コードに存在しない値が指定されました。"
							+ " value=[" + selectValue + "]"
							+ CR + codeData ;
				LogWriter.log( errMsg );
			}
			return cache;
		}
		else {
			final StringBuilder buf = new StringBuilder( len + 100 );						// 6.1.0.0 (2014/12/26) refactoring
			// 3.6.0.6 (2004/10/22) シーケンスアクセス機能を指定する seqFlag を導入
			if( seqFlag ) {
				buf.append( "<option value=\"" ).append( codeData.getCodeKey(selected) ).append( '"' );		// 6.0.2.5 (2014/10/31) char を append する。
			}
			else {
				buf.append( cache.substring( 0,adrs[selected] ) );
			}
			buf.append( " selected=\"selected\"" )
				.append( cache.substring( adrs[selected] ) );
			return buf.toString() ;
		}
	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) を追加
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 *
	 * @param	selectValue	選択肢の値
	 * @param	isSLbl	短縮ラベル使用可否 [true:使用する/false:しない]
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean isSLbl ) {
		// マッチするアドレスを探す。
		final int selected = codeData.getAddress( selectValue );

		if( selected < 0 ) {
			// マッチしなければ、選択肢そのものを返す。
			return selectValue;
		}
		else {
			// 6.2.0.0 (2015/02/27) キー:ラベル形式
			final String lbl = isSLbl ? codeData.getShortLabel(selected) : codeData.getLongLabel(selected);

			return "true".equalsIgnoreCase( addKeyLabel ) ? selectValue + ':' + lbl : lbl ;
		}
	}

}
