/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;

import static org.opengion.fukurou.util.StringUtil.isNull;				// 6.1.1.0 (2015/01/17)

/**
 * TEXTAREA エディターは、カラムのデータをテキストエリアで編集する場合に
 * 使用するクラスです。
 *
 * 従来との違いは、cols 属性の最大値を、検索時(query画面)では、HTML_COLUMNS_MAXSIZE を、
 * 登録時(result画面)では、HTML_VIEW_COLUMNS_MAXSIZE を使用します。
 * エリアの大きさは、デフォルト値を使用するか、編集パラメータに、x,y形式で
 * 指定します。
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_TEXTAREA extends AbstractEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.4.2 (2016/04/01)" ;

//	private  String	cols1	= null	 ;		// 6.4.4.2 (2016/04/01) size1 を使わずに、cols1 を使用
//	private  String	cols2	= null	 ;		// 6.4.4.2 (2016/04/01) size2 を使わずに、cols2 を使用
	private  String	cols1		 ;		// 6.4.4.2 (2016/04/01) size1 を使わずに、cols1 を使用
	private  String	cols2		 ;		// 6.4.4.2 (2016/04/01) size2 を使わずに、cols2 を使用
	private  String	rows1		 ;
	private  String	rows2		 ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 *
	 */
	public Editor_TEXTAREA() {
		super();			// PMD:Document empty constructor 対策
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.2.4.0 (2003/06/12) パラメータより、行列(row,column)情報があれば、その値を利用する。
	 * @og.rev 3.3.1.1 (2003/07/03) name , attributes 属性を final にする。
	 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
	 * @og.rev 3.5.5.0 (2004/03/12) 漢字入力(IMEモード)をONにするのを、"K" と、"KX" のみとします。
	 * @og.rev 3.5.6.0 (2004/06/18) XHTMLTag の 内部配列 TEXTAREA_KEY を隠蔽します。
	 * @og.rev 4.0.0.0 (2005/01/31) DBColumn の getClassName() を getDbType() に変更
	 * @og.rev 6.4.4.2 (2016/04/01) size1,size2 を使わずに、cols1,cols2 を使用。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
//	private Editor_TEXTAREA( final DBColumn clm ) {
	protected Editor_TEXTAREA( final DBColumn clm ) {
		super( clm );
		final String  disabled = clm.isWritable() ? null : "disabled" ;

		final int r1 = clm.getTotalSize()/Integer.parseInt(size1) + 1;		// 4.0.0 (2005/01/31) メソッド名変更
		if( r1 > 5 ) { rows1 = "5"; }
		else { rows1 = String.valueOf( r1 ); }

		final int r2 = clm.getTotalSize()/Integer.parseInt(size2) + 1;		// 4.0.0 (2005/01/31) メソッド名変更
		if( r2 > 5 ) { rows2 = "5"; }
		else { rows2 = String.valueOf( r2 ); }

		// 3.7.0.4 (2005/03/14) size に、"rows,cols" を指定できるように変更
		final String param = StringUtil.nval( clm.getEditorParam(),clm.getViewLength() );
		if( param != null && param.length() != 0 ) {
			final int st = param.indexOf( ',' );
			if( st > 0 ) {
				rows1 = param.substring( 0,st );
				rows2 = rows1 ;
//				size1 = param.substring( st+1 );
//				size2 = size1;
				cols1 = param.substring( st+1 );
				cols2 = cols1;
			}
		}

		// 6.4.4.2 (2016/04/01) size1,size2 を使わずに、cols1,cols2 を使用。
		if( cols1 == null || cols2 == null ) {
			cols1	= size1  ;
			cols2	= size2	 ;
		}

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		attributes = new Attributes()
					.set( "disabled"	, disabled )
					.set( clm.getEditorAttributes() )				// #addAttributes( Attributes ) の代替え
					.add( "class"		, clm.getDbType() );		// 4.0.0 (2005/01/31)

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		tagBuffer.add( XHTMLTag.textareaAttri( attributes ) )
				 .add( attributes.get( "optionAttributes" ) );		// 6.0.4.0 (2014/11/28)

	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_TEXTAREA( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.2.0 (2010/01/01) 先頭の'_'による書き込み制御を行わない。(他のクラスとの実装の共通化)
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 6.4.4.2 (2016/04/01) size1,size2 を使わずに、cols1,cols2 を使用。
	 *
	 * @param   value 入力値
	 *
	 * @return  データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "textarea" )
						.add( "name"	, name )
						.add( "id"		, name , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
//						.add( "cols"	, size1 )
						.add( "cols"	, cols1 )				// 6.4.4.2 (2016/04/01)
						.add( "rows"	, rows1 )
						.add( tagBuffer.makeTag() )
						.addBody( value )
						.makeTag();

	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) 名前と行番号の区切り記号を "^" から "__" に変更。
	 * @og.rev 3.5.5.0 (2004/03/12) 名前と行番号の区切り記号("__")を、HybsSystem.JOINT_STRING  に変更。
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.2.0 (2010/01/01) 先頭の'_'による書き込み制御を行わない。(他のクラスとの実装の共通化)
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 6.4.4.2 (2016/04/01) size1,size2 を使わずに、cols1,cols2 を使用。
	 *
	 * @param   row   行番号
	 * @param   value 入力値
	 *
	 * @return  データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final String newName = name + HybsSystem.JOINT_STRING + row;

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "textarea" )
						.add( "name"	, newName )
						.add( "id"		, newName , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
//						.add( "cols"	, size2 )
						.add( "cols"	, cols2 )				// 6.4.4.2 (2016/04/01)
						.add( "rows"	, rows2 )
						.add( tagBuffer.makeTag() )
						.addBody( value )
						.makeTag( row,value );

	}
}
