/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.mail;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.util.List;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import javax.mail.Address;
import javax.mail.SendFailedException;
import javax.mail.internet.InternetAddress;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.mail.MailTX;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;

/**
 * タグ mailSender2 及びバッチによる送信の共通処理部分を実装しています。
 * 送信タグ mailSender2 もしくは送信デーモンからパラメータを受取ります。
 * パラメータ中の定型文ID及びシステムIDで定型文マスタよりメールの定型文を取得して、
 * パラメータ値とマージしてメール文を合成します。同時に、宛先にセットした社員ID、
 * グループIDと定型文の宛先設定に基づき、社員マスタとグループマスタよりメールアドレス
 * 情報を取得して送信を行います。
 * エラーがなければ送信した内容を履歴テーブル、宛先テーブルに書き込みます。
 * 最後に本処理の呼出元に送信結果、エラーメッセージを返します。
 *
 * @og.group メールモジュール
 *
 * @version  4.0
 * @author   Sen.Li
 * @since    JDK1.6
 */
public abstract class AbstractMailManager {

	// 5.2.0.0 (2010/09/01) Ver4互換モード対応
	private static final String CONTENTS = HybsSystem.sysBool( "VER4_COMPATIBLE_MODE" ) ? "CONTENT" : "CONTENTS";

	private static final String	selYkno	= "SELECT GE32S02.NEXTVAL YKNO FROM DUAL";
	// 5.0.3.0 (2009/11/04) CONTENT ⇒ CONTENTS
	// 5.2.0.0 (2010/09/01) Ver4互換モード対応
//	private static final String	insGE32		= "INSERT INTO GE32(YKNO,PARA_KEY,PTN_ID,FROM_ADDR,TITLE,CONTENTS,ATTACH1,ATTACH2,ATTACH3,ATTACH4,ATTACH5,DYSET,USRSET,PGUPD,SYSTEM_ID,FGJ)"
	private static final String	insGE32		= "INSERT INTO GE32(YKNO,PARA_KEY,PTN_ID,FROM_ADDR,TITLE,"+CONTENTS+",ATTACH1,ATTACH2,ATTACH3,ATTACH4,ATTACH5,DYSET,USRSET,PGUPD,SYSTEM_ID,FGJ)"
											+ " VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,'1')";
	private static final String insGE34		= "INSERT INTO GE34(YKNO,DST_ID,GROUP_ID,DST_NAME,DST_ADDR,DST_KBN,FGJ,DYSET,USRSET,PGUPD)"
											+ " VALUES(?,?,?,?,?,?,?,?,?,?)";
	/** フラグ定数 {@value} */
	protected static final String FGJ_SEND_WAIT	= "0";
	/** フラグ定数 {@value} */
	protected static final String FGJ_SEND_OVER	= "1";
	/** フラグ定数 {@value} */
	protected static final String FGJ_ADDR_ERR	= "7";
	/** フラグ定数 {@value} */
	protected static final String FGJ_ACNT_ERR	= "8";

	// 内部データのカラム番号(履歴テーブル)
	private static final int GE32_YKNO		= 0 ;
	private static final int GE32_PARAKEY	= 1 ;
	private static final int GE32_PTN_ID 	= 2;
	private static final int GE32_FROM_ADDR	= 3;
	private static final int GE32_TITLE 	= 4;
	private static final int GE32_CONTENTS	= 5;		// 5.0.3.0 (2009/11/04) CONTENT ⇒ CONTENTS
	private static final int GE32_ATTACH1	= 6;
	private static final int GE32_ATTACH2	= 7;
	private static final int GE32_ATTACH3	= 8;
	private static final int GE32_ATTACH4	= 9;
	private static final int GE32_ATTACH5	= 10;
	private static final int GE32_DYSET		= 11;
	private static final int GE32_USRSET	= 12;
	private static final int GE32_PGUPD	= 13;
	private static final int GE32_SYSTEM_ID	= 14;
	// 内部データのカラム番号(履歴テーブル)
	private static final int GE34_YKNO	 	= 0 ;
	private static final int GE34_DST_ID 	= 1 ;
	private static final int GE34_GROUP_ID 	= 2 ;
	private static final int GE34_DST_NAME	= 3 ;
	private static final int GE34_DST_ADDR	= 4 ;
	private static final int GE34_DST_KBN	= 5 ;
	private static final int GE34_FGJ		= 6 ;
	private static final int GE34_DYSET		= 7 ;
	private static final int GE34_USRSET	= 8 ;
	private static final int GE34_PGUPD	= 9 ;
//	private static String		host		= HybsSystem.sys( "COMMON_MAIL_SERVER" );
	private static String		charset		= HybsSystem.sys( "MAIL_DEFAULT_CHARSET" );
//	private static String		smtpPort	= HybsSystem.sys( "SMTP_PORT" );				// 5.4.3.2 (2012/01/06)
//	private static String		auth		= HybsSystem.sys( "MAIL_SEND_AUTH" );			// 5.4.3.2 (2012/01/06)
//	private static String		authUser	= HybsSystem.sys( "MAIL_SEND_AUTH_USER" );		// 5.4.3.2 (2012/01/06)
//	private static String		authPass	= HybsSystem.sys( "MAIL_SEND_AUTH_PASSWORD" );	// 5.4.3.2 (2012/01/06)
	private boolean	   			debugFlag   = false;
	private final List<String>	errAddrList = new ArrayList<String>();
	private static final int MAX_RETRY 		= 3 ;	// メールアドレスエラー発生した場合、メール再送回数

	// 5.6.6.0 (2013/07/05) host等の外部指定に対応
	private String		host		= HybsSystem.sys( "COMMON_MAIL_SERVER" );		// 5.6.6.0 (2013/07/05)
	private String		smtpPort	= HybsSystem.sys( "SMTP_PORT" );				// 5.6.6.0 (2013/07/05)
	private String		auth		= HybsSystem.sys( "MAIL_SEND_AUTH" );			// 5.6.6.0 (2013/07/05)
	private String		authPort	= HybsSystem.sys( "MAIL_SEND_AUTH_PORT" );		// 5.8.1.1 (2014/11/14)
	private String		authUser	= HybsSystem.sys( "MAIL_SEND_AUTH_USER" );		// 5.6.6.0 (2013/07/05)
	private String		authPass	= HybsSystem.sys( "MAIL_SEND_AUTH_PASSWORD" );	// 5.6.6.0 (2013/07/05)

	private String	mailTitle, mailContent, fromAddr;
	private String[] attachFiles;
	private Map<String, String[]>	mailDstMap		= null;
	private Map<String,String>		initParamMap	= null;		// パラメータマップ
	private MailTX					mail			= null;

	protected final String DBID = HybsSystem.sys( "RESOURCE_DBID" );		// 5.5.5.1 (2012/08/07) リソース系DBID 付け忘れ対応

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	private static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	/** アプリケーション情報 */
	public static final ApplicationInfo appInfo;
	static {
		if( USE_DB_APPLICATION_INFO ) {
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			appInfo.setClientInfo( "MailModuel", HybsSystem.HOST_ADRS, HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "MailModuel", "MailManager", "MailManager" );
		}
		else {
			appInfo = null;
		}
	}

	/**
	 * 呼出元よりパラメータマップを受取って、メールオブジェクト(mailObj)を作成します。
	 * メールオブジェクトは定型文ID及びシステムIDに基づいて定型文マスタからメールの定型文を取得して、
	 * パラメータ値とマージしてメール文の各項目を合成します。
	 * 宛先については、宛先にセットした社員ID、グループIDと定型文の宛先設定に基づき、社員マスタとグループマスタ
	 * よりメールアドレスを取得して宛先マップを作成します。
	 * まだ、添付ファイルのセット処理も行っています。
	 * 
	 * @og.rev 5.6.6.0 (2013/07/05) host等の外部設定対応
	 * @og.rev 5.9.15.1 (2016/12/09) エラーリストのクリア
	 *
	 * @param	params	パラメータのマップ
	 */
	public void create( final Map<String, String> params ) {
		initParamMap = params;
		MailPattern mailObj = new MailPattern( params );
		fromAddr = mailObj.getFromAddr();
		setAttachFiles( params.get( "ATTACH1" )
					  , params.get( "ATTACH2" )
					  , params.get( "ATTACH3" )
					  , params.get( "ATTACH4" )
					  , params.get( "ATTACH5" ) ); // 添付ファイルのセット
		mailDstMap = mailObj.getDstMap();
		mailTitle = mailObj.getTitle();
		mailContent = mailObj.getContent();
		errAddrList.clear(); // 5.9.15.1 (2016/12/09)
		errAddrList.addAll( mailObj.getErrList() );
		
		// 5.6.6.0 (2013/07/05)
		host 		= nval( mailObj.getHost() 		,host		);
		smtpPort 	= nval( mailObj.getSmtpPort() 	,smtpPort	);
		auth 		= nval( mailObj.getAuth() 		,auth		);
		authPort	= nval( mailObj.getAuthPort()	,authPort	);		// 5.8.1.1 (2014/11/14)
		authUser	= nval( mailObj.getAuthUser() 	,authUser	);
		authPass	= nval( mailObj.getAuthPass() 	,authPass	);
	}

	/**
	 * メール送信を行うメソッドです。
	 * createメソッドより合成した内容で送信を行います。アドレスエラー発生時に、
	 * ユーザー設定(addrCheck)により再送/中止します。
	 * メールサーバーに送信して、例外"SendFailedException"をキャッチできたら、
	 * メールアカウントがエラーとなるのを分かります。そして、例外のオブジェクトから
	 * エラーとなっているアカウントを取得して宛先マップから除外して、残りのアドレスに再送できます。
	 * 送信後履歴テーブル(GE32)、宛先テーブル(GE34)に送信結果を書き込みます。
	 *
	 * og.rev 5.4.3.2 (2012/01/06) 送信時認証対応
	 *
	 */
	public void send() {
		List<String> invalidAddrBuf	= new ArrayList<String>();
		// mail = new MailTX( host, charset );
		mail = new MailTX( host, charset, smtpPort, auth, authPort,authUser, authPass ); // 5.4.3.2 認証対応
		mail.setFrom( fromAddr );			// 送信者アドレス
		mail.setFilename( attachFiles ); 	// 添付ファイルをセットします。
		mail.setSubject( mailTitle );		// メールタイトル
		mail.setMessage( mailContent );		// メール本文
		mail.setDebug( debugFlag );
		setMailDst( invalidAddrBuf ); 		// 宛先をセットします。
		// メール送信を行います。
		int retryCount = MAX_RETRY;
		while( retryCount > 0 ) {
			try {
				mail.sendmail();
			}
			catch( RuntimeException rex ) {
				Throwable cause = rex.getCause();
				if( cause instanceof SendFailedException ) {
					Address[] invAddress = ( (SendFailedException) cause ).getInvalidAddresses();
					if( invAddress != null ) {
						int invCount = invAddress.length;
						for( int i = 0; i < invCount; i++ ) {
							invalidAddrBuf.add( ( (InternetAddress) invAddress[i] ).getAddress() );
						}
					}
				}
				else {
					String errMsg = "送信時にエラー発生しました。" + rex.getMessage();
					throw new RuntimeException( errMsg,rex );
				}
			}

//			if( invalidAddrBuf.size() > 0 ) {
//				StringBuilder errMsgBuf = new StringBuilder();
//				for( int i = 0; i < invalidAddrBuf.size(); i++ ) {
//					errMsgBuf.append( "," ).append( invalidAddrBuf.get( i ) );
//				}
//				String userIds = getUserIds( invalidAddrBuf );
//				String errMsg = "アドレスエラー。ユーザーID：" + userIds + " アドレス：" + errMsgBuf.toString().substring( 1 );
//				if( "true".equals( initParamMap.get( "ADDR_CHECK" ) ) ){
//					throw new RuntimeException( errMsg );
//				}
//				else {
//					// メールアカウントチェックしない場合、無効のメールアドレスを除いて再送します。
//					setMailDst( invalidAddrBuf );
//					retryCount--;
//					invalidAddrBuf.clear();
//					errAddrList.add( errMsg );
//				}
//			}
//			else {
//				retryCount = -1;
//			}

			if( invalidAddrBuf.isEmpty() ) {
				retryCount = -1;
			}
			else {
				StringBuilder errMsgBuf = new StringBuilder();
				for( int i = 0; i < invalidAddrBuf.size(); i++ ) {
					errMsgBuf.append( ',' ).append( invalidAddrBuf.get( i ) );
				}
				String userIds = getUserIds( invalidAddrBuf );
				String errMsg = "アドレスエラー。ユーザーID：" + userIds + " アドレス：" + errMsgBuf.toString().substring( 1 );
				if( "true".equals( initParamMap.get( "ADDR_CHECK" ) ) ){
					throw new RuntimeException( errMsg );
				}
				else {
					// メールアカウントチェックしない場合、無効のメールアドレスを除いて再送します。
					setMailDst( invalidAddrBuf );
					retryCount--;
					invalidAddrBuf.clear();
					errAddrList.add( errMsg );
				}
			}
		}
		commitMailDB();		// 送信結果を履歴テーブル、宛先テーブルにセットします。
	}

	/**
	 * デバッグ情報の表示を行うかどうか[true/false]をセットします。
	 *
	 * @param   debug  [true:出力する/それ以外:しない]
	 */
	public void setDebug( final boolean debug ) {
		debugFlag = debug;
	}

	/**
	 * メール送信者アドレスをセットします。
	 *
	 * @param  from 送信者アドレス
	 */
	public void setFromAddr( final String from ) {
		fromAddr = from;
	}

	/**
	 * メールタイトルをセットします。
	 *
	 * @param  title メールタイトル
	 */
	public void setTitle( final String title ) {
		mailTitle = title;
	}

	/**
	 * メール本文をセットします。
	 *
	 * @param  content メール本文
	 */
	public void setContent( final String content ) {
		mailContent = content;
	}

	/**
	 * メール送信ホストをセットします。
	 * 初期値は、システム定数のCOMMON_MAIL_SERVER を使用します。
	 *
	 * (初期値:システム定数のCOMMON_MAIL_SERVER[={@og.value org.opengion.hayabusa.common.SystemData#COMMON_MAIL_SERVER}])。
	 * 
	 * @og.rev 5.6.6.0 (2013/07/05)
	 *
	 * @param  hostName 送信ホスト
	 */
	public void setHost( final String hostName ) {
		host = nval( hostName, host );
	}

	/**
	 * メール送信ポート番号をセットします。
	 * 初期値は、システム定数のSMTP_PORT を使用します。
	 *
	 * (初期値:システム定数のSMTP_PORT[={@og.value org.opengion.hayabusa.common.SystemData#SMTP_PORT}])。
	 * 
	 * @og.rev 5.6.6.0 (2013/07/05)
	 * 
	 * @param  port SMTPポート
	 */
	public void setPort( final String port ) {
		smtpPort = nval( port, smtpPort );
	}

	/**
	 * メール送信時認証有無をセットします。
	 * 認証を行う場合は「POP_BEFORE_SMTP」と指定して下さい。
	 * 認証時には認証ユーザと認証パスワードを設定する必要があります。
	 * 初期値は、システム定数のMAIL_SEND_AUTH を使用します。
	 *
	 * (初期値:システム定数のMAIL_SEND_AUTH[={@og.value org.opengion.hayabusa.common.SystemData#MAIL_SEND_AUTH}])。
	 * 
	 * @og.rev 5.6.6.0 (2013/07/05)
	 * 
	 * @param  useAuth 認証方式
	 */
	public void setAuth( final String useAuth ) {
		auth = nval( useAuth, auth );
	}

	/**
	 * メール送信認証ユーザをセットします。
	 * 初期値は、システム定数のMAIL_SEND_AUTH_USER を使用します。
	 *
	 * (初期値:システム定数のMAIL_SEND_AUTH_USER[={@og.value org.opengion.hayabusa.common.SystemData#MAIL_SEND_AUTH_USER}])。
	 * 
	 * @og.rev 5.6.6.0 (2013/07/05)
	 * 
	 * @param  user 認証ユーザ
	 */
	public void setAuthUser( final String user ) {
		authUser = nval( user, authUser );
	}

	/**
	 * メール送信認証パスワードをセットします。
	 * 初期値は、システム定数のMAIL_SEND_AUTH_PASSWORD を使用します。
	 *
	 * (初期値:システム定数のMAIL_SEND_AUTH_PASSWORD[={@og.value org.opengion.hayabusa.common.SystemData#MAIL_SEND_AUTH_PASSWORD}])。
	 * 
	 * @og.rev 5.6.6.0 (2013/07/05)
	 * 
	 * @param  pass 認証パスワード
	 */
	public void setAuthPass( final String pass ) {
		authPass = nval( pass, authPass );
	}

	/**
	 * メール送信者アドレスを返します。
	 *
	 * @return	送信者アドレス
	 */
	public String getFromAddr() {
		return fromAddr;
	}

	/**
	 * メールタイトルを返します。
	 *
	 * @return	メールタイトル
	 */
	public String getTitle() {
		return mailTitle;
	}

	/**
	 * メール本文を返します。
	 *
	 * @return	メール本文
	 */
	public String getContent() {
		return mailContent;
	}

	/**
	 * 送信結果を履歴テーブル(GE32)と宛先テーブル(GE34)に登録します。
	 * 登録時に、桁数オーバーにならないように、テーブル定義の桁数を上限として、
	 * 登録前に各項目の桁数整理を行います。
	 *
	 * @og.rev 5.5.5.1 (2012/08/07) リソース系DBID 付け忘れ対策
	 * @og.rev 5.9.1.3 (2015/10/30) 4000文字カットをやめる
	 */
	protected void commitMailDB(){
		// 履歴テーブルの追加
		String[] insGE32Args = new String[15];
		String ykno = getYkno();
		insGE32Args[GE32_YKNO]		= ykno;
		insGE32Args[GE32_PARAKEY] 	= initParamMap.get( "PARAKEY" );
		insGE32Args[GE32_PTN_ID] 	= trim( initParamMap.get( "PTN_ID" ), 20 );
		insGE32Args[GE32_FROM_ADDR] = trim( fromAddr, 100);
		insGE32Args[GE32_TITLE] 	= trim( mailTitle, 300);
//		insGE32Args[GE32_CONTENTS] 	= trim( mailContent,4000);		// 5.0.3.0 (2009/11/04) CONTENT ⇒ CONTENTS
		insGE32Args[GE32_CONTENTS] 	= mailContent;					// 5.9.1.3 (2015/10/30)
		insGE32Args[GE32_ATTACH1] 	= "";
		insGE32Args[GE32_ATTACH2] 	= "";
		insGE32Args[GE32_ATTACH3] 	= "";
		insGE32Args[GE32_ATTACH4] 	= "";
		insGE32Args[GE32_ATTACH5] 	= "";
		if ( attachFiles != null ) {
			int attSize = attachFiles.length;
			for( int i = 0; i < attSize; i++ ) {
				insGE32Args[6 + i] = trim( attachFiles[i], 256);
			}
		}
		insGE32Args[GE32_DYSET]  = HybsSystem.getDate( "yyyyMMddHHmmss" );
		insGE32Args[GE32_USRSET] = initParamMap.get( "LOGIN_USERID" );
		insGE32Args[GE32_PGUPD] = initParamMap.get( "PGID" );
		insGE32Args[GE32_SYSTEM_ID] = initParamMap.get( "SYSTEM_ID" );
//		DBUtil.dbExecute( insGE32, insGE32Args, appInfo );
		DBUtil.dbExecute( insGE32, insGE32Args, appInfo, DBID );	// 5.5.5.1 (2012/08/07)

		// 宛先テーブル追加
		String[] insGE34Args = new String[10];
		insGE34Args[GE34_YKNO]= ykno;
		for( String dstId : mailDstMap.keySet() ) {
			insGE34Args[GE34_DST_ID] 	= trim( mailDstMap.get( dstId )[MailPattern.IDX_DST_ID]	 , 10 );
			insGE34Args[GE34_GROUP_ID] 	= trim( mailDstMap.get( dstId )[MailPattern.IDX_GROUP_ID], 20 );
			insGE34Args[GE34_DST_NAME] 	= trim( mailDstMap.get( dstId )[MailPattern.IDX_DST_NAME], 20 );
			insGE34Args[GE34_DST_ADDR] 	= trim( mailDstMap.get( dstId )[MailPattern.IDX_DST_ADDR], 100 );
			insGE34Args[GE34_DST_KBN] 	= mailDstMap.get( dstId )[MailPattern.IDX_DST_KBN];
			insGE34Args[GE34_FGJ] 		= mailDstMap.get( dstId )[MailPattern.IDX_FGJ];
			insGE34Args[GE34_DYSET] 	= HybsSystem.getDate( "yyyyMMddHHmmss" );
			insGE34Args[GE34_USRSET] 	= initParamMap.get( "LOGIN_USERID" );
			insGE34Args[GE34_PGUPD] 	= initParamMap.get( "PGID" );
//			DBUtil.dbExecute( insGE34, insGE34Args, appInfo );
			DBUtil.dbExecute( insGE34, insGE34Args, appInfo, DBID );		// 5.5.5.1 (2012/08/07)
		}
	}

	/**
	 * パラメータマップをセットします。
	 *
	 * @param	params	パラメータのマップ
	 */
	protected void setInitParams( final Map<String, String> params ) {
		initParamMap = params;
	}

	/**
	 * 添付ファイル配列をセットします。
	 *
	 * @param	attach1	添付ファイル名1
	 * @param	attach2	添付ファイル名2
	 * @param	attach3	添付ファイル名3
	 * @param	attach4	添付ファイル名4
	 * @param	attach5	添付ファイル名5
	 */
	protected void setAttachFiles( final String attach1
								, final String attach2
								, final String attach3
								, final String attach4
								, final String attach5 ) {
		List<String> fileList = new ArrayList<String>();
		if( attach1 != null && attach1.length() != 0 ) { fileList.add( attach1 ); }
		if( attach2 != null && attach2.length() != 0 ) { fileList.add( attach2 ); }
		if( attach3 != null && attach3.length() != 0 ) { fileList.add( attach3 ); }
		if( attach4 != null && attach4.length() != 0 ) { fileList.add( attach4 ); }
		if( attach5 != null && attach5.length() != 0 ) { fileList.add( attach5 ); }
		attachFiles = fileList.toArray( new String[fileList.size()] );
	}

	/**
	 * メール宛先マップをセットします。
	 *
	 * @param   mailDst	メール宛先マップ
	 */
	protected void setMailDstMap( final Map<String, String[]> mailDst ) {
		mailDstMap = mailDst;
	}

	/**
	 * メール宛先マップをセットします。
	 *
	 * @return	メール宛先マップ
	 */
	protected Map<String, String[]> getMailDstMap() {
		return mailDstMap;
	}

	/**
	 * 指定の長さ以内の文字列を返します。
	 *
	 * @og.rev 5.9.1.3 (2015/10/30) 文字数ではなくByte数に変更
	 *
	 * @param	src		オリジナルの文字列
	 * @param	maxLen	指定の長さ
	 *
	 * @return	指定の長さに短縮された文字列
	 */
	protected String trim( final String src, final int maxLen ) {
		String rtn = src;
		if( src != null && src.length() > maxLen ) {
//			rtn = src.substring( 0, maxLen );
			rtn = StringUtil.cut( src, maxLen );
		}
		return rtn;
	}

	/**
	 * アドレスチェックのエラーリストを返します。
	 *
	 * @return	エラーリスト
	 */
	protected List<String> getErrList(){
		return errAddrList;
	}

	/**
	 * 宛先マップを元に、送信オブジェクトに宛先をセットします。
	 * セットする際に、アカウントエラーとなっているアドレスを除外します。
	 * 宛先が存在しない場合、例外を投げます。
	 *
	 * @og.rev 4.3.7.5 (2009/07/08) 送信先名称が設定されていない場合は、アドレスを<>で囲わない
	 *
	 * @param invalidAddr 宛先のリスト
	 */
	private void setMailDst( final List<String> invalidAddr ){
		ArrayList<String> toList = new ArrayList<String>();
		ArrayList<String> ccList = new ArrayList<String>();
		ArrayList<String> bccList = new ArrayList<String>();

		Map<Integer, ArrayList<String>> tempMap = new HashMap<Integer, ArrayList<String>>();
		tempMap.put( Integer.valueOf( MailPattern.KBN_TO ),  toList );
		tempMap.put( Integer.valueOf( MailPattern.KBN_CC ),  ccList );
		tempMap.put( Integer.valueOf( MailPattern.KBN_BCC ), bccList );

		for( String dstId : mailDstMap.keySet() ) {
			String[] dstInfo = mailDstMap.get( dstId );
			Integer kbn = Integer.valueOf( dstInfo[MailPattern.IDX_DST_KBN] );
			if( !invalidAddr.contains( dstInfo[MailPattern.IDX_DST_ADDR] )
					&& !FGJ_ADDR_ERR.equals( dstInfo[MailPattern.IDX_FGJ] )){
				dstInfo[MailPattern.IDX_FGJ] = FGJ_SEND_OVER;

				// 4.3.7.5 (2009/07/08)
				String name = dstInfo[MailPattern.IDX_DST_NAME];
				if( name != null && name.length() > 0 ) {
					tempMap.get( kbn ).add( dstInfo[MailPattern.IDX_DST_NAME] +  "<"+ dstInfo[MailPattern.IDX_DST_ADDR] + ">" );
				}
				else {
					tempMap.get( kbn ).add( dstInfo[MailPattern.IDX_DST_ADDR] );
				}
			}
			else {
				if( FGJ_SEND_OVER.equals( dstInfo[MailPattern.IDX_FGJ] ) ) {
					dstInfo[MailPattern.IDX_FGJ] = FGJ_ACNT_ERR;
				}
			}
		}

		mail.clearTo();		// 宛先(TO)をクリア
		mail.clearCc();		// 宛先(CC)をクリア
		mail.clearBcc();	// 宛先(BCC)をクリア
		boolean haveValidAddr = false ;
//		if( toList.size() > 0 ) { 	// toのセット
		if( ! toList.isEmpty() ) { 	// toのセット
			haveValidAddr = true;
			String[] to = toList.toArray( new String[toList.size()] );
			mail.setTo( to );
		}
//		if( ccList.size() > 0 ) {	// ｃｃのセット
		if( ! ccList.isEmpty() ) {	// ｃｃのセット
			haveValidAddr = true;
			String[] cc = ccList.toArray( new String[ccList.size()] );
			mail.setCc( cc );
		}
//		if( bccList.size() > 0 ) {	// bccのセット
		if( ! bccList.isEmpty() ) {	// bccのセット
			haveValidAddr = true;
			String[] bcc = bccList.toArray( new String[bccList.size()] );
			mail.setBcc( bcc );
		}
		if( !haveValidAddr ){		// 宛先全部無効の場合、例外を投げます。
			String errMsg = "宛先のメールアドレスが有効ではありません。"
						+ "TO , CC , BCC のいづれにもアドレスが設定されていません。"; // 5.1.8.0 (2010/07/01) errMsg 修正
			throw new RuntimeException( errMsg );
		}
	}

	/**
	 * 要求NOを採番します。
	 * この要求NOで履歴テーブル(GE32)と宛先テーブル(GE30)の関連付けを持たせます。
	 *
	 * @og.rev 5.5.5.1 (2012/08/07) リソース系DBID 付け忘れ対策
	 *
	 * @return	要求NO
	 */
	private String getYkno() {
//		String[][] tmp = DBUtil.dbExecute( selYkno, new String[0], appInfo );
		String[][] tmp = DBUtil.dbExecute( selYkno, new String[0], appInfo, DBID );		// 5.5.5.1 (2012/08/07)
		if( tmp == null || tmp.length == 0 ) {
			String errMsg = "要求NO採番エラー"
						+ " SQL=" + selYkno ;		// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new RuntimeException( errMsg );
		}
		return tmp[0][0];
	}

	/**
	 * メールアドレスのリストよりユーザーIDを逆引きします。
	 *
	 * @param	addressList	メールアドレスのリスト
	 *
	 * @return	ユーザーID
	 */
	private String getUserIds( final List<String> addressList ){
		StringBuilder idBuf = new StringBuilder();
		Map<String,String>  addressMap = new HashMap<String, String>();
		for( String userId : mailDstMap.keySet() ) {
			String[] dstInfo = mailDstMap.get( userId );
			addressMap.put( dstInfo[MailPattern.IDX_DST_ADDR], userId );
		}
		for(int i=0; i < addressList.size(); i++){
			idBuf.append( ',' ).append( addressMap.get( addressList.get( i ) ) );
		}
		String rtn = "";
		if ( idBuf.length() > 0 ) {
			rtn = idBuf.toString().substring( 1 );
		}
		return rtn;
	}
}
