/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.transfer;

import java.io.File;
import java.io.PrintWriter;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.FileUtil;
// import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.system.DateSet;							// 6.4.2.0 (2016/01/29)

/**
 * 伝送要求に対してのデータをファイルに書込みします。
 * 但し、書き込まれるデータについては、旧伝送システムの形式と互換性を持たせるため、
 * データの前30Byteに空白で埋め、さらに全体で標準では500Byteになるように行末にも空白埋めをします。
 * 500byte以外にしたい場合は、書き込みパラメータの第２引数に整数で指定してください。
 * 
 * 先頭データに限っては、スペース以外で埋める事も可能です。
 * その場合は実行対象の第２引数から順番に
 * 状況コード(1byte)、データコード(1byte)、送り先(2byte)、テキスト種別(4byte) 送り元(2byte)
 * を指定してください。
 * 送り元を指定した場合はテキストの後に8byteスペース+送り元+26byteスペース+連番　を付加した上で
 * 行末の空白埋めを行います（後ろ70byteという指定ではないので注意）
 *
 * 書込みするファイル名は、実行対象で指定します。ファイル名は絶対パスで指定して下さい。
 * また、書込するテキストファイルのエンコードは書込パラメーターで指定することができます。
 * 指定しない場合、UTF-8が適用されます。
 *
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class TransferExec_SAMCB implements TransferExec {

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TransferExec_SAMCB() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	// 書込ファイルオブジェクト

	// 書込ファイルのエンコード

	/**
	 * ファイルに書込みします。
	 *
	 * @param vals 伝送データ(配列)
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 *
	 * @og.rev 5.5.3.3 (2012/06/15) close処理
	 * @og.rev 5.8.1.1 (2014/11/14) パラメータで桁数指定可能にする
	 * @og.rev 5.8.1.2 (2014/11/21) 前方,後方のスペース埋めの箇所の互換性を上げる
	 */
	@Override
	public void execute( final String[] vals, final TransferConfig config, final Transaction tran ) {
		String fileEncode = "UTF-8";

		// 5.8.1.2
		final String[] obj = StringUtil.csv2Array( config.getExecObj(), ' ' );
		final File fileWrite = new File( obj[0] );
		String preText = "";
		String suffText = "";
		preText +=  obj.length > 1 ? obj[1] : ""; // 状況コード
		preText +=  obj.length > 2 ? obj[2] : ""; // データコード
		preText +=  obj.length > 3 ? StringUtil.stringFill( obj[3], 8, fileEncode ) : ""; // ホストNO
		preText +=  obj.length > 4 ? StringUtil.stringFill( obj[4], 4, fileEncode ) : ""; // テキスト種別
		preText = StringUtil.stringFill( preText, 14, fileEncode );
		if( obj.length > 1 ){
			preText += DateSet.getDate( "yyMMddHHmmss" );
			preText += DateSet.getDate( "MMdd" );
		}
		preText  +=  obj.length > 4 ? StringUtil.stringFill( obj[4], 4, fileEncode ) : ""; // テキスト種別

		suffText +=  obj.length > 5 ? StringUtil.stringFill( "", 8, fileEncode ) + StringUtil.stringFill( obj[5], 28, fileEncode ) : ""; // 送り元

		// 5.8.1.1 (2014/11/14) fillByte追加
		int fillByte = 500;
		final String execPrm = config.getExecPrm();
		if( execPrm != null && execPrm.length() > 0 ){
			final String[] prm = StringUtil.csv2Array( execPrm, ' ' );
			fileEncode =  prm[0];
			if( prm.length > 1 ) {
				fillByte = Integer.parseInt( prm[1] );
			}
		}

		final PrintWriter writer = FileUtil.getPrintWriter( fileWrite,fileEncode );
		String line = null;
		// 5.8.1.2 (2014/11/21) iが欲しいので書き方を変更する
		for( int i=0; i<vals.length; i++ ){
			// 前30Byteを空白埋め 
			// 5.8.1.2 (2014/11/21) 前方をスペース以外で埋められるようにする
			final String preSpace = StringUtil.stringFill( preText, 30, fileEncode );
			// 全体で500Byteになるように後ろに空白埋め
			// 5.8.1.1 500Byte以外も指定可能とする
			// 5.8.1.2 (2014/11/21)
			if( obj.length > 5 ) {
				line = StringUtil.stringFill( preSpace + vals[i] + suffText + StringUtil.intFill( Integer.toString( i ), 14 ), fillByte, fileEncode );
			}
			else{
				line = StringUtil.stringFill( preSpace + vals[i] + suffText, fillByte, fileEncode );
			}

			writer.println( line );
		}
		writer.close(); // 5.5.3.3 (2012/06/15)
	}
}
