/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report2;

// import java.util.Map;
// import java.util.HashMap;
import java.util.concurrent.ConcurrentMap;							// 6.4.3.3 (2016/03/04)
import java.util.concurrent.ConcurrentHashMap;						// 6.4.3.3 (2016/03/04)

/**
 * 帳票処理を行う各スレッドを管理するクラスです。
 *
 * 各スレッドは、内部的にプールされます。
 * スレッドのIDはOOoQueue#getThreadId()で返される値です。
 * スレッドが生成されるタイミングは、そのIDで初めてスタック要求が来た(insertQueue()が呼ばれた)時です。
 *
 * 指定のスレッドを終了するには、funishThread( key )を呼び出します。
 * 全てのスレッドを終了するには、funishAllThreads()を呼び出します。
 *
 * 現時点での実装では、生成されたスレッドに対しての監視は行っていません。
 * これは、特定のスレッドがフリーズした際、外部から強制終了を行おうとすると、
 * 監視スレッドもフリーズしてしまう問題があるためです。
 * (但し、1つのsoffice.binのプロセスに対してシリアルに対して処理している限りでは、
 *  フリーズ問題は発生しないようです)
 *
 * @og.group 帳票システム
 *
 * @version  4.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public final class ExecThreadManager {

	/** 6.4.3.1 (2016/02/12) 一連の処理で同期を取りたいので、 ConcurrentHashMap は使いません。  */
//	private static final Map<String, ExecThread> EXEC_POOL = new HashMap<>();				// 6.4.1.1 (2016/01/16) pool → EXEC_POOL refactoring
	private static final ConcurrentMap<String, ExecThread> EXEC_POOL = new ConcurrentHashMap<>();		// 6.4.3.3 (2016/03/04)
	private static boolean debug	;		// 4.3.0.0 (2008/07/15) デバッグ追加

	/**
	 * オブジェクトの生成を禁止します
	 */
	private ExecThreadManager() {}

	/**
	 * キューを該当するスレッドにスタックする。
	 *
	 * @og.rev 4.3.0.0 (2008/07/15) スレッドIDにシステムIDを付加
	 *
	 * @param	queue	ExecQueueオブジェクト
	 */
	public static void insertQueue( final ExecQueue queue ) {
		// 4.3.3.6 (2008/11/15) この部分は不要なので元に戻します
		final ExecThread oet = getExecThread( queue.getThreadId() );
		oet.stackQueue( queue );
	}

	/**
	 * キューを該当するスレッドにスタックする
	 *
	 * このメソッドでは、既に同じスレッドが存在するかどうかをチェックせずに必ず
	 * 新しいスレッドを生成し、キューを処理します。
	 * また、処理が完了した後、そのスレッドは、WAITすることなく終了します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param	queue	ExecQueueオブジェクト
	 */
	public static void insertQueueOnNewThread( final ExecQueue queue ) {
		final ExecThread oet = new ExecThread( queue.getThreadId(), debug );
		oet.start();
//		System.out.println( "[INFO]THREAD CREATED:THREAD-ID=" + queue.getThreadId() );
		oet.stackQueue( queue );
		oet.finishAfterExec();
	}

	/**
	 * 該当するスレッドIDを持つスレッドを取得します。
	 * スレッドプールに存在しない場合は、新規に作成されます。
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) Map#compute で対応する。
	 *
	 * @param	threadId	スレッドID
	 *
	 * @return	ExecThreadスレッド
	 */
	private static ExecThread getExecThread( final String threadId ) {
		// Map#compute ： 戻り値は、新しい値。追加有り、置換有り、削除有り
		return EXEC_POOL.compute( threadId, (id,oet) ->
					oet == null || !oet.isAlive() ? ExecThread.startExecThread( id, debug ) : oet );

//		ExecThread oet = null;
//		synchronized( EXEC_POOL ) {
//			oet = EXEC_POOL.get( threadId );
//
//			// スレッドが終了している場合は、再度作成
//			if( oet != null && !oet.isAlive() ) {
//				EXEC_POOL.remove( threadId );
//				oet = null;
//			}
//
//			if( oet == null ) {
//				// oet = new ExecThread( threadId );
//				oet = new ExecThread( threadId, debug );
//				oet.start();
//				System.out.println( "[INFO]THREAD CREATED:THREAD-ID=" + threadId );
//				EXEC_POOL.put( threadId, oet );
//			}
//		}
//
//		return oet;
	}

	/**
	 * 全てのスレッドを終了します。
	 *
	 * ※ REP21/result2.jsp で、使用。
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) Map#forEach で対応する。
	 */
	public static void finishAllThreads() {
		EXEC_POOL.forEach( (id,oet) -> oet.finish() );
		System.out.println( "[INFO]ALL THREADS FINISHED" );

//		synchronized( EXEC_POOL ) {
//			for( final ExecThread oet : EXEC_POOL.values() ) {
//				oet.finish();
//			}
//			EXEC_POOL.clear();
//			System.out.println( "[INFO]ALL THREADS FINISHED" );
//		}
	}

	/**
	 * 指定のスレッドを終了します。
	 *
	 * ※ REP21/entry.jsp で、使用。
	 *
	 * @param threadId スレッドID
	 */
	public static void finishThread( final String threadId ) {
//		synchronized( EXEC_POOL ) {
			final ExecThread oet = EXEC_POOL.remove( threadId );
			// Map上には null はないが、キーが合わない場合は、null が戻る。
			if( oet != null ) {
				oet.finish();
				System.out.println( "[INFO]THREAD CREATED:THREAD-ID=" + threadId );
			}
//		}
	}

	/**
	 * スレッド情報のマップを返します。
	 *
	 * ※ REP21/result2.jsp で、使用。
	 *
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 * @og.rev 6.4.3.3 (2016/03/04) 戻すMapが、not null制限つきであることを示すため、ConcurrentMap に置き換えます。
	 * @og.rev 6.4.3.3 (2016/03/04) Map#forEach で対応する。
	 *
	 * @return	スレッド情報のマップ
	 */
//	public static Map<String, String> getThreadInfo() {
	public static ConcurrentMap<String, String> getThreadInfo() {
//		final Map<String, String> info = new HashMap<>();
		final ConcurrentMap<String, String> infoMap = new ConcurrentHashMap<>();

		EXEC_POOL.forEach( (id,oet) -> infoMap.put( id,oet.toString() ) );

//		synchronized( EXEC_POOL ) {
//			for( final Map.Entry<String, ExecThread> entry : EXEC_POOL.entrySet() ) {
//				// EXEC_POOL は、key,val not null制限です。
//				info.put( entry.getKey(), entry.getValue().toString() );
//			}
//		}
		return infoMap;
	}

	/**
	 * デバッグフラグの設定。
	 *
	 * @og.rev  4.3.0.0 (2008/07/15) デバッグ追加
	 *
	 * @param   flag デバッグフラグ [true:デバッグ/false:通常]
	 */
	public static void setDebug ( final boolean flag ){
		debug = flag;
	}
}
