/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)

import static org.opengion.fukurou.util.StringUtil.nval ;
import static org.opengion.fukurou.system.HybsConst.BR;			// 6.1.0.0 (2014/12/26) refactoring

import java.util.prefs.Preferences;
import java.util.prefs.BackingStoreException;
import java.util.Locale ;

/**
 * 指定の Windowsレジストリにアクセスするためのタグです(特殊な環境設定が必要です)。
 *
 * 検索した結果は、DBTableModel にセットされるため、
 * JDBCQuery と同様に、viewタグで表示させることが可能です。
 *
 * 注意１：
 *   通常であれば、以下の２つのルートパス以下のレジストリにアクセスできます。
 *     HKEY_CURRENT_USER/Software/JavaSoft/Prefs
 *     HKEY_LOCAL_MACHINE/Software/JavaSoft/Prefs
 *
 * 注意２：
 *   ルートパスを強制的に変更する為、java.util.prefs.WindowsPreferenceクラスを
 *   直接書き換えた、ogPreferences.jar を用意しています。
 *   これを、tomcat/endorsed フォルダにコピーして使います。
 *   その場合は、
 *     HKEY_CURRENT_USER/Software/Muratec
 *     HKEY_LOCAL_MACHINE/Software/Muratec
 *   以下の２つのルートパス以下のレジストリにアクセスできます。
 *
 * @og.formSample
 * ●形式：&lt;og:regQuery baseKey="･･･" ･･･ /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:regQuery
 *       baseKey          ○【TAG】検索ベースキーを設定します(HKEY_CURRENT_USER/Software/XXXX の XXXX を指定します)(必須)。
 *       hkeyType           【TAG】HKEY_CURRENT_USER(="user") を読むか、HKEY_LOCAL_MACHINE(="system") を読むかを指定します(初期値:user)。
 *       key                【TAG】検索キーを設定します
 *       value              【TAG】検索バリューを設定します
 *       maxRowCount        【TAG】レジストリの最大検索件数をセットします(初期値:0[無制限])
 *       orderBy            【TAG】検索した結果を表示する表示順をファイル属性名で指定します
 *       maxLevel           【TAG】検索時の最大展開レベル(0は無制限)を指定します(初期値:1)
 *       like               【TAG】キーおよびバリューについて,like 検索を行うかどうか[true/false]を指定します(初期値:false)
 *       tableId            【TAG】(通常は使いません)結果をDBTableModelに書き込んで、sessionに登録するときのキーを指定します
 *       command            【TAG】コマンド (NEW,RENEW)をセットします(初期値:NEW)
 *       displayMsg         【TAG】検索結果を画面上に表示するメッセージIDを指定します(初期値:VIEW_DISPLAY_MSG[={@og.value SystemData#VIEW_DISPLAY_MSG}])
 *       notfoundMsg        【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])
 *       overflowMsg        【TAG】オーバーフロー時に画面上に表示するメッセージIDを指定します(初期値:MSG0007)
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/application]を指定します(初期値:session)
 *       mainTrans          【TAG】(通常は使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します(初期値:false)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    &lt;og:regQuery  debug=&quot;false&quot;
 *        command     = &quot;NEW&quot;                       NEW,RENEW が使用できます。
 *        baseKey     = &quot;/&quot;                         検索時のベースとなるレジストリキー名
 *        key         = &quot;driver&quot;                    検索したいレジストリキー名(初期値は全件)
 *        value       = &quot;Ne&quot;                        検索したいレジストリ値(初期値は全件)
 *        maxRowCount = &quot;0&quot;                         最大検索件数(0で無制限)
 *        maxLevel    = &quot;0&quot;                         最大検索階層レベル(0で無制限)
 *        like        = &quot;true&quot;                      true で曖昧検索/false は一致検索
 *    /&gt;
 *
 * like は、key / value を設定したときのみ有効です。また、key / value を両方同時に設定した場合は、
 * like 属性は両方に同時に適用されます。
 *
 * @og.rev 3.1.0.0 (2003/03/20) Windowsレジストリにアクセスできる、RegistryQueryTag.java を新規に作成。
 * @og.group その他入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class RegistryQueryTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.3.4 (2016/03/11)" ;
	private static final long serialVersionUID = 643420160311L ;

	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	public static final String CMD_NEW	 = "NEW" ;
	/** command 引数に渡す事の出来る コマンド  再検索 {@value} */
	public static final String CMD_RENEW = "RENEW" ;
//	/** command 引数に渡す事の出来る コマンド リスト  */
//	private static final String[] COMMAND_LIST = { CMD_NEW , CMD_RENEW };

//	private static final String[] COLUMN_KEY   = { "LEBEL","KEY","VALUE","CHILD","PATH" };
	// 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	private static final ArraySet<String> COMMAND_SET = new ArraySet<>( "LEBEL","KEY","VALUE","CHILD","PATH" );

	// 5.6.8.2 (2013/09/20) エラーメッセージ等に使う文字列
	private static final String[] HKEY_TYPE = { "HKEY_CURRENT_USER","HKEY_LOCAL_MACHINE" } ;

	private String	tableId 		= HybsSystem.TBL_MDL_KEY;		// sessionｷｰ
	private String	command			= CMD_NEW;						// ｺﾏﾝﾄﾞ
	private String	baseKey 		;								// 検索ベースｷｰ
	private String	key				;								// 検索キー
	private String	value			;								// 検索バリュー
	private int 	maxRowCount		;								// 最大検索数(0は無制限)
	private String	orderBy			;								// ソート項目
	private String	displayMsg		= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
	private String	overflowMsg		= "MSG0007";					// 検索結果が、制限行数を超えましたので、残りはカットされました。
	private String	notfoundMsg		= "MSG0077";					// 対象データはありませんでした。
	private int		maxLevel		= 1;							// 下位層展開最大レベル(0は無制限)
	private boolean	like			;								// あいまい検索ﾌﾗｸﾞ

	private int		executeCount	;								// 検索/実行件数
	private boolean	isMainTrans		= true;							// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
	private int		hkeyType		;								// 5.6.8.2 (2013/09/20) 0:HKEY_CURRENT_USER(="user") , 1:HKEY_LOCAL_MACHINE(="system")

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public RegistryQueryTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.6.0.8 (2004/11/19) DBTableModel をセーブする時に、トランザクションチェックを行います。
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

//		if( check( command, COMMAND_LIST ) ) {
		if( check( command, COMMAND_SET ) ) {
			useMainTrans( isMainTrans );			// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
			startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)

			final DBTableModel table = makeDBTable();
			// 3.6.0.8 (2004/11/19) トランザクションチェックを行います。
			if( ! commitTableObject( tableId, table ) ) {
				jspPrint( "RegistryQueryTag Query処理が割り込まれました。DBTableModel は登録しません。" );
				return SKIP_PAGE;
			}

			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

			// 実行件数の表示 command="NEW" のときのみ、displayMsg を表示させます。
			// 4.0.0 (2005/11/30) 出力順の変更。一番最初に出力します。
			if( CMD_NEW.equals( command ) ) {
				if( executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
					buf.append( executeCount )
						.append( getResource().getLabel( displayMsg ) )
						.append( BR );
				}
				else if( executeCount == 0 && notfoundMsg != null && notfoundMsg.length() > 0 ) {
					buf.append( getResource().getLabel( notfoundMsg ) )
						.append( BR );
				}
			}

			if( maxRowCount > 0 && maxRowCount <= executeCount ) {
				buf.append( getResource().getLabel( overflowMsg ) )
					.append( BR );
			}

			jspPrint( buf.toString() );
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 * @og.rev 5.6.8.2 (2013/09/20) hkeyType 追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();

		tableId 		= HybsSystem.TBL_MDL_KEY;			// sessionｷｰ
		orderBy			= null;								// ソート項目
		command			= CMD_NEW;							// ｺﾏﾝﾄﾞ
		baseKey 		= null;								// 検索ベースｷｰ
		key				= null;								// 検索キー
		value			= null;								// 5.6.8.2 (2013/09/20) 検索バリュー 入れ忘れ
		executeCount	= 0;								// 検索/実行件数
		maxRowCount		= 0;								// 最大検索数(0は無制限)
		displayMsg		= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
		overflowMsg		= "MSG0007";						// 検索結果が、制限行数を超えましたので、残りはカットされました。
		notfoundMsg		= "MSG0077";						// 対象データはありませんでした。
		maxLevel		= 1;								// 下位層展開最大レベル(0は無制限)
		like			= false;							// あいまい検索ﾌﾗｸﾞ
		isMainTrans		= true;								// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
		hkeyType		= 0;								// 5.6.8.2 (2013/09/20) 0:HKEY_CURRENT_USER(="user") , 1:HKEY_LOCAL_MACHINE(="system")
	}

	/**
	 * レジストリ から、値を取り出し、DBTableModel を作成します。
	 *
	 * @og.rev 5.6.8.2 (2013/09/20) hkeyType 追加に伴う各種修正
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @return	テーブルモデル
	 */
	private DBTableModel makeDBTable() {
		final DBTableModel table = DBTableModelUtil.newDBTable();
		table.init( COMMAND_SET.size() );

		// オリジナルの forEach。カウンタ初期値とラムダ式を与えると、ラムダ式の引数に、カウンタと値が設定される。
		COMMAND_SET.forEach( 0 , (i,v) -> {
			final DBColumn dbColumn = getResource().makeDBColumn( v );
			table.setDBColumn( i,dbColumn );
		} );

//		table.init( COLUMN_KEY.length );
//		for( int i=0; i<COLUMN_KEY.length; i++ ) {
//			final DBColumn dbColumn = getResource().makeDBColumn( COLUMN_KEY[i] );
//			table.setDBColumn( i,dbColumn );
//		}

		// 5.6.8.2 (2013/09/20) 0:HKEY_CURRENT_USER(="user") , 1:HKEY_LOCAL_MACHINE(="system")
		final Preferences hayabusaRoot = ( hkeyType == 0 ) ? Preferences.userRoot() : Preferences.systemRoot();

		try {
			if( ! hayabusaRoot.nodeExists( baseKey ) ) {
				final String errMsg = "Preferences BaseKey が Windows Registry に存在しませんでした。"
								+ "[" + HKEY_TYPE[hkeyType] + "\\" + hayabusaRoot.absolutePath() + "]" ;
				throw new HybsSystemException( errMsg );
			}

			showChild( table,hayabusaRoot.node(baseKey),1 );
		}
		catch( final BackingStoreException ex ) {
			final String errMsg = "レジストリ から、値を取り出す事が出来ませんでした。"
								+ "[" + HKEY_TYPE[hkeyType] + "\\" + hayabusaRoot.absolutePath() + "]" ;
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		return table ;
	}

	/**
	 * Preferences の子要素を検索します。
	 *
	 * この処理は、再帰定義により、階層を順次下がっていきます。
	 * レベル制限、検索最大数制限のチェックを行っています。
	 * addTableValue メソッドを呼び出して、検索結果を、DBTableModel に順次追加していきます。
	 *
	 * @og.rev 5.6.8.2 (2013/09/20) BackingStoreException を throwしない様に修正
	 *
	 * @param	table	DBTableModelオブジェクト
	 * @param	prefs	Preferencesオブジェクト
	 * @param	lvl		レベル制限
	 */
	private void showChild( final DBTableModel table,
							final Preferences prefs,
							final int lvl ) {

		if( maxRowCount > 0 && maxRowCount <= executeCount ) { return ; }
		if( maxLevel > 0 && lvl > maxLevel ) { return; }

		String name = null;
		try {
			final String [] child = prefs.childrenNames();

			if( child != null && child.length > 0 ) {
		 		for( int i=0; i<child.length; i++ ) {
					name = child[i];

					// name に全角文字が使われている場合、正常に処理できないので、Tableへの書き込みだけ行います。
					if( isZenkakuName( name ) ) {
						addTableValue( name,table,prefs,lvl,null,false );
					}
					else {
						final Preferences childPrefs = prefs.node( name.toLowerCase(Locale.JAPAN) );

						addTableValue( name,table,childPrefs,lvl,null,true );
						showChild( table,childPrefs,lvl+1 );
					}
				}
			}
			else {
				showPrefs( table,prefs,lvl );
			}
		}
		// エラーが発生しても、処理は継続します。
		catch( final BackingStoreException ex ) {
			final String errMsg = "BackingStoreエラーが発生しました。[" + (executeCount+1) + "]"
						+ " key=[" + name + "] "
						+ ex.getMessage() ;
			System.out.println( errMsg );
			addTableValue( name,table,prefs,lvl,errMsg,false );
		}
		// エラーが発生しても、処理は継続します。
		catch( final RuntimeException ex ) {
			final String errMsg = "RuntimeExceptionエラーが発生しました。[" + (executeCount+1) + "]"
						+ " key=[" + name + "] "
						+ ex.getMessage() ;
			System.out.println( errMsg );
			addTableValue( name,table,prefs,lvl,errMsg,false );
		}
 	}

	/**
	 * レジストリのキー情報が、全角文字を含むかかどうか判定します。(含むなら、true)。
	 *
	 * name に全角文字が使われている場合、正常に処理できないので、判定します。
	 *
	 * @param	name	レジストリのキー情報
	 * @return	全角文字が使われている場合、true を返します。
	 */
	private boolean isZenkakuName( final String name ) {
		boolean isZenkaku = false;
		for( int i=0; i<name.length(); i++ ) {
			final char ch =name.charAt(i);
			if( ch < 0x0020 || ch > 0x007f ) {
				isZenkaku = true;
				break;
			}
		}
		return isZenkaku;
	}

	/**
	 * Preferences の属性を検索します。
	 *
	 * addTableValue メソッドを呼び出して、検索結果を、DBTableModel に順次追加していきます。
	 *
	 * @param	table	DBTableModelオブジェクト
	 * @param	prefs	Preferencesオブジェクト
	 * @param	lvl		レベル制限
	 */
	private void showPrefs( final DBTableModel table,
							final Preferences prefs,
							final int lvl ) throws BackingStoreException {

		final String [] keys = prefs.keys();

		if( keys != null ) {
	 		for( int i=0; i<keys.length; i++ ) {
				final String name = keys[i];
				addTableValue( name,table,prefs,lvl,null,false );
			}
		}
 	}

	/**
	 * 検索された Preferencesを、DBTableModel に順次追加していきます。
	 *
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @param	name	検索キー
	 * @param	table	DBTableModelオブジェクト
	 * @param	prefs	Preferencesオブジェクト
	 * @param	lvl		レベル制限
	 * @param	msg		値にメッセージを書き込みたい場合 (正常時ならnullでかまわない)
	 * @param	flag	値の取得方法 [true:空文字列/false:Preferencesから、検索キーを使用して取得]
	 */
	private void addTableValue( final String name,
								final DBTableModel table,
								final Preferences prefs,
								final int lvl,
								final String msg,
								final boolean flag ) {

		if( maxRowCount > 0 && maxRowCount <= executeCount ) { return ; }

		final String lowerName = name.toLowerCase(Locale.JAPAN);
//		final String val	   = (msg != null) ? msg : ( flag ? "" : prefs.get(lowerName, "") );

		if( key != null ) {
			if( like ) {
				if( lowerName.indexOf( key ) < 0 ) { return; }
			}
			else {
				if( ! lowerName.equalsIgnoreCase( key ) ) { return; }
			}
		}

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		final String val = msg == null ? ( flag ? "" : prefs.get(lowerName, "") ) : msg ;
//		final String val = msg != null ? msg : ( flag ? "" : prefs.get(lowerName, "") );
		if( value != null ) {
			if( like ) {
				if( (val.toLowerCase(Locale.JAPAN)).indexOf( value ) < 0 ) { return; }
			}
			else {
				if( ! val.equalsIgnoreCase( value ) ) { return; }
			}
		}

//		String[] clmVals = new String[COLUMN_KEY.length];
		String[] clmVals = new String[COMMAND_SET.size()];
		clmVals[0] = String.valueOf( lvl );				// LEVEL
		clmVals[1] = name;								// KEY
		clmVals[2] = val;								// VALUE
		clmVals[3] = String.valueOf( flag );			// CHILD
		clmVals[4] = prefs.absolutePath() ;				// PATH

		table.addColumnValues( clmVals );
		executeCount++ ;
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id テーブルID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ), tableId );
	}

	/**
	 * 【TAG】検索した結果を表示する表示順をファイル属性名で指定します。
	 *
	 * @og.tag
	 * 現仕様では、複数のキーを指定することは出来ません。
	 *
	 * @param	ordr	ソートキー
	 */
	public void setOrderBy( final String ordr ) {
		orderBy = nval( getRequestParameter( ordr ),orderBy );
	}

	/**
	 * 【TAG】コマンド (NEW,RENEW)をセットします(初期値:NEW)。
	 *
	 * @og.tag
	 * コマンドは,HTMLから(get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 * 何も設定されない、または、null の場合は、"NEW" が初期値にセットされます。
	 *
	 * @param	cmd コマンド (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.RegistryQueryTag.CMD_NEW">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		final String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】検索ベースキーを設定します(HKEY_CURRENT_USER/Software/XXX の XXX をベースとします)。
	 *
	 * @og.tag
	 * レジストリの検索で、HKEY_CURRENT_USER か、HKEY_LOCAL_MACHINE の区別を、hkeyType 属性で付ける事が
	 * できるようにしました。昔は、HKEY_CURRENT_USER/Software/ 以下の検索ができていましたが、
	 * 今現在は、できなくなっているようです。（または、使い方が間違っているか）
	 * そこで、標準(/JavaSoft/Prefs)、/Muratec、/Microsoft だけ、検索できるように、
	 * XXXXPreferencesFactory を作成しました。
	 * 詳細は、opengionV6/src/jdk170uXX_WindowsPreference を参照願います。
	 * これにより、従来通りの方法で（制限はかかりますが）レジストリを検索できます。
	 * 
	 * ベースキー に指定できるのは、"/Muratec" , "/Microsoft" で始まるキーだけです。
	 * それに続く "/" で区切った階層構造も表現可能です。
	 * 標準(/JavaSoft/Prefs) は、何も指定しないことで、標準であることを示します。
	 *
	 * @param	bkey 検索ベースキー
	 */
	public void setBaseKey( final String bkey ) {
		baseKey = nval( getRequestParameter( bkey ),baseKey );
	}

	/**
	 * 【TAG】検索キーを設定します。
	 *
	 * @og.tag 検索キーを設定します。
	 *
	 * @param	ky 検索キー
	 */
	public void setKey( final String ky ) {
		key = nval( getRequestParameter( ky ),key );
		if( key != null ) { key = key.toLowerCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】検索バリューを設定します。
	 *
	 * @og.tag 検索バリューを設定します。
	 *
	 * @param	val 検索バリュー
	 */
	public void setValue( final String val ) {
		value = nval( getRequestParameter( val ),value );
		if( value != null ) { value = value.toLowerCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】レジストリの最大検索件数をセットします(初期値:0[無制限])。
	 *
	 * @og.tag
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 * 初期値は、0 は、無制限です。
	 *
	 * @param	count 最大件数
	 */
	public void setMaxRowCount( final String count ) {
		maxRowCount = nval( getRequestParameter( count ),maxRowCount );
	}

	/**
	 * 【TAG】検索結果を画面上に表示するメッセージIDを指定します
	 *		(初期値:VIEW_DISPLAY_MSG[={@og.value SystemData#VIEW_DISPLAY_MSG}])。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したメッセージをリソースから取得して
	 * 表示します。
	 * 件数を表示させる場合は、displayMsg = "MSG0033"[　件検索しました] をセットしてください。
	 * 表示させたくない場合は, displayMsg = "" をセットしてください。
	 * (初期値:システム定数のVIEW_DISPLAY_MSG[={@og.value SystemData#VIEW_DISPLAY_MSG}])。
	 *
	 * @param	id 処理結果表示メッセージID
	 */
	public void setDisplayMsg( final String id ) {
		displayMsg = getRequestParameter( id );
	}

	/**
	 * 【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])。
	 *
	 * @og.tag
	 * ここでは、検索結果がゼロ件の場合のみ、特別なメッセージを表示させます。
	 * 従来は、displayMsg と兼用で、『0　件検索しました』という表示でしたが、
	 * displayMsg の初期表示は、OFF になりましたので、ゼロ件の場合のみ別に表示させます。
	 * 表示させたくない場合は, notfoundMsg = "" をセットしてください。
	 * 初期値は、MSG0077[対象データはありませんでした]です。
	 *
	 * @param	id ゼロ件時表示メッセージID
	 */
	public void setNotfoundMsg( final String id ) {
		final String ids = getRequestParameter( id );
		if( ids != null ) { notfoundMsg = ids; }
	}

	/**
	 * 【TAG】検索データが最大検索数をオーバーした場合に表示するメッセージリソースIDを指定します
	 *		(初期値:MSG0007[検索結果が、制限行数を超えましたので、残りはカットされました])。
	 *
	 * @og.tag
	 * 表示させたくない場合は, overflowMsg = "" をセットしてください。
	 *
	 * @param	id 最大検索数オーバー時メッセージID
	 */
	public void setOverflowMsg( final String id ) {
		overflowMsg = getRequestParameter( id );
	}

	/**
	 * 【TAG】検索時の最大展開レベル(0は無制限)を指定します(初期値:1)。
	 *
	 * @og.tag
	 * ０を指定すると、無制限に階層を展開します。
	 * 初期値は、１レベルです。
	 *
	 * @param	lvl 検索時の最大展開レベル
	 */
	public void setMaxLevel( final String lvl ) {
		maxLevel = nval( getRequestParameter( lvl ),maxLevel );
	}

	/**
	 * 【TAG】キーおよびバリューについて,like 検索を行うかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * like検索とは、キーの一部の文字を含む場合にマッチしたとして、値を取り出します。
	 * ここでの設定は、キーもバリューも同時に適用されます。また、大文字小文字の区別も行いません。
	 *
	 * @param	lik 曖昧検索を行うかどうか [true:行う/false:行わない]
	 */
	public void setLike( final String lik ) {
		like = nval( getRequestParameter( lik ),like );
	}

	/**
	 * 【TAG】レジストリの読み込むルートを[user/system]で指定します(初期値:user)。
	 *
	 * @og.tag
	 * HKEY_CURRENT_USER/Software/XXXX を読む場合は、"user" を、HKEY_LOCAL_MACHINE/Software/XXXX 
	 * を読む場合は、"system" を指定します。それ以外の指定は、エラーにしています。
	 * ここでの設定は、大文字小文字の区別は行いません。
	 * 初期値は、"user"(HKEY_CURRENT_USER) です。
	 *
	 * @og.rev 5.6.8.2 (2013/09/20) 新規追加
	 *
	 * @param	type 読み込むルート [user/system]
	 */
	public void setHkeyType( final String type ) {
		final String temp = nval( getRequestParameter( type ),null );
		if( temp != null ) {
			if(        "user".equalsIgnoreCase( temp ) ) { hkeyType = 0; }
			else if( "system".equalsIgnoreCase( temp ) ) { hkeyType = 1; }
			else {
				final String errMsg = "hkeyType は、[user] か、[system] のどちらかを指定してください。"
							+ " hkeyType[" + temp + "]" ;
				throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "regQuery" ;
	}

	/**
	 * 【TAG】(通常は使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * この値は、ファイルダウンロード処理に影響します。この値がtrueに指定された時にcommitされたDBTableModelが
	 * ファイルダウンロードの対象の表になります。
	 *
	 * このパラメーターは、通常、各タグにより実装され、ユーザーが指定する必要はありません。
	 * 但し、1つのJSP内でDBTableModelが複数生成される場合に、前に処理したDBTableModelについてファイルダウンロードをさせたい
	 * 場合は、後ろでDBTableModelを生成するタグで、明示的にこの値をfalseに指定することで、ファイルダウンロード処理の対象から
	 * 除外することができます。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param  flag メイントランザクションかどうか [true:メイン/false:その他]
	 */
	public void setMainTrans( final String flag ) {
		isMainTrans = nval( getRequestParameter( flag ),isMainTrans );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION		)
				.println( "tableId"		,tableId		)
				.println( "command"		,command		)
				.println( "baseKey"		,baseKey		)
				.println( "key"			,key			)
				.println( "value"		,value			)
				.println( "maxRowCount"	,maxRowCount	)
				.println( "orderBy"		,orderBy		)
				.println( "displayMsg"  ,displayMsg		)
				.println( "overflowMsg" ,overflowMsg	)
				.println( "maxLevel"	,maxLevel		)
				.println( "like"		,like			)
				.println( "executeCount",executeCount	)
//				.println( "COLUMN_KEY"	,COLUMN_KEY		)
				.println( "COMMAND_SET"	,COMMAND_SET	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
