/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;
import java.io.File;										// 6.2.0.0 (2015/02/27)

import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBColumn;

import org.opengion.fukurou.model.ExcelModel;

/**
 * ネイティブEXCELファイルの書き出しクラスです。
 *
 * DefaultTableWriter を継承していますので，ラベル，名前，データの出力部のみ
 * オーバーライドして，MIcrosoft Excelファイルの出力機能を実現しています。
 *
 * 出力形式は、openXML形式にも対応しています。
 * 出力ファイルの拡張子が、.xlsならExcel2003のバイナリ形式、.xlsxならExcel2007の
 * openXML形式で出力されます。
 *
 * useCellStyle      = [false/true];  データを書き込んだ範囲に罫線を入れる(true)かどうか(初期値:false)
 * useAutoCellSize   = [false/true];  セルの幅をデータの幅に自動的に合わせる(true)かどうか(初期値:false)
 * useActiveWorkbook = [false/true];  セルの有効範囲を設定する(true)かどうか(初期値:false)
 * pageBreakColumn   = PBCLM;         シートブレイクするキーとなるカラム名を指定します(このカラムの値がシート名になります)
 * fileBreakColumn   = FBCLM;         ファイルブレイクするキーとなるカラム名を指定します(このカラムの値がファイル名になります)
 *
 * appendモードの場合は、オリジナルが雛形ファイルになります。雛形ファイルを指定した場合は、
 * 同じファイル名で無ければエラーになります。
 *
 * @og.group ファイル出力
 *
 * @og.rev 4.3.4.3 (2008/12/22) 一部protected化
 * @og.rev 4.3.6.7 (2009/05/22) ooxml形式対応
 * @og.rev 6.0.1.2 (2014/08/08) ロジックの完全書き換え(保証もできない位･･･)
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableWriter_Excel extends TableWriter_Default {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.5.0.0 (2016/09/30)" ;

	private	int		nRowIndex		;				// 6.0.1.2 (2014/08/08) private化
	private String  filename		;				// 3.5.4.3 (2004/01/05)
	private String  directory		;				// 6.0.2.0 (2014/09/19) ディレクトリとファイルを分けて管理
	private String  fileKey			;				// 6.0.2.0 (2014/09/19) ディレクトリとファイルを分けて管理
	private String  sheetName		= "Sheet";		// 3.5.4.3 (2004/01/05) 書き込むSheet名
	private String	fontName		;				// 3.8.5.3 (2006/08/07)
	private short	fontPoint		= -1;			// 3.8.5.3 (2006/08/07)
	private String  refFilename		;				// 3.5.4.3 (2004/01/05)
	private String  refSheetName	;				// 3.5.4.3 (2004/01/05)
	// 5.1.4.0 (2010/03/01) 行番号情報を、出力する(true)/しない(false)を指定
	private boolean		useNumber	= true;
	private int		skipRowCount	;				// 6.0.1.2 (2014/08/08) データの書き込み開始位置設定

	private ExcelModel excel		;				// 6.0.2.0 (2014/09/19) 新規作成

	// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
	private boolean	useCellStyle		;			// データを書き込んだ範囲に罫線を入れるかどうかを指定します
	private boolean	useAutoCellSize		;			// セルの幅をデータの幅に自動的に合わせるかどうかを指定します
	private boolean	useActiveWorkbook	;			// セルの有効範囲を設定するかどうかを指定します
	private String	fileBreakColumn		;			// ファイルブレイクするキーとなるカラム名を指定します(このカラムの値がファイル名になります)
	private String	pageBreakColumn		;			// シートブレイクするキーとなるカラム名を指定します(このカラムの値がシート名になります)
	private boolean noPageBreak			;			// 6.1.0.0 (2014/12/26) pageBreak,fileBreak 修正
	private String	hLinkKeyColumn		;			// Hyperlinkを作成するキーとなるカラム名を指定
	private String	hLinkValColumn		;			// Hyperlinkを作成する値となるカラム名を指定

	private String	addTitleSheet		;			// Sheet一覧を先頭Sheetに作成する場合のSheet名を指定します

	private int		pageBreakClmNo		= -1;		// シートブレイクするキーとなるカラムNo
	private int		fileBreakClmNo		= -1;		// ファイルブレイクするキーとなるカラムNo
	private int		hLinkKeyClmNo		= -1;		// Hyperlinkを作成するキーとなるカラムNo
	private int		hLinkValClmNo		= -1;		// Hyperlinkを作成する値となるカラムNo
	private int		tblRow				;			// シートブレイクしても、データは続きから出力します。

	private boolean[] nvar	 			;			// 6.0.2.0 (2014/09/19) writeData( DBTableModel ,PrintWriter ) で使うカラム情報
	private boolean[] isNums 			;			// 6.0.2.0 (2014/09/19) writeData( DBTableModel ,PrintWriter ) で使うカラム情報

	// 5.9.12.1 (2016/09/09) シート上書き設定
	private boolean		sheetOverwrite		;		// 5.9.12.1 (2016/09/09)
	private String[]	recalcSheetNames	;		// 5.9.12.1 (2016/09/09)

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TableWriter_Excel() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * DBTableModel から 各形式のデータを作成して,PrintWriter に書き出します。
	 * このメソッドは、EXCEL 書き出し時に使用します。
	 *
	 * 雛形シートを使う場合は、雛形シートの後ろに追加していきます。
	 * 最後には、雛形シートとして存在しな部分は、すべて削除します。
	 * 雛形シートを使わない場合は、新規シートに追加していくことになります。
	 * appendモードの場合は、オリジナルが雛形ファイルになります。
	 * 雛形ファイルの指定は、無効になります。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @og.rev 5.1.4.0 (2010/03/01) columns 対応 、useNumber属性対応
	 * @og.rev 6.0.1.2 (2014/08/08) ロジックの完全書き換え(保証もできない位･･･)
	 * @og.rev 6.0.2.0 (2014/09/19) EXCEL新機能追加
	 * @og.rev 6.1.0.0 (2014/12/26) Excel関係改善
	 * @og.rev 6.2.0.0 (2015/02/27) ファイル引数を、String → File に変更
	 * @og.rev 6.2.2.3 (2015/04/10) 雛形シートにそのままデータを書き込んでいく。シートは削除不要
	 * @og.rev 5.9.12.1 (2016/09/09) sheetOverwrite,recalcSheetNames
	 * @og.rev 6.5.0.0 (2016/09/30) セルの計算式の再計算をさせる recalcSheetNames 属性の追加。(5.9.12.1 (2016/09/09))
	 *
	 * @see #isExcel()
	 */
	@Override
	public void writeDBTable() {
		if( ! createDBColumn() ) { return ; }		// 親クラスを改造。２回実行されない。

		useNumber = isUseNumber();

		// 3.5.6.0 (2004/06/18) 移動
		if( filename == null ) {
			final String errMsg = "ファイルが指定されていません。";
			throw new HybsSystemException( errMsg );
		}

		// appendモードの場合は、オリジナルが雛形ファイルになります。
		// 雛形ファイルの指定は、無効になります。
		if( isAppend() ) {
			// 互換性のため、雛形ファイルが出力ファイルと同じでない場合はエラーとします。
			if( refFilename != null && !filename.equalsIgnoreCase( refFilename ) ) {
				final String errMsg = "追加(appendモード)の時、雛形ファイルの指定は無効です。" + CR
								+ " filename=[" + filename + "] , refFilename=[" + refFilename + "]"  + CR ;
				throw new HybsSystemException( errMsg );
			}

			refFilename = filename ;
		}

		// 雛形があれば、雛形ファイルを読み込みます。なければ、オリジナルのファイルを読み込みます。
		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
//		String  nameUse = filename;
		final boolean isRefFile = refFilename != null && refFilename.length() > 0;
//		if( isRefFile ) {
//			nameUse = refFilename ;
//		}
		final String nameUse = isRefFile ? refFilename : filename ;		// 6.3.9.1 (2015/11/27)

		initWriteData();						// 6.0.2.0 (2014/09/19) カラム関連の初期化

		final boolean isDebug = isDebug();		// 6.1.0.0 (2014/12/26) デバッグ情報

		// 6.1.0.0 (2014/12/26) useRenderer 時に使用。できるだけループ内処理を避ける。
		final DBColumn fileBreakClm = fileBreakClmNo >= 0 && isUseRenderer() ? dbColumn[fileBreakClmNo] : null ;
		final DBColumn pageBreakClm = pageBreakClmNo >= 0 && isUseRenderer() ? dbColumn[pageBreakClmNo] : null ;

		// ファイルブレイクの初期値を求めます。(fileKey は、setFilename(String,String) で、初期値セット済み)
		if( fileBreakClmNo >= 0 ) { fileKey = table.getValue( tblRow,fileBreakClmNo ); }
		// 処理のループを抜ける条件は、すべてのデータを書出し終わった場合。fileKey = null をセットします。
		while( fileKey != null ) {
			// fileKey は、super.writeDBTable( null ) で、writeData( DBTableModel ,PrintWriter )で、再セットされる。
			final String fileKey_ren = fileBreakClm == null ? fileKey : fileBreakClm.getWriteValue( fileKey );

			// ExcelModelで作成。新規の場合は、nameUseの拡張子(.xlsx/.xls)に応じて、タイプが変わります。
			excel = new ExcelModel( new File( nameUse ) , isRefFile );		// 6.2.0.0 (2015/02/27)

			// 雛形シートを使わない場合のみ、フォント設定を行う。
			if( !isRefFile ) { excel.setFont( fontName,fontPoint ); }
			// 6.0.2.0 (2014/09/19) 新機能追加
			if( useCellStyle ) { excel.setCellStyle(); }
			excel.useAutoCellSize( useAutoCellSize );
			excel.setRecalcSheetName( recalcSheetNames );					// 6.5.0.0 (2016/09/30) セルの計算式の再計算をさせる recalcSheetNames 属性の追加。
			excel.setAddTitleSheet( addTitleSheet );

			if( isRefFile ) { excel.setRefSheetName( refSheetName ); }

			// シートブレイクする場合は、sheetName名をあらかじめ取得する。
			if( pageBreakClmNo >= 0 ) { sheetName = table.getValue( tblRow,pageBreakClmNo ); }
			noPageBreak = true ;	// ファイルブレイク、シートブレイクの場合は、false をセット
			while( noPageBreak ) {
				// 6.1.0.0 (2014/12/26) デバッグ情報
				if( isDebug ) { System.out.println( " file=" + fileKey + " , sheet=" + sheetName + " , row=" + tblRow ) ; }

				// シート名のセット。重複チェックと新しいシート名の作成処理を行う。
				final String sheetName_ren = pageBreakClm == null ? sheetName : pageBreakClm.getWriteValue( sheetName );
		//		excel.createSheet( sheetName_ren );
				excel.createSheet( sheetName_ren , sheetOverwrite );

				nRowIndex = skipRowCount;		// 6.0.1.2 (2014/08/08) 開始行の指定

				// この super.writeDBTable(null) 内から、各種実態のメソッドが呼ばれる。
				// シートブレイク,ファイルブレイク は、writeData( DBTableModel ,PrintWriter ) で処理される。
				// シートブレイクすると、新しい sheetName がセットされ、noPageBreak=false がセットされる。
				// ファイルブレイクすると、新しい fileKey がセットされ、noPageBreak=false がセットされる。
				super.writeDBTable( null );

				// 指定以降の余計な行を削除
				excel.removeRow( nRowIndex );
			}

			// 6.0.2.0 (2014/09/19) activeWorkbook は、全シートに対して、最後に適用する。
			if( useActiveWorkbook ) { excel.activeWorkbook( false ); }		// false は Cellの末尾削除は行わない。

			// メモリ中のデータをファイルに書き込む
			excel.saveFile( new File( directory,fileKey_ren )  );			// 6.2.0.0 (2015/02/27)
		}
	}

	/**
	 * PrintWriter に DBTableModelのラベル情報を書き込みます。
	 * 第一カラム目は、ラベル情報を示す "#Label" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 * @og.rev 6.2.4.2 (2015/05/29) StringUtil#tagCut(String) をラベルに適用します。
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeLabel( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex = 0;
		excel.createRow( nRowIndex++ );
		if( useNumber ) { headerCellValue( "Label",nColIndex++ ); }

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) カラム飛ばし
				nColIndex++;
				continue;
			}

			final String val = StringUtil.tagCut( dbColumn[clm].getLabel() );		// 6.2.4.2 (2015/05/29)
			headerCellValue( val,nColIndex++ );

			// 6.2.3.0 (2015/05/01) keyLabelは、後ろのカラムのラベルを付けません。
			if( isKeyLbl[i] ) {
				headerCellValue( "",nColIndex++ );
			}
		}

		// 余計なセルを削除
		excel.removeCell( nColIndex );
	}

	/**
	 * PrintWriter に DBTableModelの項目名情報を書き込みます。
	 * 第一カラム目は、項目名情報を示す "#Name" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeName( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex = 0;
		excel.createRow( nRowIndex++ );
		if( useNumber ) { headerCellValue( "Name",nColIndex++ ); }

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) カラム飛ばし
				nColIndex++;
				continue;
			}

			final String name = table.getColumnName( clm );
			headerCellValue( name,nColIndex++ );

			// 6.2.3.0 (2015/05/01) keyLabelは、後ろのカラムの項目名を付けません。
			if( isKeyLbl[i] ) {
				headerCellValue( "",nColIndex++ );
			}
		}

		// 余計なセルを削除
		excel.removeCell( nColIndex );
	}

	/**
	 * PrintWriter に DBTableModelのサイズ情報を書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#Size" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) DBColumn の size と maxlength の 意味を変更
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeSize( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex = 0;
		excel.createRow( nRowIndex++ );
		if( useNumber ) { headerCellValue( "Size",nColIndex++ ); }

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) カラム飛ばし
				nColIndex++;
				continue;
			}

			// 4.0.0 (2005/01/31) メソッド名変更
			final String val = String.valueOf( dbColumn[clm].getTotalSize() );
			headerCellValue( val,nColIndex++ );

			// 6.2.3.0 (2015/05/01) keyLabelは、後ろのカラムのサイズを付けません。
			if( isKeyLbl[i] ) {
				headerCellValue( "",nColIndex++ );
			}
		}

		// 余計なセルを削除
		excel.removeCell( nColIndex );
	}

	/**
	 * PrintWriter に DBTableModelのクラス名情報を書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#Class" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeClass( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex = 0;
		excel.createRow( nRowIndex++ );
		if( useNumber ) { headerCellValue( "Class",nColIndex++ ); }

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) カラム飛ばし
				nColIndex++;
				continue;
			}

			final String val = dbColumn[clm].getClassName();
			headerCellValue( val,nColIndex++ );

			// 6.2.3.0 (2015/05/01) keyLabelは、後ろのカラムのクラス名を付けません。
			if( isKeyLbl[i] ) {
				headerCellValue( "",nColIndex++ );
			}
		}

		// 余計なセルを削除
		excel.removeCell( nColIndex );
	}

	/**
	 * PrintWriter に セパレーターを書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#----" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeSeparator( final DBTableModel table,final PrintWriter writer ) {
		final String SEP = "----" ;

		short nColIndex = 0;
		excel.createRow( nRowIndex++ );
		if( useNumber ) { headerCellValue( SEP,nColIndex++ ); }

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) カラム飛ばし
				nColIndex++;
				continue;
			}
			headerCellValue( SEP,nColIndex++ );

			// 6.2.3.0 (2015/05/01) keyLabelは、後ろのカラムのセパレーターを書き込みます。
			if( isKeyLbl[i] ) {
				headerCellValue( SEP,nColIndex++ );
			}
		}

		// 余計なセルを削除
		excel.removeCell( nColIndex );
	}

	/**
	 * ExcelModel#setCellValue(String.int) の第一カラムの場合に、コメントの # を追加する簡易メソッドです。
	 *
	 * colNo == 0 の場合だけ、引数の val に、"#" を追加します。
	 * これは、openGion特有のコメント行を意味するため、ヘッダーレコードにのみ適用します。
	 *
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 *
	 * @param	val   設定文字列
	 * @param	colNo 行番号
	 */
	private void headerCellValue( final String val,final int colNo ) {
		if( colNo == 0 ) {
			excel.setCellValue( "#" + val,colNo );
		}
		else {
			excel.setCellValue( val,colNo );
		}
	}

	/**
	 * writeData( DBTableModel ,PrintWriter ) の呼び出し前に、一度だけ実行すればよい処理をまとめました。
	 * 
	 * 通常のTableWriterは、１回の DBTableModel の読み込みで、１ファイルだけ出力します。
	 * そのため、writeData( DBTableModel ,PrintWriter ) の呼び出し処理の初めに、初期化している処理があります。
	 * EXCELでは、Sheet 処理と、File 処理のループがあり、本来１回だけ初期化すればよいのですが、
	 * writeData( DBTableModel ,PrintWriter ) が複数回呼ばれるため、無駄な処理になってしまいます。
	 * ただし、今までは、ローカル変数でしたが、インスタンス変数に変更しています。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規追加
	 */
	private void initWriteData() {
		final DBTableModel table = getDBTableModel();
		pageBreakClmNo	= table.getColumnNo( pageBreakColumn,false );	// シートブレイクするキーとなるカラムNo
		fileBreakClmNo	= table.getColumnNo( fileBreakColumn,false );	// ファイルブレイクするキーとなるカラムNo

		hLinkKeyClmNo	= table.getColumnNo( hLinkKeyColumn,false );	// Hyperlinkを作成するキーとなるカラムNo
		hLinkValClmNo	= table.getColumnNo( hLinkValColumn,false );	// Hyperlinkを作成する値となるカラムNo

		nvar	= new boolean[numberOfColumns];
		isNums	= new boolean[numberOfColumns];				// 6.0.2.0 (2014/09/19) NUMBER型かどうか
		final boolean useRenderer = isUseRenderer();		// 5.2.1.0 (2010/10/01)

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) { continue; }			// 6.0.1.2 (2014/08/08) カラム飛ばし

			// 5.7.6.3 (2014/05/23) ここでレンデラ時の文字タイプ判定を行う
			if( useRenderer && dbColumn[clm].isStringOutput() ){
				isNums[i]   = false;
			}
			else {
				final NativeType nativeType = dbColumn[clm].getNativeType();
				switch( nativeType ) {
					case INT    :
					case LONG   :
					case DOUBLE :
						isNums[i]   = true;
							break;
					case STRING :
					case CALENDAR :
					default :
							isNums[i]   = false;
							break;
				}
			}
			nvar[i] = "NVAR".equals( dbColumn[clm].getDbType()) ;
		}
	}

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 * このクラスでは，データを ダブルコーテーション(")で囲みます。
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * @og.rev 3.8.0.1 (2005/06/17) DBTypeが NVAR の場合は、元のUnicodeに戻します。
	 * @og.rev 3.8.5.3 (2006/08/07) DBType の nativeType に対応した、CELL_TYPE をセットします。
	 * @og.rev 4.1.1.2 (2008/02/28) NativeタイプをEnum型(fukurou.model.NativeType)に変更
	 * @og.rev 5.1.4.0 (2010/03/01) columns 対応
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 5.2.1.0 (2010/10/01) useRenderer 対応
	 * @og.rev 5.7.6.3 (2014/05/23) stringOutput対応
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト(未使用)
	 */
	@Override
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		final int numberOfRows    = table.getRowCount();
		final boolean useRenderer = isUseRenderer();				// 5.2.1.0 (2010/10/01)

		if( useAutoCellSize ) {
			excel.setDataStartRow( nRowIndex );				// データ行の書き込み開始位置の行番号を設定
		}

		int rowNo = 1;
		for( ; tblRow<numberOfRows; tblRow++ ) {			// tblRow は、シートブレイクしても連続してカウントアップ
			// まずはファイルブレイク判定
			if( fileBreakClmNo >= 0 ) {
				// ファイルブレイクすると、noPageBreak=false と、noFileBreak=false がセットされる。
				final String nowFBC = table.getValue( tblRow,fileBreakClmNo );
				if( !fileKey.equals( nowFBC ) ) {
					fileKey = nowFBC;
					noPageBreak = false;
					return;
				}
			}
			if( pageBreakClmNo >= 0 ) {
				// シートブレイクすると、新しい sheetName がセットされ、noPageBreak=false がセットされる。
				final String nowPBC = table.getValue( tblRow,pageBreakClmNo );
				if( !sheetName.equals( nowPBC ) ) {
					sheetName = nowPBC;
					return;
				}
			}

			short nColIndex = 0;
			excel.createRow( nRowIndex++ );
			if( useNumber ) { excel.setCellValue( String.valueOf( rowNo++ ),nColIndex++,true ); }

			for( int i=0; i<numberOfColumns; i++ ) {
				final int clm = clmNo[i];
				if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) カラム飛ばし
					nColIndex++;
					continue;
				}

				String val = table.getValue( tblRow,clm );

				// 6.2.3.0 (2015/05/01) keyLabelは、後ろのカラムのラベルを付けません。
				if( isKeyLbl[i] ) {
					excel.setCellValue( val,nColIndex++,isNums[i] );
				}

				if( nvar[i] ) {
					val = StringUtil.getReplaceEscape( val );
				}
				// 5.2.1.0 (2010/10/01) useRenderer 対応
				else if( useRenderer ) {
					// 6.0.4.0 (2014/11/28) データ出力用のレンデラー
					val = dbColumn[clm].getWriteValue( val );
				}

				excel.setCellValue( val,nColIndex,isNums[i] );

				// ハイパーリンク設定は、文字の設定の後で行います。
				if( clm == hLinkKeyClmNo && hLinkValClmNo >= 0 ) {
					final String shtNm = table.getValue( tblRow,hLinkValClmNo );
					excel.setCellLink( shtNm,nColIndex );
				}

				nColIndex++;
			}

			// 余計なセルを削除
			excel.removeCell( nColIndex );
		}
		noPageBreak = false;	// 最後まで終了した
		fileKey		= null;		// 最後まで終了した
	}

	/**
	 * 出力先ディレクトリとファイル名をセットします。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * 雛形ファイルとの関係から、内部的に filename は合成しておきます。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 * @og.rev 6.0.2.0 (2014/09/19) ディレクトリとファイルを分けて管理します。
	 *
	 * @param   dir   出力先ディレクトリ
	 * @param   fname 出力ファイル名
	 */
	@Override
	public void setFilename( final String dir , final String fname ) {
		filename	= StringUtil.urlAppend( dir,fname ) ;

		directory	= dir;
		fileKey		= fname;
	}

	/**
	 * DBTableModelのデータとして読み込むときのシート名を設定します。
	 * 初期値は、"Sheet1" です。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   shtName シート名
	 */
	@Override
	public void setSheetName( final String shtName ) {
		if( shtName != null ) { this.sheetName = shtName; }
	}

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、シート名のセット、雛型参照ファイル名のセット、
	 * 書き込み元ファイルのFileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return	EXCEL対応機能を持っているかどうか(常に true)
	 */
	@Override
	public boolean isExcel() {
		return true;
	}

	/**
	 * EXCEL雛型参考ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   rfname EXCEL雛型参考ファイル名
	 */
	@Override
	public void setRefFilename( final String rfname ) {
		refFilename = rfname;
	}

	/**
	 * 雛形のシート名を、そのまま使用する(true)か、新規、または、外部指定のシート名を使用する(false)を指定します。(初期値:false[外部指定のシート名を使用])。
	 *
	 * ※ Ver5では、追記モード時に、指定シートが存在した場合は上書きします(初期値:false[上書きしない])。5.9.12.1 (2016/09/09)
	 *    Ver6では、追記モード時に、雛形を指定できないため、雛形のシート名を、そのまま使用する(true)か、
	 *    新規、または、外部指定のシート名を使用する(false)を指定する属性になります。
	 * 
	 * @og.rev 6.5.0.0 (2016/09/30) sheetOverwrite で、雛形シートの使用時に、元のシート名を使用します。
	 *
	 * @param   flag 元のシート名を使用するかどうか[true:使用する/false:新規、または、外部指定のシート名を使用]
	 */
	@Override
	public void setSheetOverwrite( final boolean flag ) {
		sheetOverwrite = flag;
	}

	/**
	 * EXCELで、出力処理の最後にセルの計算式の再計算をさせるシート名をカンマ区切りで指定します。
	 *
	 * @og.rev 6.5.0.0 (2016/09/30) recalcSheetName で、セル内の計算式を再計算させるシート名を指定。5.9.12.1 (2016/09/09)
	 *
	 * @param  sheet 対象シート名をカンマ区切りで指定
	 */
	@Override
	public void setRecalcSheetName( final String sheet ){
		recalcSheetNames = StringUtil.csv2Array( sheet);
	}

	/**
	 * EXCEL雛型参考ファイルのシート名を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELファイルを書き出す時に、雛型として参照するシート名を指定します。
	 * これにより、複数の形式の異なるデータを順次書き出したり(appendモードを併用)する
	 * ことや、シートを指定して新規にEXCELを作成する場合にフォームを設定する事が可能になります。
	 * 初期値は、null(第一シート) です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @param   rshtName シート名
	 */
	@Override
	public void setRefSheetName( final String rshtName )  {
		if( rshtName != null ) { refSheetName = rshtName; }
	}

	/**
	 * データの書き込み開始位置を設定します(初期値:0)。
	 *
	 * TAB区切りテキストやEXCEL等のデータの書き込みの開始位置を指定します。
	 * 属性名は、行を飛ばす処理ということで、readTable タグと同じ名称です。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１件読み飛ばし、２行目から読み込みます。)
	 * 行の読み飛ばしと、カラムの読み飛ばし(columns)、refFileURL、refFilename、
	 * refSheetName とともに使用すれば、ある程度のレイアウト設定が可能です。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.1.2 (2014/08/08) 新規作成
	 *
	 * @param	skipCount 書き込み開始位置
	 */
	@Override
	public void setSkipRowCount( final int skipCount ) {
		skipRowCount = skipCount ;
	}

	/**
	 * EXCEL出力時のデフォルトフォント名を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELファイルを書き出す時に、デフォルトフォント名を指定します。
	 * フォント名は、EXCELのフォント名をそのまま使用してください。
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontName( String )
	 * に設定されます。
	 * 雛形ファイルを使用しない場合のみ、有効です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   font デフォルトフォント名
	 */
	@Override
	public void setFontName( final String font ) {
		fontName = font ;
	}

	/**
	 * EXCEL出力時のデフォルトフォントポイント数を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELファイルを書き出す時に、デフォルトポイント数を指定します。
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontHeightInPoints( short )
	 * に設定されます。
	 * 雛形ファイルを使用しない場合のみ、有効です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   point デフォルトフォントポイント数
	 */
	@Override
	public void setFontPoint( final short point ) {
		fontPoint = point;
	}

	/**
	 * EXCEL出力時に、データを書き込んだ範囲に罫線を入れるかどうかを指定します。
	 *
	 * データを書き込んでEXCELを作成しても、ノーマルのセルに値がセットされている
	 * だけなので、ある程度加工が必要です。
	 * そこで、データのセットされたセルに罫線を入れることで、それなりのデータが
	 * 出力された感じになります。
	 * この設定と、useAutoCellSize="true" で、セルの幅を自動調整すれば、見栄えが良くなります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useCellStyle 罫線を入れるかどうか(true:入れる/false:入れない)
	 * @see		#setUseAutoCellSize( boolean )
	 */
	@Override
	public void setUseCellStyle( final boolean useCellStyle ) {
		this.useCellStyle = useCellStyle;
	}

	/**
	 * EXCEL出力時に、セルの幅をデータの幅に自動的に合わせるかどうかを指定します。
	 *
	 * データを書き込んでEXCELを作成しても、ノーマルのセルに値がセットされている
	 * だけなので、ある程度加工が必要です。
	 * そこで、データのセットされたセルの幅を自動調整することで、それなりのデータが
	 * 出力された感じになります。
	 * この設定と、useCellStyle="true" で、セルの罫線を自動設定すれば、見栄えが良くなります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useAutoCellSize データの幅に自動的に合わせるかどうか(true:自動調整/false:何もしない)
	 * @see		#setUseCellStyle( boolean )
	 */
	@Override
	public void setUseAutoCellSize( final boolean useAutoCellSize ) {
		this.useAutoCellSize = useAutoCellSize;
	}

	/**
	 * EXCEL出力時に、セルの有効範囲を設定するかどうかを指定します。
	 *
	 * セルの有効範囲というのは、EXCELでの 空行、空列の存在しない範囲を指します。
	 * 通常、空行でも、データとして残っている場合は、EXCELのセルオブジェクトは存在します。
	 * ここで、useActiveWorkbook="true" とすると、空行、空列を削除します。
	 * 
	 * 雛形を使用した場合は、データより多めに設定した計算などは、この処理で
	 * 削除されますので、データサイズにフィットさせることができます。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useActiveWorkbook セルの有効範囲を設定するかどうか(true:設定する/false:そのまま)
	 */
	@Override
	public void setUseActiveWorkbook( final boolean useActiveWorkbook ) {
		this.useActiveWorkbook = useActiveWorkbook;
	}

	/**
	 * EXCEL出力時に、シートブレイクするキーとなるカラム名を指定します(このカラムの値がシート名になります)。
	 *
	 * EXCEL帳票では、帳票雛形に、PAGE_BRAKE キーを設定しましたが、TableWriterでは、
	 * メモリ上のカラムの値が変更したときに、シートブレイクさせることができます。
	 * このカラムの値がキーブレイクすると、新しいシートに書き出し始めます。
	 * シート名は、このカラムの値（キーブレイクする値）です。
	 * 
	 * 雛形ファイルを使用する場合、雛形シートもキーブレイクに伴って、＋１されます。
	 * つまり、雛形シートとデータシートは同時に変更されます。
	 * ただし、雛形シートは、最後の雛形シートで止まります。
	 * これは、雛形シートにヘッダー雛形とボディ雛形を用意しておき、最初のキーブレイクで
	 * ヘッダーからボディの書き込みにチェンジするイメージで使用できます。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	pageBreakColumn シートブレイクするキーとなるカラム名を指定
	 * @see		#setFileBreakColumn( String )
	 */
	@Override
	public void setPageBreakColumn( final String pageBreakColumn ) {
		this.pageBreakColumn = pageBreakColumn;
	}

	/**
	 * EXCEL出力時に、ファイルブレイクするキーとなるカラム名を指定します(このカラムの値がファイル名になります)。
	 *
	 * EXCEL帳票では、メモリ上のカラムの値が変更したときに、ファイルブレイクすることができます。
	 * このカラムの値がキーブレイクすると、新しいファイルに書き出し始めます。
	 * ファイル名は、このカラムの値（キーブレイクする値）＋ 元の出力ファイル名の拡張子(.xlsなど)です。
	 * この設定を使用する場合は、出力ファイル名は無視されますが、拡張子だけは使用されます。
	 * 
	 * 雛形ファイルを使用する場合、雛形ファイルもキーブレイクに伴って、再利用されます。
	 * 例えば、pageBreakColumn と併用する場合、キーブレイクで雛形シートも最初から適用になります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	fileBreakColumn ファイルブレイクするキーとなるカラム名を指定
	 * @see		#setPageBreakColumn( String )
	 */
	@Override
	public void setFileBreakColumn( final String fileBreakColumn ) {
		this.fileBreakColumn = fileBreakColumn;
	}

	/**
	 * EXCEL出力時に、Hyperlinkを作成するキーとなるカラム名と値となるカラム名を指定します。
	 *
	 * ここで、作成するハイパーリンクは、EXCELのシートに対するハイパーリンクです。
	 * それ以外のリンク(本当のURLやファイル等)のリンクは(今は)作成できません。
	 * ハイパーリンクを作成するには、①作成するカラム と ②作成する値 が必要です。
	 * このメソッドで設定するのは、「①:②」という形式でカラム名を指定します。
	 * ②がなければ、①と同じとします。
	 * ②の値のシートの存在有無は、無視します。ハイパーリンクを作成するシートを作成する前に
	 * ハイパーリンクを作成するケースが存在します。
	 * (例えば、各シートへのリンクを持った一覧を作成してから、明細の各シートを作成する様なケース)
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	hyperLinkColumn Hyperlinkを作成するキーとなるカラム名と値となるカラム名を指定
	 */
	@Override
	public void setHyperLinkColumn( final String hyperLinkColumn ) {
		if( hyperLinkColumn != null && hyperLinkColumn.length() > 0 ) {
			final int sep = hyperLinkColumn.indexOf( ':' );					// ①:② の形式かどうかの確認
			if( sep >= 0 ) {
				hLinkKeyColumn = hyperLinkColumn.substring( 0,sep );	// ①キーカラム
				hLinkValColumn = hyperLinkColumn.substring( sep+1 );	// ②値カラム
			}
			else {
				hLinkKeyColumn = hyperLinkColumn;						// ①キーカラム = ②値カラム
				hLinkValColumn = hyperLinkColumn;
			}
		}
	}

	/**
	 * EXCEL出力時に、Sheet一覧を先頭Sheetに作成する場合のSheet名を指定します。
	 *
	 * これは、Workbook に含まれる Sheet 一覧を作成する場合に、利用可能です。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	sheetName EXCELファイルのシート名
	 */
	@Override
	public void setAddTitleSheet( final String sheetName ) {
		addTitleSheet = sheetName ;
	}
}
