/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;

import java.util.Map;
import java.util.HashMap;

/**
 * TableFilter_KEY_BREAK は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、指定のカラムに対して、キーブレイクが発生したときのデータのみ、残します。
 * キーブレイクは、グループキーと、ブレークキーがあります。
 * グループキーは、一塊のレコードを管理し、グループキーごとに、キーブレイクを判定します。
 * つまり、グループキーの並び順に依存しない形で、キーブレイク可能です。
 * 例：機種と日付と、状況Fがあったとして、日付、機種、状況F でソートし、機種をグループキー、
 * 状況Fをブレイクキーとすれば、日付の順に、機種の中で、状況Fがブレークしたときのみ、
 * データを残す、ということが可能になります。
 *
 * GRP_KEY  : グループの判定を行うカラムを、CSV形式で設定します。
 * BRK_KEY  : キーブレイクの判定を行うカラムを、CSV形式で設定します。
 * USE_LAST : キーブレイクと関係なく、グループの最後のデータを登録するかどうかを指定します。(初期値:false 登録しない)
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 *
 * @og.formSample
 * ●形式：
 *      ① &lt;og:tableFilter classId="KEY_BREAK"
 *				keys="GRP_KEY,BRK_KEY,USE_LAST"
 * 				vals='"CLM1,CLM2....","CLM5,CLM6....",true' /&gt;
 *
 *      ② &lt;og:tableFilter classId="KEY_BREAK" &gt;
 *               {
 *                   GRP_KEY  : CLM1,CLM2....   ;
 *                   BRK_KEY  : CLM5,CLM6....   ;
 *                   USE_LAST : true            ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
 *
 * @version  6.7  2017/05/19
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.8,
 */
public class TableFilter_KEY_BREAK extends AbstractTableFilter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.7.9.1 (2017/05/19)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
	 */
	public TableFilter_KEY_BREAK() {
		super();
		initSet( "GRP_KEY"	, "グループの判定を行うカラムを、CSV形式で設定します。" );
		initSet( "BRK_KEY"	, "キーブレイクの判定を行うカラムを、CSV形式で設定します。" );
		initSet( "USE_LAST"	, "キーブレイクと関係なく、グループの最後のデータを登録するかどうかを指定します。(初期値:false 登録しない)"		);
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		final String[]	grpClms	= StringUtil.csv2Array( getValue( "GRP_KEY" ) );
		final String[]	brkClms	= StringUtil.csv2Array( getValue( "BRK_KEY" ) );

		final boolean	useLast = StringUtil.nval( getValue( "USE_LAST"  ), false ) ;

		final DBTableModel table  = getDBTableModel();
	//	final DBTableModel rtnTbl = table.newModel();

		final int[]    grpClmNo = new int[grpClms.length];		// グループキーカラムの番号
		final int[]    brkClmNo = new int[brkClms.length];		// ブレイクキーカラムの番号

		for( int i=0; i<grpClms.length; i++ ) {
			grpClmNo[i] = table.getColumnNo( grpClms[i],false );	// カラムが存在しなければ、-1
		}

		for( int i=0; i<brkClms.length; i++ ) {
			brkClmNo[i] = table.getColumnNo( brkClms[i],false );	// カラムが存在しなければ、-1
		}

		final int rowCnt = table.getRowCount();

		final Map<String,String> brkKeysMap = new HashMap<>() ;		// グループキー単位の最後のブレークキー

		String[] data    = null;	// エラー時に表示するため。
		for( int row=0; row<rowCnt; row++ ) {
			try {
				data = table.getValues( row );

				final String grpKeys	= getKeys( grpClmNo , data );	// グループキー
				final String oldBlkKeys	= brkKeysMap.getOrDefault( grpKeys , "" );		// Mapにキーが無い場合は、空文字列

				final String brkKeys = getKeys( brkClmNo , data );	// ブレークキー
				if( oldBlkKeys.equalsIgnoreCase( brkKeys ) ) {
					table.rowDelete( row );					// キーが同じなので、削除ﾌﾗｸﾞを立てる。(論理削除)
				}
				else {
					brkKeysMap.put( grpKeys , brkKeys );	// キーが異なるので、Mapにｾｰﾌﾞします。
				}
			}
			catch( final RuntimeException ex ) {
				// 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
				makeErrorMessage( "TableFilter_KEY_BREAK Error",ErrorMessage.NG )
					.addMessage( row+1,ErrorMessage.NG,"KEY_BREAK"
						, StringUtil.array2csv( data )
					)
					.addMessage( ex );
			}
		}

		// レコードを削除するので、逆順に行います。
		for( int row=rowCnt-1; row>=0; row-- ) {
			final String modType = table.getModifyType( row );
			if( DBTableModel.DELETE_TYPE.equals( modType ) ) {			// 削除対象 (DELETE_TYPE は、fukurou.model.DataModel で定義
				if( useLast ) {											// 若干複雑
					data = table.getValues( row );
					final String grpKeys = getKeys( grpClmNo , data );	// グループキー
					if( brkKeysMap.containsKey( grpKeys ) ) {			// brkKeysMap を流用。useLast の場合、逆順で
						brkKeysMap.remove( grpKeys );					// 最後のグループキーのデータは、
						continue;										// 削除しないので、continue する。
					}
				}
				table.removeValue( row );	// 物理削除
			}
		}

		if( useLast ) { table.resetModify(); }	// useLast 使用時には、DELETEが残るため、一応クリアしておきます。

		return table;
	}

	/**
	 * キーの配列ｱﾄﾞﾚｽと、１行分のデータ配列から、キーとなる文字列を作成します。
	 *
	 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
	 *
	 * @param clms キーの配列ｱﾄﾞﾚｽ
	 * @param rowData １行分のデータ配列
	 * @return キーとなる文字列
	 */
	private String getKeys( final int[] clms , final String[] rowData ) {
		final StringBuilder buf = new StringBuilder();
		for( int i=0; i<clms.length; i++ ) {
			if( clms[i] >= 0 ) {
				buf.append( rowData[clms[i]] ).append( ':' );
			}
		}
		return buf.toString();
	}
}
