/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.HybsEntry ;
import org.opengion.fukurou.xml.XSLT;
import org.opengion.fukurou.system.LogWriter;

import java.util.Map ;
import java.util.LinkedHashMap ;
import java.io.File;

/**
 * XSLT変換結果を指定のファイルに出力します。
 *
 * Process_XSLT は、AbstractProcess を継承した、ChainProcess インターフェース
 * の実装クラスです。
 * 上流(プロセスチェインのデータは上流から渡されます。)からのLineModel の
 * ファイルオブジェクトに対して、指定の XSL ファイルを適用して、XSL変換を行います。
 * 出力結果は、ファイル、または 標準出力に出力できます。
 *
 * 上流プロセスでは、Name 属性として、『File』を持ち、値は、Fileオブジェクト
 * である、Process_FileSearch を使用するのが、便利です。それ以外のクラスを
 * 使用する場合でも、Name属性と、File オブジェクトを持つ LineModel を受け渡し
 * できれば、使用可能です。
 *
 * -param_XXXX=固定値 を使用して、XSLTにパラメータを設定できます。
 *
 * それ以外では、org.opengion.fukurou.xml.XSLT で、入力ファイル情報の設定が可能に
 * なっている為、内部情報を使用するかどうか -useFileInfo を指定できます。
 * -useFileInfo=true とセットすると、以下の４項目が内部的にセットされます。
 *
 * 入力ファイル(inXMLのフルパス)     : FILEPATH  (例: G:\webapps\gf\jsp\DOC10\query.jsp)
 * 入力親フォルダ(inXMLの親フォルダ) : ADDRESS   (例: DOC10)
 * 入力ファイル(inXMLのファイル名)   : FILENAME  (例: query.jsp)
 * 入力ファイル(inXMLの更新日付  )   : MODIFIED  (例: yyyyMMddHHmmss形式)
 *
 * xsl ファイルでは、xsl:param で宣言し、xsl:value-of で取り出します。
 * &lt;xsl:param name="ADDRESS" select="" /&gt; と宣言しておき、必要な箇所で
 * &lt;xsl:value-of select="$ADDRESS"     /&gt; とすれば、取得できます。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * @og.formSample
 *  Process_XSLT -xslfile=xslファイル -outfile=OUTFILE -append=true
 *
 *    -xslfile=xslファイル        ：変換を行う XSLファイル
 *   [-outfile=出力ファイル名   ] ：変換結果の出力ファイル名
 *   [-append=[false/true]      ] ：出力ファイルを、追記する(true)か新規作成する(false)か
 *   [-useFileInfo=[false/true] ] ：入力ファイル情報を、XSLTのパラメータにセットする(true)かしないか(false)か
 *   [-addROWSET=テーブル名     ] ：ヘッダー/フッターに ROWSET を追記します。
 *   [-headerXX=ヘッダー文字列  ] ：出力ファイルに、ヘッダー文字列を追記します。
 *                                         添え字(XX)が異なれば複数のヘッダーが指定できます。
 *   [-footerXX=フッター文字列  ] ：出力ファイルに、フッター文字列を追記します。
 *                                         添え字(XX)が異なれば複数のフッターが指定できます。
 *   [-param_XXXX=固定値        ] ：-param_SYSTEM_ID=GE
 *                                    XSLパーサーに対して、paramater を設定します。
 *                                    キーが異なれば、複数のパラメータを指定できます。
 *   [ -errAbend=[true/false]   ] ：異常発生時に、処理を中断(true)するか、継続(false)するかを指定する(初期値:true[中断する])
 *   [ -errXmlIn=[false/true]   ] ：異常発生時に、出力ファイルに、XML形式でエラーを追記するかを指定する(初期値:false[使用しない])
 *   [ -jspInclude=[true/false] ] ：jsp:directive.include 発見時に、そのファイルを INCLUDE するかを指定する(初期値:true[使用する])
 *   [ -realPath=実際の実行環境 ] ：jspInclude="true" 時に、/jsp/common/以下のファイルの取得先を指定します(初期値:null)
 *   [ -display=[false/true]    ] ：結果を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *   [ -debug=[false/true]      ] ：デバッグ情報を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_XSLT extends AbstractProcess implements ChainProcess {
	private static final String PARAM_KEY	= "param_" ;
	private static final String HEADER_KEY	= "header" ;
	private static final String FOOTER_KEY	= "footer" ;
	private static final String FILE_KEY	= "File";

	private static final String HEADER_XML    = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" ;
	private static final String HEADER_ROWSET = "<ROWSET tableName=\"TABLENAME\">" ;
	private static final String FOOTER_ROWSET = "</ROWSET>" ;

	private XSLT		xslt		;
	private HybsEntry[]	footerEntry	;
	private String		xslfile		;
	private String		outfile		;
	private boolean		errAbend	= true;		// 中断する
	private boolean		errXmlIn	;			// エラーXML形式
	private boolean		jspInclude	= true;		// 4.2.3.0 (2008/05/26)
	private String		realPath	;			// 5.7.6.2 (2014/05/16) 追加
	private boolean		display		;			// 表示しない
	private boolean		debug		;			// 5.7.3.0 (2014/02/07) デバッグ情報

	private int		clmNo		= -1;
	private int		inCount		;
	private int		errCount	;
	private String	tableName	;

	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> MUST_PROPARTY   ;		// ［プロパティ］必須チェック用 Map
	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> USABLE_PROPARTY ;		// ［プロパティ］整合性チェック Map

	static {
		MUST_PROPARTY = new LinkedHashMap<>();
		MUST_PROPARTY.put( "xslfile",	"変換を行う XSLファイル(必須)" );

		USABLE_PROPARTY = new LinkedHashMap<>();
		USABLE_PROPARTY.put( "outfile",		"変換結果の出力ファイル名" );
		USABLE_PROPARTY.put( "append",		"出力ファイルを、追記する(true)か新規作成する(false)か" );
		USABLE_PROPARTY.put( "useFileInfo",	"入力ファイル情報を、XSLTのパラメータにセットする(true)かしないか(false)か" );
		USABLE_PROPARTY.put( "addROWSET"	,	"ヘッダー/フッターに ROWSET を追記します。");
		USABLE_PROPARTY.put( "header",		"出力ファイルに、ヘッダー文字列を追記します。" +
									CR + "添え字(XX)が異なれば複数のヘッダーが指定できます。" );
		USABLE_PROPARTY.put( "footer",		"出力ファイルに、フッター文字列を追記します。" +
									CR + "添え字(XX)が異なれば複数のヘッダーが指定できます。" );
		USABLE_PROPARTY.put( "param_",		"XSLパーサーに対して、paramater を設定します。" +
									CR + "キーが異なれば、複数のパラメータを指定できます。" +
									CR + "例: -param_SYSTEM_ID=GE" );
		USABLE_PROPARTY.put( "errAbend",	"異常発生時に、処理を中断(true)するか、継続(false)するか" +
									CR + "(初期値:true:中断する)" );
		USABLE_PROPARTY.put( "errXmlIn",	"異常発生時に、出力ファイルに、XML形式でエラーを追記するかを指定する" +
									CR + "(初期値:false:使用しない)" );
		USABLE_PROPARTY.put( "jspInclude","jsp:directive.include 発見時に、そのファイルを INCLUDE するかを指定する" +
									CR + "(初期値:true:使用する)" );
		USABLE_PROPARTY.put( "realPath","jspInclude=\"true\" 時に、/jsp/common/以下のファイルの取得先を指定します。" +
									CR + "(初期値:null)" );		// 5.7.6.2 (2014/05/16) 追加
		USABLE_PROPARTY.put( "display",	"結果を標準出力に表示する(true)かしない(false)か" +
									CR + "(初期値:false:表示しない)" );
		USABLE_PROPARTY.put( "debug",	"デバッグ情報を標準出力に表示する(true)かしない(false)か" +
									CR + "(初期値:false:表示しない)" );		// 5.7.3.0 (2014/02/07) デバッグ情報
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_XSLT() {
		super( "org.opengion.fukurou.process.Process_XSLT",MUST_PROPARTY,USABLE_PROPARTY );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 4.2.3.0 (2008/05/26) jsp:directive.include 処理の実施可否を引数指定します。
	 * @og.rev 5.7.6.2 (2014/05/16) realPath 引数を追加します。
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		final Argument arg = getArgument();

		xslfile				= arg.getProparty( "xslfile" );
		outfile				= arg.getProparty( "outfile" );
		tableName			= arg.getProparty( "addROWSET" );
//		final boolean isAppend	= arg.getProparty( "append",false );
//		final boolean useFileInfo		= arg.getProparty( "useFileInfo",false );
//		final HybsEntry[] paramEntry	= arg.getEntrys( PARAM_KEY );		// 配列
//		final HybsEntry[] headerEntry	= arg.getEntrys( HEADER_KEY );		// 配列
		footerEntry			= arg.getEntrys( FOOTER_KEY );					// 配列
		errAbend			= arg.getProparty("errAbend",errAbend);
		errXmlIn			= arg.getProparty("errXmlIn",errXmlIn);
		jspInclude			= arg.getProparty("jspInclude",jspInclude);		// 4.2.3.0 (2008/05/26) 追加
		realPath			= arg.getProparty("realPath"  ,realPath);		// 5.7.6.2 (2014/05/16) 追加
		display				= arg.getProparty("display",display);
		debug				= arg.getProparty("debug",debug);				// 5.7.3.0 (2014/02/07) デバッグ情報

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( outfile == null ) {
			// 出力先ファイル名が、指定されていない場合
			final String errMsg = "outfile が指定されていません。";
			throw new OgRuntimeException( errMsg );
		}

//		if( outfile != null ) {
			final File file = new File( outfile );		// 5.5.2.6 (2012/05/25) findbugs対応
			final File dir = file.getParentFile() ;

			// 親ディレクトリを示さない場合は null 。ディレクトリが存在しない、かつ、ディレクトリが作成できない場合の処理
			if( dir != null && ! dir.exists() && ! dir.mkdirs() ) {
				final String errMsg = "ディレクトリが作成できませんでした。[" + dir + "]" ;
				throw new OgRuntimeException( errMsg );
			}
//		}
//		else {
//			// 出力先ファイル名が、指定されていない場合
//			final String errMsg = "outfile が指定されていません。";
//			throw new OgRuntimeException( errMsg );
//		}

		xslt = new XSLT();

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final boolean isAppend			= arg.getProparty( "append",false );
		final boolean useFileInfo		= arg.getProparty( "useFileInfo",false );
		final HybsEntry[] paramEntry	= arg.getEntrys( PARAM_KEY );		// 配列
		final HybsEntry[] headerEntry	= arg.getEntrys( HEADER_KEY );		// 配列

		xslt.setOutFile( outfile,isAppend );
		xslt.setXslFile( xslfile );
		xslt.setParamEntry( paramEntry );
		xslt.useFileInfo( useFileInfo );
		xslt.errClose( errAbend );			// エラー時に出力ファイルを閉じるかどうか。
		xslt.useErrXmlIn( errXmlIn );		// エラー時にXML形式で出力ファイルに追記するかどうか。
		xslt.jspInclude( jspInclude );		// 4.2.3.0 (2008/05/26) jsp:directive.include するかどうか
		xslt.setRealPath( realPath );		// 5.7.6.2 (2014/05/16) realPath 引数を追加します。

		if( tableName != null ) {
			xslt.setOutData( HEADER_XML );
			xslt.setOutData( HEADER_ROWSET.replace( "TABLENAME",tableName ) );
		}

		final int size   = headerEntry.length;
		for( int i=0; i<size; i++ ) {
			xslt.setOutData( headerEntry[i].getValue() );
		}
	}

	/**
	 * 引数の LineModel を処理するメソッドです。
	 * 変換処理後の LineModel を返します。
	 * 後続処理を行わない場合(データのフィルタリングを行う場合)は、
	 * null データを返します。つまり、null データは、後続処理を行わない
	 * フラグの代わりにも使用しています。
	 * なお、変換処理後の LineModel と、オリジナルの LineModel が、
	 * 同一か、コピー(クローン)かは、各処理メソッド内で決めています。
	 * ドキュメントに明記されていない場合は、副作用が問題になる場合は、
	 * 各処理ごとに自分でコピー(クローン)して下さい。
	 *
	 * @param   data	オリジナルのLineModel
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel action( final LineModel data ) {
		inCount++ ;
		if( display ) { println( data.dataLine() ); }
		if( clmNo < 0 ) { clmNo = data.getColumnNo( FILE_KEY ); }
		final File file = (File)data.getValue( clmNo );

		if( ! file.isFile() ) { return data; }

		final String filePath = file.getPath();

		try {
			if( debug ) { println( filePath ); }			// 5.7.3.0 (2014/02/07) デバッグ情報
			xslt.transform( filePath );
		}
		catch( final RuntimeException ex ) {
			errCount++ ;
			if( errAbend ) { throw ex; }
			else {
				logging( ex.getMessage() );
				logging( "xslfile  = " + xslfile );
				logging( "outfile  = " + outfile );
				logging( "xmlFile  = " + filePath );
			}
		}

		return data ;
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		if( xslt != null ) {
			if( isOK ) {
				final int size = footerEntry.length;
				for( int i=0; i<size; i++ ) {
					xslt.setOutData( footerEntry[i].getValue() );
				}
				if( tableName != null ) {
					xslt.setOutData( FOOTER_ROWSET );
				}
			}
			xslt.close();
		}
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		final String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "XSL File   : " + xslfile   + CR
				+ TAB + "OUT File   : " + outfile   + CR
				+ TAB + "Table Name : " + tableName + CR
				+ TAB + "File Count : " + inCount   + CR
				+ TAB + "Err  Count : " + errCount ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 * @og.rtnNotNull
	 */
	public String usage() {
		final StringBuilder buf = new StringBuilder( 1000 )
		.append( "XSLT変換結果を指定のファイルに出力します。"									).append( CR )
		.append( CR )
		.append( "Process_XSLT は、AbstractProcess を継承した、ChainProcess インターフェース"	).append( CR )
		.append( "の実装クラスです。"															).append( CR )
		.append( "上流(プロセスチェインのデータは上流から渡されます。)からのLineModel の"		).append( CR )
		.append( "ファイルオブジェクトに対して、指定の XSL ファイルを適用して、XSL変換を"		).append( CR )
		.append( "行います。出力結果は、ファイル、または 標準出力に出力できます。"				).append( CR )
		.append( CR )
		.append( "上流プロセスでは、Name 属性として、『File』を持ち、値は、Fileオブジェクト"	).append( CR )
		.append( "である、Process_FileSearch を使用するのが、便利です。それ以外のクラスを"		).append( CR )
		.append( "使用する場合でも、Name属性と、File オブジェクトを持つ LineModel を受け渡し"	).append( CR )
		.append( "できれば、使用可能です。"														).append( CR )
		.append( CR )
		.append( "-param_XXXX=固定値 を使用して、XSLTにパラメータを設定できます。"				).append( CR )
		.append( CR )
		.append( "それ以外では、org.opengion.fukurou.xml.XSLT で、入力ファイル情報の設定が可能に").append( CR )
		.append( "なっている為、内部情報を使用するかどうか -useFileInfo を指定できます。"		).append( CR )
		.append( "-useFileInfo=true とセットすると、以下の４項目が内部的にセットされます。"		).append( CR )
		.append( CR )
		.append( "入力ファイル(inXMLのフルパス)     : FILEPATH  (例: G:/temp/DOC10/query.jsp)"	).append( CR )
		.append( "入力親フォルダ(inXMLの親フォルダ) : ADDRESS   (例: DOC10)"					).append( CR )
		.append( "入力ファイル(inXMLのファイル名)   : FILENAME  (例: query.jsp)"				).append( CR )
		.append( "入力ファイル(inXMLの更新日付  )   : MODIFIED  (例: yyyyMMddHHmmss形式)"		).append( CR )
		.append( CR )
		.append( "xsl ファイルでは、xsl:param で宣言し、xsl:value-of で取り出します。"			).append( CR )
		.append( "<xsl:param name=\"ADDRESS\" select=\"\" /> と宣言しておき、必要な箇所で"		).append( CR )
		.append( "<xsl:value-of select=\"$ADDRESS\"     /> とすれば、取得できます。"			).append( CR )
		.append( CR )
		.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR )
		.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR )
		.append( "繋げてください。"																).append( CR )
		.append( CR ).append( CR )
		.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_XSLT().usage() );
	}
}
