/*
 * Copyright (c) 2017 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.fileexec;

import java.nio.file.Path;
import java.nio.file.PathMatcher;

import java.util.Set;
import java.util.Locale;
// import java.util.logging.Logger;
import java.util.concurrent.CopyOnWriteArraySet;

/**
 * PathMatcherSet は、ファイル監視を行うクラスで利用する、ファイルの選別(PathMatcher)を管理するクラスです。
 *
 *<pre>
 * PathMatcherオブジェクトを複数持っており（Set)それらが、その、判定によって、
 * イベントを起こすかどうか、フィルタリングします。
 *
 *</pre>
 * @og.rev 7.0.0.0 (2017/07/07) 新規作成
 *
 * @version  7.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.8,
 */
public class PathMatcherSet implements PathMatcher {
	private static final XLogger LOGGER= XLogger.getLogger( PathMatcherSet.class.getName() );		// ログ出力

	// パスの照合操作を行うPathMatcher のSetオブジェクト
	private final Set<PathMatcher> pathMchSet = new CopyOnWriteArraySet<>();		// 未設定のときは、すべてにマッチングさせる。(startメソッドで)

	/**
	 * デフォルトコンストラクター
	 *
	 */
	public PathMatcherSet() {
		super();
	}

	/**
	 * すべてのPathMatcherSet を、追加登録します。
	 *
	 * 引数が、null の場合は、登録しません。
	 *
	 * @param	pmSet パスの照合操作のパターン
	 * @return	このセットが変更された場合はtrue
	 */
	public boolean addAll( final PathMatcherSet pmSet ) {
		return pmSet != null && pathMchSet.addAll( pmSet.pathMchSet );
	}

	/**
	 * 内部の PathMatcherに、要素が含まれてい無い場合に、true を返します。
	 *
	 * @return	このセットに要素が1つも含まれていない場合はtrue
	 */
	public boolean isEmpty() {
		return pathMchSet.isEmpty();
	}

	/**
	 * すべての要素をセットから削除します。
	 *
	 */
	public void clear() {
		pathMchSet.clear();
	}

	/**
	 * PathMatcher を、追加登録します。
	 *
	 * 引数が、null の場合は、登録しません。
	 *
	 * @param	pathMch パスの照合操作のパターン
	 * @return	自分自身
	 * @see		java.nio.file.PathMatcher
	 * @see		#addStartsWith(String...)
	 * @see		#addEndsWith(String...)
	 */
	public PathMatcherSet addPathMatcher( final PathMatcher pathMch ) {
		if( pathMch != null ) {
			pathMchSet.add( pathMch );
		}

		return this;
	}

	/**
	 * 指定のパスが、指定の文字列と、先頭一致(startsWith) したパスのみ、有効とします。
	 *
	 * これは、#addPathMatcher(PathMatcher) の簡易指定版です。
	 * 指定の先頭一致(一般にはファイル名の先頭)のうち、ひとつでも一致すれば、true となります。
	 * 先頭文字列の判定には、大文字小文字の区別を行いません。
	 *
	 * @param	startKey パスの先頭一致のパターン
	 * @return	自分自身
	 * @see		#addPathMatcher(PathMatcher)
	 * @see		#addEndsWith(String...)
	 */
	public PathMatcherSet addStartsWith( final String... startKey ) {
		if( startKey != null ) {
			pathMchSet.add(
				path -> {
					// 大文字小文字の区別を行いません。
					final String fname = path.getFileName().toString().toUpperCase(Locale.JAPAN);
					for( final String key : startKey ) {
						if( key != null && fname.startsWith( key.toUpperCase(Locale.JAPAN) ) ) { return true; }
					}
					return false;
				}
			);
		}

		return this;
	}

	/**
	 * 指定のパスが、指定の文字列と、終端一致(endsWith) したパスのみ、有効とします。
	 *
	 * これは、#addPathMatcher(PathMatcher) の簡易指定版です。
	 * 指定の終端文字列(一般には拡張子)のうち、ひとつでも一致すれば、true となります。
	 * 指定しない場合(null)は、すべて許可されたことになります。
	 * 終端文字列の判定には、大文字小文字の区別を行いません。
	 *
	 * @param	endKey パスの終端一致のパターン
	 * @return	自分自身
	 * @see		#addPathMatcher(PathMatcher)
	 * @see		#addStartsWith(String...)
	 */
	public PathMatcherSet addEndsWith( final String... endKey ) {
		if( endKey != null ) {
			pathMchSet.add(
				path -> {
					// 大文字小文字の区別を行いません。
					final String fname = path.getFileName().toString().toUpperCase(Locale.JAPAN);
					for( final String key : endKey ) {
						if( key != null && fname.endsWith( key.toUpperCase(Locale.JAPAN) ) ) { return true; }
					}
					return false;
				}
			);
		}

		return this;
	}

	/**
	 * 指定されたパスがこのマッチャのパターンに一致するかどうかを示します。
	 *
	 * 内部の PathMatcher が、すべて true を返す場合のみ、true を返します。
	 * 未登録の場合は、true が返され、評価されません。
	 * これは、#allMatch( Path ) と同じ結果を返します。
	 *
	 * @param	path 照合するパス
	 * @return	パスがこのマッチャのパターンに一致した場合にのみtrue
	 * @see		#allMatch( Path )
	 */
	@Override
	public boolean matches( final Path path ) {
		return allMatch( path );
	}

	/**
	 * すべての要素が、条件を満たす場合にのみ、有効となります。
	 *
	 * 内部の PathMatcher が、すべて true を返す場合のみ、true を返します。
	 * 未登録の場合は、true が返され、評価されません。
	 * これは、#matches( Path ) と同じ結果を返します。
	 *
	 * @param	path 判定対象の Pathオブジェクト
	 * @return	内部の PathMatcher が、すべて true を返す場合のみ、true
	 * @see		#matches( Path )
	 */
	public boolean allMatch( final Path path ) {
		// stream().allMatch で、Collectionが未登録時も、true ですが、明示的に示しておきます。
		return pathMchSet.isEmpty() || pathMchSet.stream().allMatch( pMch -> pMch.matches( path ) );
	}

	/**
	 * いずれかの要素が、条件を満たす場合に、有効となります。
	 *
	 * 内部の PathMatcher の、いずれかが、 true を返す場合に、true を返します。
	 * 未登録の場合は、true が返され、評価されません。
	 * この動きは、Set#anyMatch(java.util.function.Predicate)とは異なりますので、ご注意ください。
	 *
	 * @param	path 判定対象の Pathオブジェクト
	 * @return	内部の PathMatcher の、いずれかが、 true を返す場合に、true
	 */
	public boolean anyMatch( final Path path ) {
		// stream().anyMatch の場合は、Collectionが未登録時は、false が返る為、明示的に処理が必要です。
		return pathMchSet.isEmpty() || pathMchSet.stream().anyMatch( pMch -> pMch.matches( path ) );
	}

	/**
	 * 一致する要素が、ひとつも存在しない場合に、有効となります。
	 *
	 * 内部の PathMatcher の要素のすべてに、false を返す場合に、true を返します。
	 * 未登録の場合は、true が返され、評価されません。
	 *
	 * @param	path 判定対象の Pathオブジェクト
	 * @return 内部の PathMatcher の要素のすべてに、false を返す場合に、true
	 */
	public boolean noneMatch( final Path path ) {
		// stream().noneMatch で、Collectionが未登録時も、true ですが、明示的に示しておきます。
		return pathMchSet.isEmpty() || pathMchSet.stream().noneMatch( pMch -> pMch.matches( path ) );
	}

	/** main メソッドから呼ばれる ヘルプメッセージです。 {@value}  */
	public static final String USAGE = "Usage: java jp.euromap.eu63.util.PathMatcherSet [dir] [-start ････] [-end ････]" ;

	/**
	 * 引数に監視対象のフォルダをフィルターします。
	 *
	 * {@value #USAGE}
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		// ********** 【整合性チェック】 **********
		if( args.length < 1 ) {
			System.out.println( USAGE );
			return;
		}

		// ********** 【引数定義】 **********
		Path	sPath	= new java.io.File( "." ).toPath();		// スキャンパス の初期値

		String	startsWith	= null;		// ファイル先頭文字列の一致キー
		String	endsWith	= null;		// ファイル終端文字列の一致キー

		// ********** 【引数処理】 **********
		for( int i=0; i<args.length; i++ ) {
			final String arg = args[i];

			if(      "-help" .equalsIgnoreCase( arg ) ) { System.out.println( USAGE ); return ; }
			else if( "-start".equalsIgnoreCase( arg ) ) { startsWith = args[++i]; }		// 記号を見つけたら、次の引数をセットします。
			else if( "-end"  .equalsIgnoreCase( arg ) ) { endsWith   = args[++i]; }		// 記号を見つけたら、次の引数をセットします。
			else {
				sPath = new java.io.File( arg ).toPath();
			}
		}

		// ********** 【本体処理】 **********
		final PathMatcherSet pmSet = new PathMatcherSet();
		pmSet.addStartsWith( startsWith );
		pmSet.addEndsWith( endsWith );

		try {
			java.nio.file.Files.walk( sPath )
								.filter(  path -> pmSet.allMatch( path ) )
								.forEach( path -> System.out.println( path ) );
		}
		catch( final  java.io.IOException ex ) {
			ex.printStackTrace();
		}
	}
}
