/*
 * Copyright (c) 2017 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.fileexec;

import java.util.ResourceBundle;
import java.util.PropertyResourceBundle;
import java.util.Locale;
import java.util.Arrays;
import java.text.MessageFormat;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.net.URL;
import java.net.URLConnection;

// import java.util.logging.Logger;
// import java.util.logging.Level;
// import java.util.function.Supplier;						// 6.8.1.0 (2017/07/14)
import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * MsgUtilは、共通的に使用されるリソースからメッセージを作成する、ユーティリティークラスです。
 *
 *<pre>
 * 現状は、{@value F_BS_NM}.properties ファイルをリソースとして使用します。
 * このリソースファイルを、各言語別に作成することで、アプリケーションのメッセージを国際化できます。
 * 通常のリソース変換以外に、キーワードと引数で、RuntimeException を返す簡易メソッドも提供します。
 *
 *</pre>
 * @og.rev 7.0.0.0 (2017/07/07) 新規作成
 *
 * @version  7.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.8,
 */
public final class MsgUtil {
	private static final XLogger LOGGER= XLogger.getLogger( MsgUtil.class.getName() );		// ログ出力

	/** 初期設定されているリソースバンドルのbaseName {@value} */
	public static final String F_BS_NM = "org.opengion.fukurou.message" ;

	private static final int	BUFFER_MIDDLE    = 200 ;
	private static final int	STACKTRACE_COUNT = 5 ;

	/**
	 * デフォルトコンストラクターをprivateにして、
	 * オブジェクトの生成をさせないようにする。
	 */
	private MsgUtil() {}

	/**
	 * "jp.euromap.eu63.message" の、Locale.getDefault() リソースから取得するメッセージを文字列で返します。
	 *
	 * id と引数を受け取り、ResourceBundle と、MessageFormat.format で加工した
	 * 文字列を返します。
	 * baseName は、F_BS_NM で、Locale に、Locale.getDefault() を指定したメッセージを作成します。
	 *
	 * @og.rev 6.4.3.1 (2016/02/12) 新規追加
	 *
	 * @param id	リソースのキーとなるID。
	 * @param args	リソースを、MessageFormat.format で加工する場合の引数。
	 *
	 * @return MessageFormat.formatで加工された文字列
	 * @see		#F_BS_NM
	 */
	public static String getMsg( final String id , final Object... args ) {

		// リソースバンドルのすべてがキャッシュに格納される・・・はず。
		final ResourceBundle resource = ResourceBundle.getBundle( F_BS_NM , Locale.getDefault() , UTF8_CONTROL );

		try {
			return id + ":" + MessageFormat.format( resource.getString( id ) , args );
		}
		catch( final RuntimeException ex ) {
			final String errMsg = id + "[" + Arrays.toString ( args ) + "]" ;
			LOGGER.warning( ex , () -> errMsg );
			return errMsg ;
		}
	}

	/**
	 * メッセージを作成して、RuntimeExceptionの引数にセットして、throw します。
	 *
	 * @og.rev 6.4.3.1 (2016/02/12) 新規追加
	 *
	 * @param id	リソースのキーとなるID。
	 * @param args	リソースを、MessageFormat.format で加工する場合の引数。
	 * @return		メッセージを書き込んだ、RuntimeException
	 *
	 * @see		#getMsg( String,Object... )
	 * @see		#throwException( Throwable,String,Object... )
	 */
	public static RuntimeException throwException( final String id , final Object... args ) {
		return throwException( null , id , args );
	}

	/**
	 * メッセージを作成して、RuntimeExceptionの引数にセットして、throw します。
	 *
	 * @og.rev 6.4.3.1 (2016/02/12) 新規追加
	 *
	 * @param th	発生元のThrowable( null値は許容されます )
	 * @param id	リソースのキーとなるID。
	 * @param args	リソースを、MessageFormat.format で加工する場合の引数。
	 * @return		メッセージを書き込んだ、RuntimeException
	 *
	 * @see		#getMsg( String,Object... )
	 * @see		#throwException( String,Object... )
	 */
	public static RuntimeException throwException( final Throwable th , final String id , final Object... args ) {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( getMsg( id , args ) );

		if( th != null ) {
			buf.append( "\n\t" ).append( th.getMessage() );
		}

		// ラムダ式で、Exception が throw された場合、上位にアップされない。(非検査例外(RuntimeException系)なら、スローできる・・・はず)
		// 原因がわかるまで、とりあえず、printStackTrace しておきます。
		final String errMsg = buf.toString();
		final RuntimeException ex = new RuntimeException( errMsg , th );
		LOGGER.warning( ex , () -> errMsg );
		return ex;
	}

	/**
	 * エラーメッセージを作成して、LOGGER で出力します。
	 *
	 * @og.rev 6.4.3.1 (2016/02/12) 新規追加
	 *
	 * @param id	リソースのキーとなるID。
	 * @param args	リソースを、MessageFormat.format で加工する場合の引数。
	 * @return 作成されたエラーメッセージ文字列
	 *
	 * @see		#getMsg( String,Object... )
	 */
	public static String errPrintln( final String id , final Object... args ) {
		return errPrintln( null , id , args );
	}

	/**
	 * Throwable付きのエラーメッセージを作成して、LOGGER で出力します。
	 *
	 * @og.rev 6.4.3.1 (2016/02/12) 新規追加
	 *
	 * @param th	発生元のThrowable( null値は許容されます )
	 * @param id	リソースのキーとなるID。
	 * @param args	リソースを、MessageFormat.format で加工する場合の引数。
	 * @return 作成されたエラーメッセージ文字列
	 *
	 * @see		#getMsg( String,Object... )
	 */
	public static String errPrintln( final Throwable th , final String id , final Object... args ) {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( getMsg( id , args ) );

		if( th != null ) {
			buf.append( "\n\t" ).append( th.getMessage() );

			int cnt = 0;
			for( final StackTraceElement stEle : th.getStackTrace() ) {
				final String clnNm = stEle.getClassName();

				if( clnNm.contains( "MsgUtil" ) ) { continue; }

				if( clnNm.contains( "jp.euromap.eu63" ) || cnt < STACKTRACE_COUNT ) {
					buf.append( "\n\t" ).append( stEle.toString() );
					cnt++;
				}
			}
		}

		LOGGER.warning( () -> buf.toString() );

		return buf.toString();
	}

//	/**
//	 * 汎用的な、debug に使用する Logger 処理
//	 *
//	 * @og.rev 6.8.1.0 (2017/07/14) 新規追加
//	 *
//	 * @param logger		使用するLoggerオブジェクト
//	 * @param msgSupplier	 呼び出されると、目的のログ・メッセージを生成する関数
//	 *
//	 * @see		Logger#log( Level,Throwable,Supplier<String> )
//	 */
//	public static void debug( final Logger logger , final Supplier<String> msgSupplier ) {
//		logger.log( DEBUG , msgSupplier );
//	}

	/**
	 * ResourceBundle.Controlは、バンドル・ロード処理中にResourceBundle.getBundleファクトリによって呼び出される一連のコールバック・メソッドを定義します。
	 *
	 * @og.rev 6.4.3.1 (2016/02/12) 新規追加
	 */
	private static final ResourceBundle.Control UTF8_CONTROL = new ResourceBundle.Control() {
		/**
		 * 指定された形式とロケールを持つ指定されたバンドル名のリソース・バンドルを、指定されたクラス・ローダーを必要に応じて使用してインスタンス化します。
		 *
		 * 指定されたパラメータに対応する使用可能なリソース・バンドルが存在しない場合、このメソッドはnullを返します。
		 * 予想外のエラーが発生したためにリソース・バンドルのインスタンス化が行えない場合には、単純にnullを返す代わりに、
		 * ErrorまたはExceptionをスローすることでエラーを報告する必要があります。 
		 * reloadフラグがtrueの場合、それは、以前にロードされたリソース・バンドルの有効期限が切れたためにこのメソッドが呼び出されたことを示します。 
		 *
		 * @og.rev 6.4.3.1 (2016/02/12) 新規追加
		 *
		 * @param baseName	リソース・バンドルの基底バンドル名。完全指定クラス名
		 * @param locale	リソース・バンドルのインスタンス化対象となるロケール
		 * @param format	ロードされるリソース・バンドルの形式
		 * @param loader	バンドルをロードするために使用するClassLoader
		 * @param reload	バンドルの再ロードを示すフラグ。有効期限の切れたリソース・バンドルを再ロードする場合はtrue、それ以外の場合はfalse
		 *
		 * @return ResourceBundle.Controオブジェクト
		 *
		 * @throws NullPointerException			bundleName、locale、format、またはloaderがnullの場合、またはtoBundleNameからnullが返された場合
		 * @throws IllegalArgumentException		formatが不明である場合、または指定されたパラメータに対して見つかったリソースに不正なデータが含まれている場合。
		 * @throws ClassCastException			ロードされたクラスをResourceBundleにキャストできない場合
		 * @throws IllegalAccessException		クラスまたはその引数なしのコンストラクタにアクセスできない場合。
		 * @throws InstantiationException		クラスのインスタンス化が何かほかの理由で失敗する場合。
		 * @throws ExceptionInInitializerError	このメソッドによる初期化に失敗した場合。
		 * @throws SecurityException			セキュリティ・マネージャが存在し、新しいインスタンスの作成が拒否された場合。詳細は、Class.newInstance()を参照してください。
		 * @throws IOException					何らかの入出力操作を使ってリソースを読み取る際にエラーが発生した場合
		 */
		@Override
		public ResourceBundle newBundle( final String baseName, 
										 final Locale locale, 
										 final String format, 
										 final ClassLoader loader, 
										 final boolean reload ) throws IllegalAccessException, InstantiationException, IOException {
			// The below is a copy of the default implementation.
			final String bundleName   = toBundleName( baseName , locale );
			final String resourceName = toResourceName( bundleName, "properties" );
			InputStream stream = null;
			if( reload ) {
				final URL url = loader.getResource( resourceName );
				if( url != null ) {
					final URLConnection urlConn = url.openConnection();
					if( urlConn != null ) {
						urlConn.setUseCaches( false );
						stream = urlConn.getInputStream();
					}
				}
			} else {
				stream = loader.getResourceAsStream( resourceName );
			}

			ResourceBundle bundle = null;
			if( stream != null ) {
				try {
					// Only this line is changed to make it to read properties files as UTF-8.
					bundle = new PropertyResourceBundle( new BufferedReader( new InputStreamReader( stream,UTF_8 ) ) );
				} finally {
					stream.close();
				}
			}
			return bundle;
		}
	};

	/**
	 * リソース一覧を表示する main メソッドです。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		// ResourceBundle.getBundle は、キャッシュされる・・・はず。
		final ResourceBundle resource = ResourceBundle.getBundle( F_BS_NM , Locale.getDefault() , UTF8_CONTROL );		// リソースバンドルのすべてがキャッシュに格納されます。

		for( final String key : new java.util.TreeSet<String>( resource.keySet() ) ) {
			System.out.println( key + ":\t" + resource.getString( key ) );
		}
	}
}
