/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.io.File;
import java.io.PrintWriter;
import java.util.Locale;
// import java.util.Map;

import org.opengion.fukurou.system.OgBuilder ;				// 6.4.4.1 (2016/03/18)
import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.db.Transaction;					// 5.5.2.6 (2012/05/25)
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FixLengthData;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableFilter_TABLE は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、テーブル一覧の検索結果より、GF05 のテーブルカラム定義テーブルから
 * 必要な情報を取得し、テーブル作成スクリプトを作成します。
 * 出力ファイルは、テーブル名＋"S.sql" という命名規則で作成します。
 * 検索では、(SYSTEM_ID,TBLSYU,TABLE_NAME,NAME_JA,TABLESPACE_NAME,INITIAL_EXTENT,COMMENTS)
 * の項目を取得する必要があります。
 *
 * 6.1.0.0 (2014/12/26) より、NEXT_EXTENT は使用しなくなりました。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       DIR : {&#064;BASE_DIR}/sql/install/01_TABLE ;    出力ファイルの基準フォルダ(必須)
 *       XML : false ;                                    XML出力を行うかどうか[true/false]を指定します(初期値:false)。
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select SYSTEM_ID,TBLSYU,TABLE_NAME,NAME_JA,TABLESPACE_NAME,INITIAL_EXTENT,COMMENTS from GF02
 * 
 *      ① &lt;og:tableFilter classId="TABLE" keys="DIR,XML" vals='"{&#064;BASE_DIR}/sql/install/01_TABLE,"' /&gt;
 *
 *      ② &lt;og:tableFilter classId="TABLE" &gt;
 *               {
 *                   DIR : {&#064;BASE_DIR}/sql/install/01_TABLE ;
 *                   XML : false ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 4.0.0.0 (2005/08/31) 新規作成
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_TABLE extends AbstractTableFilter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.5.0.1 (2016/10/21)" ;

	// 6.1.0.0 (2014/12/26) NEXT_EXTENT は、使いません。
	private static final String[] DBKEY = {"SYSTEM_ID","TBLSYU","TABLE_NAME","NAME_JA",
							"TABLESPACE_NAME","INITIAL_EXTENT","COMMENTS" };

	// 5.1.1.0 (2009/12/01) データのアクセス用の配列番号のIDを private ⇒ protected にします。
	/** データのアクセス用の配列番号 {@value} */
	protected static final int SYSTEM_ID		= 0;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TBLSYU			= 1;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLE_NAME		= 2;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int NAME_JA			= 3;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLESPACE_NAME	= 4;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int INITIAL_EXTENT	= 5;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int COMMENTS			= 6;		// 6.1.0.0 (2014/12/26) NEXT_EXTENT は、使いません。

	private static final String GF05_SEL = "SELECT CLM,SEQNO,NAME_JA,CLS_NAME,USE_LENGTH,DATA_DEFAULT,NOT_NULL,'' AS OPTS"
											+ " FROM GF05"
											+ " WHERE SYSTEM_ID=? AND TBLSYU=? AND TABLE_NAME=?"
											+ " AND   FGJ='1'"
											+ " ORDER BY SEQNO" ;

	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_CLM			= 0;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_SEQNO		= 1;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_NAME_JA		= 2;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_CLS_NAME	= 3;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_USE_LENGTH	= 4;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_DATA_DEFAULT= 5;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_NOT_NULL	= 6;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_OPTIONS		= 7;

 //	private static final String ENCODE = "Windows-31J" ;
	private static final String ENCODE = "UTF-8" ; // 4.3.6.6 (2009/05/15)

	private static final String CMNT  = "************************************************************************" ;

	private static final int X = FixLengthData.X ;		// type 定数
	private static final int S = FixLengthData.S ;		// type 定数
	private static final int K = FixLengthData.K ;		// type 定数
	private static final int T = FixLengthData.T ;		// addLen 定数
	private static final int T2= FixLengthData.T2 ;		// addLen 定数

	/** 各種定数  */
	// 6.0.2.3 (2014/10/10) AbstractTableFilter へ移動

	/** XML形式かどうか */

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) keysMap を、サブクラスから設定させるように変更。
	 */
	public TableFilter_TABLE() {
		super();
		initSet( "DIR"	, "出力ファイルの基準フォルダ(必須)"						);
		initSet( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"	);
	}

//	/**
//	 * keys の整合性チェックを行うための初期設定を行います。
//	 *
//	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
//	 * @og.rev 6.4.1.1 (2016/01/16) keysMap を、サブクラスから設定させるため、廃止。
//	 *
//	 * @param	keysMap keys の整合性チェックを行うための Map
//	 */
//	@Override
//	protected void init( final Map<String,String> keysMap ) {
//		keysMap.put( "DIR"	, "出力ファイルの基準フォルダ(必須)"						);
//		keysMap.put( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"	);
//	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 4.3.7.0 (2009/06/01) トリガー、SEQUENCE作成機能、XML出力機能追加
	 * @og.rev 5.1.1.0 (2009/12/01) XML_START_TAG に、tableName をセットします。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 * @og.rev 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
	 * @og.rev 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュを作成します。
	 * @og.rev 6.3.7.0 (2015/09/04) AutoCloseableを使用したtry-with-resources構築に対応。
	 * @og.rev 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
	 *
	 * @return	実行結果のテーブルモデル
	 */
	public DBTableModel execute() {
//		final DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		isXml = StringUtil.nval( getValue( "XML" ), false );

		execEndTag = isXml ? CR + EXEC_END_TAG : ";" ;	// 6.0.2.3 (2014/10/10)

		final File dir = new File( getValue( "DIR" ) );
		if( ! dir.exists() && ! dir.mkdirs() ) {
			final String errMsg = "所定のフォルダが作成できませんでした。[" + dir + "]" ;
			// 4.3.4.4 (2009/01/01)
			throw new HybsSystemException( errMsg );
		}

		// カンマ,カラム,クラス,(,桁数,),初期値,NOT_NULL,拡張機能,コメント開始,行番号,名称,コメント終了
		final int[] addLen = new int[] { 0,T,0,0,0,T2,T,T,T2,0,1,T,0 };	// 各データ間のスペース
		final int[] type   = new int[] { X,X,X,X,S,X, X,X,X, X,S,K,X };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		final FixLengthData fixData = new FixLengthData( addLen,type );

//		String[] data  = null;						// 6.3.9.1 (2015/11/27)
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
		final Transaction tran = getTransaction();	// 5.5.2.6 (2012/05/25)
		final int[] clmNo = getTableColumnNo( DBKEY );
		final int rowCnt  = table.getRowCount();
		for( int row=0; row<rowCnt; row++ ) {
			// 6.3.7.0 (2015/09/04) AutoCloseableを使用したtry-with-resources構築に対応。
//			data  = table.getValues( row );
			final String[] data		= table.getValues( row );				// 6.3.9.1 (2015/11/27)
			final String systemId	= data[clmNo[SYSTEM_ID]];
			final String tblsyu		= data[clmNo[TBLSYU]];
			final String tableName	= data[clmNo[TABLE_NAME]];

//			String tableName = null;
//			try {
//				data  = table.getValues( row );
//				final String systemId	= data[clmNo[SYSTEM_ID]];
//				final String tblsyu	= data[clmNo[TBLSYU]];
//				tableName		= data[clmNo[TABLE_NAME]];

			try( final PrintWriter writer = FileUtil.getPrintWriter( new File( dir,tableName + ( isXml ? "S.xml" : "S.sql" ) ),ENCODE ) ) {

				if( isXml ) { writer.println( XML_START_TAG.replace( "xxx",tableName ) ); }		// 5.1.1.0 (2009/12/01) tableName をセット
				writer.print( makeHeadLine( clmNo,data ) );

				final String[] vals = new String[] { systemId,tblsyu,tableName };
				final String[][] gf05 = DBUtil.dbExecute( GF05_SEL,vals,tran );	// 5.1.9.0 (2010/08/01) Transaction 対応

				String uniqName = null;
				fixData.clear();
				// 値セット：まずは、最大長を求める必要がある。
				for( int i=0; i<gf05.length; i++ ) {
					final String[] outData = makeLineList( gf05[i],i==0 );
					fixData.addListData( outData );

					// 4.3.7.0 (2009/06/01)
					if( "UNIQ".equalsIgnoreCase( gf05[i][GF05_CLM] ) || "UNIQSEQ".equalsIgnoreCase( gf05[i][GF05_CLM] ) ) {
						uniqName = gf05[i][GF05_CLM].toUpperCase( Locale.JAPAN );
					}
				}
				// 固定長化：最大長であわせた文字列を出力します。
				for( int i=0; i<gf05.length; i++ ) {
					writer.println( fixData.getFixData( i ) );
				}
				writer.println( makeEndLine( clmNo,data ) );

				// 4.3.7.0 (2009/06/01) UNIQ項目のSEQとトリガーを作成
				if( uniqName != null ) {
					writer.println( makeUniqSeq( clmNo,data ) );
					writer.println( makeUniqTrig( clmNo,data, uniqName ) );
				}

				if( isXml ) { writer.println( XML_END_TAG ); }
//				writer.close();
			}
			catch( final RuntimeException ex ) {		// catch は、close() されてから呼ばれます。
				// 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
				final ErrorMessage errMessage = makeErrorMessage( "TableFilter_TABLE Error",ErrorMessage.NG )
						.addMessage( row+1,ErrorMessage.NG,"TABLE"
							, "TABLE=[" + tableName + "]"
							, StringUtil.array2csv( data )
						)
						.addMessage( ex );

				// BAT から呼び出す場合があるため、標準エラー出力にも情報を出しておきます。
				System.out.println( errMessage );

//				final ErrorMessage errMessage = makeErrorMessage( "TableFilter_TABLE Error",ErrorMessage.NG );
//				errMessage.addMessage( row+1,ErrorMessage.NG,"TABLE",ex.toString() );
//				errMessage.addMessage( row+1,ErrorMessage.NG,"TABLE",StringUtil.array2csv( data ) );
//				errMessage.addMessage( row+1,ErrorMessage.NG,"TABLE","TABLE=[" + tableName + "]" );
//				// BAT から呼び出す場合があるため、標準エラー出力にも情報を出しておきます。
//				System.out.println( errMessage );
			}
		}

		return table;
	}

	/**
	 * ヘッダー部分の処理を実行します。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
	 * @og.rev 5.6.6.2 (2013/07/19) EXEC_START_TAG の付け忘れ
	 * @og.rev 6.4.4.2 (2016/04/01) StringBuilderの代わりに、OgBuilderを使用する。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ヘッダー部分の文字列
	 * @og.rtnNotNull
	 */
	protected String makeHeadLine( final int[] clmNo,final String[] data ) {
		final String TBL_NAME = data[clmNo[TABLE_NAME]];

		final String LINE1 = TBL_NAME + " ( " + data[clmNo[NAME_JA]] + " )" ;
		final String LINE2 = data[clmNo[COMMENTS]] ;
		final String LINE3 = "Created : " + HybsSystem.getDate() ;

		// 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
		final int[] addLen = new int[] { 0,0,0 };	// 各データ間のスペース
		final int[] type   = new int[] { X,K,X };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		final FixLengthData fixData = new FixLengthData( addLen,type );

		final String[][] outData = new String[][] {
			{ "/**",	CMNT ,	"**/" },
			{ "/* ",	LINE1,	" */" },
			{ "/* ",	LINE2,	" */" },
			{ "/* ",	LINE3,	" */" },
			{ "/**",	CMNT ,	"**/" },
		};

		fixData.addAllListData( outData );

		// 6.4.4.2 (2016/04/01)
		final OgBuilder buf = new OgBuilder();
		fixData.getAllFixData( buf.getBuilder() );		// OgBuilder の内部 Builder に、fixData のデータを書き込む。
		return buf.appendIfCR( isXml , EXEC_START_TAG )
					.appendCR( "CREATE TABLE " , TBL_NAME , " (" )
					.toString();

//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
//		fixData.getAllFixData( buf );
//
//		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }			// 5.6.6.2 (2013/07/19) 出力忘れ
//
//		buf.append( "CREATE TABLE " ).append( TBL_NAME ).append( " (" ).append( CR );
//
//		return buf.toString();
	}

	/**
	 * 各行部分(カラム定義)の処理を実行します。
	 * カンマ,カラム,クラス,(,桁数,),初期値,NOT_NULL,拡張機能,コメント開始,行番号,名称,コメント終了
	 * の順に配列にセットします。
	 *
	 * @og.rev 5.5.1.9 (2012/04/18) useLen.length=0対応
	 * @og.rev 5.9.13.2 (2016/10/28) シングルクォート対応
	 *
	 * @param	data	１行分のデータ配列
	 * @param	first	最初の行かどうか[true:最初/false:それ以降]
	 *
	 * @return	各行部分(カラム定義)配列
	 */
	protected String[] makeLineList( final String[] data,final boolean first ) {
		// カンマ,カラム,クラス(桁数),初期値,NOT_NULL,独自拡張,行番号,名称,終了
//		String[] outData = new String[13];
		final String[] outData = new String[13];				// 6.3.9.1 (2015/11/27)
		final String clsName = data[GF05_CLS_NAME];

		outData[0] = first ? "   " : " , " ;					// 0:カンマ
		outData[1] = data[GF05_CLM] ;							// 1:カラム

		if( clsName.startsWith( "CLOB" ) || clsName.startsWith( "DATE" ) ) {
			data[GF05_USE_LENGTH] = null;
		}
		final String useLen = data[GF05_USE_LENGTH];
		if( useLen != null && ! useLen.equals( "0" ) && useLen.length() > 0 ) { // 5.5.1.9 (2012/04/18)
			outData[2] = clsName ;								// 2:クラス
			outData[3] = " ( " ;								// 3:(
			outData[4] = useLen ;								// 4:桁数
			outData[5] = " )" ;									// 5:)
		}
		else {
			outData[2] = clsName ;								// NUMBER型の桁数指定なしのケース
		}

		final String def = data[GF05_DATA_DEFAULT];
		if( def != null && def.length() > 0 ) {
//			final String comma = clsName.indexOf( "CHAR" ) >= 0 ? "'" : "" ;
			final String comma = clsName.indexOf( "CHAR" ) >= 0 && def.indexOf( "'" ) < 0 ? "'" : "" ;	// 5.9.13.2 (2016/10/28)
			outData[6] = "DEFAULT " + comma + def + comma ;		// 6:初期値
		}

		final String notNull = data[GF05_NOT_NULL];
		if( notNull != null && notNull.equals( "1" ) ) {
			outData[7] = "NOT NULL"	;							// 7:NOT_NULL
		}

		final String options = data[GF05_OPTIONS];
		if( options != null ) {
			outData[8] = options	;							// 8:拡張機能
		}

		final String nameJA = data[GF05_NAME_JA] ;					// 名称
		if( nameJA != null ) {
			outData[9]  = "/* " ;								// 9:コメント開始
			outData[10] = data[GF05_SEQNO] ;					// 10:行番号
			outData[11] = nameJA ;								// 11:名称
			outData[12] = "*/" ;								// 12:コメント終了
		}

		return outData ;
	}

	/**
	 * 定義の最後の部分の処理を実行します。
	 *
	 * 6.1.0.0 (2014/12/26) より、
	 *   １．TABLESPACE_NAME を指定しない場合は、TABLESPACE 句を出力しません。
	 *   ２．INITIAL_EXTENT を 0 で指定した場合は、STORAGE 句を出力しません。
	 *   ３．NEXT と PCTINCREASE は、出力しません。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュを作成します。
	 * @og.rev 6.1.0.0 (2014/12/26) TABLESPACE_NAME,INITIAL_EXTENT が未設定の場合、設定しません。
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	定義の最後の部分
	 * @og.rtnNotNull
	 */
	protected String makeEndLine( final int[] clmNo,final String[] data ) {

	 	// 6.1.0.0 (2014/12/26) TABLESPACE_NAME,INITIAL_EXTENT が未設定の場合、設定しません。
		final String tblSpcse = data[clmNo[TABLESPACE_NAME]] ;
		final String initExt  = data[clmNo[INITIAL_EXTENT]] ;

		return new OgBuilder()
				.append( ")" )
				.appendIf( !StringUtil.isNull( tblSpcse )
							, CR , "TABLESPACE " , tblSpcse )
				.appendIf( !StringUtil.isNull( initExt ) && initExt.charAt(0) != '0'
							, CR , "STORAGE( INITIAL " , initExt , "K )" )
				.append( execEndTag )
				.toString();

//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
//			.append( ')' );							// 6.0.2.5 (2014/10/31) char を append する。
//		if( !StringUtil.isNull( tblSpcse ) ) {
//			buf.append( CR ).append( "TABLESPACE " ).append( tblSpcse );
//		}
//		if( !StringUtil.isNull( initExt ) && initExt.charAt(0) != '0' ) {
//			buf.append( CR ).append( "STORAGE( INITIAL " ).append( initExt ).append( "K )" );
//		}
//		buf.append( execEndTag );
//
//		return buf.toString();
	}

	/**
	 * ユニークシーケンスの作成処理を実行します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) シーケンス名を[TABLE_NAME]S00に変更
	 * @og.rev 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュを作成します。
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ユニークシーケンス
	 * @og.rtnNotNull
	 */
	protected String makeUniqSeq( final int[] clmNo,final String[] data ) {
		return new OgBuilder()
				.appendCR()
				.appendIfCR( isXml , EXEC_START_TAG )
				.appendCR( "CREATE SEQUENCE " , data[clmNo[TABLE_NAME]] , "S00 " )
				.append( "  INCREMENT BY 1 START WITH 1 MAXVALUE 999999999 CYCLE NOCACHE" )
				.append( execEndTag )
				.toString();

//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
//
//		buf.append( CR );
//		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }
//
//		// 5.1.9.0 (2010/08/01) シーケンス名を[TABLE_NAME]S00に変更
//		buf.append( "CREATE SEQUENCE " ).append( data[clmNo[TABLE_NAME]] ).append( "S00 " ).append( CR )
//			.append( "  INCREMENT BY 1 START WITH 1 MAXVALUE 999999999 CYCLE NOCACHE" )
//			// 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュを使用
//			.append( execEndTag );
//
//		return buf.toString();
	}

	/**
	 * ユニークシーケンスと関連付けるトリガの作成処理を実行します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) トリガー名を[TABLE_NAME]T00に変更
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 * @param   uniqName	ユニークトリガ名
	 *
	 * @return	ユニークシーケンスと関連付けるトリガ
	 * @og.rtnNotNull
	 */
	protected String makeUniqTrig( final int[] clmNo,final String[] data, final String uniqName ) {
		final String TBL_NAME = data[clmNo[TABLE_NAME]] ;

		return new OgBuilder()
				.appendCR()
				.appendIfCR( isXml , EXEC_START_TAG )
				.appendCR( "CREATE OR REPLACE TRIGGER "	, TBL_NAME , "T00 " )
				.appendCR( "  BEFORE INSERT ON "		, TBL_NAME )
				.appendCR( "  FOR EACH ROW " )
				.appendCR( "  BEGIN " )
				.append( "    SELECT "					, TBL_NAME , "S00.NEXTVAL INTO :NEW." )
				.appendCR( uniqName , " FROM DUAL; " )
				.appendCR( "  END; " )
				.appendCase( isXml , EXEC_END_TAG , "/" )		// isXmlで選択されます。true/false
				.toString();

//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
//
//		buf.append( CR );
//		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }
//
//		// 5.1.9.0 (2010/08/01) トリガー名を[TABLE_NAME]T00に変更
//		buf.append( "CREATE OR REPLACE TRIGGER " ).append( TBL_NAME ).append( "T00 " ).append( CR )
//			.append( "  BEFORE INSERT ON ").append( TBL_NAME ).append( CR )
//			.append( "  FOR EACH ROW " ).append( CR )
//			.append( "  BEGIN " ).append( CR )
//		// 5.1.9.0 (2010/08/01) シーケンス名を[TABLE_NAME]S00に変更
//			.append( "    SELECT " ).append( TBL_NAME ).append( "S00.NEXTVAL INTO :NEW." )
//			.append( uniqName ).append( " FROM DUAL; " ).append( CR )
//			.append( "  END; " ).append( CR );
//
//		if( isXml )	{ buf.append( EXEC_END_TAG ); }		// ここはキャッシュではなく、EXEC_END_TAG を使う。
//		else		{ buf.append( '/' ); }				// 6.0.2.5 (2014/10/31) char を append する。
//
//		return buf.toString();
	}
}
