/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.File;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.report2.DocConverter_OOO;

/**
 * ドキュメントの変換・マージを行うタグです。
 *
 * 変換を行うことのできる入出力のフォーマット以下の通りです。
 *
 * [対応フォーマット]
 *  入力[Calc(ods)   ,Excel(xls)     ] ⇒ 出力[Calc(ods)   ,Excel(xls)     ,PDF]
 *  入力[Writer(odt) ,Word(doc)      ] ⇒ 出力[Writer(odt) ,Word(doc)      ,PDF]
 *  入力[Impress(odp),PowerPoint(ppt)] ⇒ 出力[Impress(odp),PowerPoint(ppt),PDF]
 *
 * 入出力のフォーマットは、入出力ファイルの拡張子により自動判別されます。
 * また、入出力ファイル名が同じ場合は何も処理されません。
 *
 * 入力ファイルを、CSV形式で複数指定することで、複数の入力ファイルをマージして
 * 出力することもできます。
 *
 * ※2 現状は、ファイルのマージは、入力ファイルがExcelまたはCalcの場合のみ対応しています。
 * ※1 この機能を利用するには、OpenOfficeが正しく設定されている必要があります。
 *
 * @og.formSample
 * ●形式：&lt;og:docConvert fileURL="…" inputFile="…" outputFile="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:docConvert
 *       fileURL            【TAG】操作するファイルのディレクトリを指定します (初期値:FILE_URL[=filetemp/])
 *       inputFile        ○【TAG】入力ファイル名を指定します(必須)。
 *       outputFile       ○【TAG】出力ファイル名を指定します(必須)。
 *       delInput           【TAG】ドキュメント変換後、元のファイルを削除するかどうかを指定します(初期値:false[=削除しない])
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    ・Calc(ods)ファイルをPDFに変換
 *        &lt;og:docConvert inputFile="temp.ods" outputFile="out.pdf" /&gt;
 *
 *    ・Excel(xls)ファイルを連結
 *        &lt;og:docConvert inputFile="temp1.xls,temp2.xls" outputFile="out.xls" /&gt;
 *
 * @og.group その他部品
 *
 * @version  4.0
 * @author	 Hiroki Nakamura
 * @since    JDK5.0,
 */
public class DocConvertTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.1 (2016/02/05)" ;
	private static final long serialVersionUID = 642120160205L ;

	private String	fileURL 	= HybsSystem.sys( "FILE_URL" );
	private String[]inputFile	;
	private String	outputFile	;
	private boolean	delInput	;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public DocConvertTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( SKIP_BODY )
	 */
	@Override
	public int doStartTag() {
		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		// 出力ファイルで拡張子だけが指定された場合は、入力ファイル名を付加する
		if( outputFile.indexOf( '.' ) < 0 ) {
			final int inSufIdx = inputFile[0].lastIndexOf( '.' );
			if( inSufIdx >= 0 ) {
				outputFile = inputFile[0].substring( 0, inSufIdx ) + "." + outputFile;
			}

		}

		final String directory = HybsSystem.url2dir( fileURL );
		String[] input = new String[inputFile.length];
		for( int i=0; i<input.length; i++ ) {
			input[i] = StringUtil.urlAppend( directory,inputFile[i] );
		}
		final String output = StringUtil.urlAppend( directory,outputFile );

		// ドキュメント変換
		DocConverter_OOO.convert( input, output );

		if( delInput ) {
			for( int i=0; i<input.length; i++ ) {
				if( !( new File( input[i] ) ).delete() ) {
					System.err.println( "入力ファイルの削除に失敗しました。file=[" + input[i] + "]" );
				}
			}
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL 	= HybsSystem.sys( "FILE_URL" );
		inputFile	= null;
		outputFile	= null;
		delInput	= false;
	}

	/**
	 * 【TAG】操作するファイルのディレクトリを指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリのファイルを操作します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * (初期値:システム定数のFILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に、"/" を追加する処理を廃止。
	 *
	 * @param	url ファイルURL
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
//			final char ch = furl.charAt( furl.length()-1 );
//			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】入力ファイル名を指定します。
	 *
	 * @og.tag
	 * 入力ファイル名を指定します。
	 *
	 * 入力ファイルは、CSV形式で複数指定することができます。
	 * この場合、複数の入力ファイルをマージして出力を行います。
	 *
	 * ※現状は、ファイルのマージは、入力ファイルがExcelまたはCalcの場合のみ対応しています。
	 *   また、マージを行う場合、印刷範囲など、ドキュメントに関連する情報は、1つ目のファイルの
	 *   情報が使用されます。
	 *
	 * @param	fname 入力ファイル名
	 */
	public void setInputFile( final String fname ) {
		inputFile = getCSVParameter( fname );
	}

	/**
	 * 【TAG】出力ファイル名を指定します。
	 *
	 * @og.tag
	 * 出力ファイル名を指定します。
	 * 出力ファイルには、拡張子のみ(xls,ods等)を指定することもできます。
	 * この場合、出力ファイル名は、入力ファイル名と同じになります。(拡張子のみが変換される)
	 *
	 * @param	fname 出力ファイル名
	 */
	public void setOutputFile( final String fname ) {
		outputFile = nval( getRequestParameter( fname ),outputFile );
	}

	/**
	 * 【TAG】ドキュメント変換後、元のファイルを削除するかどうかを指定します(初期値:false[=削除しない])。
	 *
	 * @og.tag
	 * ドキュメント変換後、元のファイルを削除するかどうかを指定します。
	 * 初期値は、false(削除しない)です。
	 *
	 * @param	flag 元ファイルの削除 [true:する/false:しない]
	 */
	public void setDelInput( final String flag ) {
		delInput = nval( getRequestParameter( flag ),delInput );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "fileURL"		,fileURL	)
				.println( "inputFile"	,inputFile		)
				.println( "outputFile"	,outputFile		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
