/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractDBType;

/**
 * DATA_DEFAULT カラムで、内容の整合性を整えます。
 *
 * DATA_DEFAULT(初期値)カラムは、データベース上に設定されていますが、
 * DB定義スクリプトや、データベースの種類によって、本来必要な形式で
 * 取得できないときがあります。
 * ここでは、初期値の後ろに コメントがある場合は、削除し、
 * データそのものに、シングルクオートをはずします。
 * 定義情報出力時には、カラムの属性(数字型、文字型)
 * また、postgreSQL 対応として、::属性情報が入るので、削除します。
 * シーケンス使用時にも、nextval が自動的にセットされますが、
 * 削除します。
 *
 * このクラスは、valueAction メソッドで、action="VALSET" でのみ
 * 動作します。
 *
 * @og.group データ属性
 * @og.rev 5.1.3.0 (2010/02/01) 新規作成
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_DD extends AbstractDBType {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public DBType_DD() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * action で指定されたコマンドを実行して、値の変換を行います。
	 * oldValue(旧データ)は、元のDBTableModelに設定されていた値です。通常は、
	 * この値を使用してカラム毎に変換を行います。newValue(新データ)は、引数で
	 * 指定された新しい値です。この値には、パラメータを指定して変換方法を
	 * 制御することも可能です。
	 * 指定のアクションがカラムで処理できない場合は、エラーになります。
	 *
	 * @og.rev 5.1.3.0 (2010/02/01) 新規追加
	 * @og.rev 5.7.6.2 (2014/05/16) -- コメント対応。処理の全面見直し
	 * @og.rev 5.7.6.3 (2014/05/23) 処理対象以外は、newValue を返すだけ。
	 *
	 * @param   action アクションコマンド
	 * @param   oldValue 入力データ(旧データ)
	 * @param   newValue 入力データ(新データ)
	 *
	 * @return	実行後のデータ
	 */
	@Override
	public String valueAction( final String action,final String oldValue,final String newValue ) {
		if( "VALSET".equals( action ) && oldValue != null ) {
			String val = oldValue.trim() ;

			if( val.length() > 0 ) {
				// 共通：初期値に /* */ コメントが入ることがあるので、削除
				int pos = val.indexOf( "/*" );
				if( pos > 0 ) { val = val.substring( 0,pos ); }

				// 共通：初期値に -- コメントが入ることがあるので、削除
				pos = val.indexOf( "--" );
				if( pos > 0 ) { val = val.substring( 0,pos ); }

				// postgreSQL 対応：属性情報が入るので、削除
				pos = val.indexOf( "::" );
				if( pos > 0 ) { val = val.substring( 0,pos ); }

				// postgreSQL 対応：シーケンス使用時にも、nextval が自動的にセットされるので、削除
				pos = val.indexOf( "nextval" );
				if( pos > 0 ) { val = val.substring( 0,pos ); }

				val = val.trim();				// 一旦、trim します。

				// 共通：5.7.6.3 (2014/05/23) シングルクオートはずし(先頭か、末尾に付いている場合は、すべてのシングルクオートを外します。)
//				if( val.startsWith("'") || val.endsWith("'") ) {
				if( StringUtil.startsChar( val, '\'' ) || val.endsWith("'") ) {			// 6.4.1.1 (2016/01/16) １文字 String.startsWith
					val = StringUtil.replace( val,"'","" );
				}

			}

			return val ;
		}
		else {
			// 5.7.6.3 (2014/05/23) 処理対象以外は、newValue を返すだけ。
			return newValue;
		}
	}
}
