package org.opengion.plugin.column;

import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.StringFormat;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionCellEditor;					// 6.2.2.0 (2015/03/27)
import org.opengion.hayabusa.db.SelectionFactory;						// 5.7.3.0 (2014/02/07)

import static org.opengion.fukurou.util.StringUtil.isNull;				// 6.1.1.0 (2015/01/17)

/**
 * DATALIST_R エディターは、コードリソースから取得した値より、datalistを作成して
 * 入力候補となるデータリストを定義する編集用エディタークラスです。
 * datalist は、HTML5 から採用されたタグです。
 * 
 * 入力フィールドとdatalistタグの関係付は、カラムIDに、"カラムID.sel" で結びつけます。
 * 
 * &lt;input name="カラムID" list="カラムID.sel" /&gt;
 * &lt;div style="display:none;"&gt;
 *   &lt;datalist id="カラムID.sel"&gt;
 *     &lt;option value="KEY1"&gt;LABEL1&lt;/option&gt;
 *     &lt;option value="KEY2"&gt;LABEL2&lt;/option&gt;
 *     &lt;option value="KEY3"&gt;LABEL3&lt;/option&gt;
 *   &lt;/datalist&gt;
 * &lt;/div&gt;
 * 
 * divタグは、HTML5 非対応ブラウザを使用した場合、datalist の option がそのまま
 * テキストとして見えてしまうのを避けるためです。
 * 
 * @og.rev 5.9.18.1 (2017/03/24)
 * @og.group データ編集
 * 
 * @version 5.9.18.1
 * @author T.OTA
 * @since JDK6.0
 *
 */
public class Editor_DATALIST_R extends AbstractEditor implements SelectionCellEditor {
	private static final String	VERSION			= "6.8.1.5 (2017/09/08)";
	private static final String	DIV1			= "<div style=\"display:none;\">";
	private static final String	DIV2			= "</div>";

	/** セレクションオブジェクト */
	protected Selection			selection		;
	private Selection			bkSel			;		// Selection オブジェクトのキャッシュ機能
	private final boolean		seqFlag			;
	private final String		useSLabel		;		// 6.2.0.0 (2015/02/27) SLABEL 対応
	private final String		addKeyLabel		;		// 6.2.0.0 (2015/02/27) キー:ラベル形式
	private final TagBuffer		selTagBuffer	= new TagBuffer();

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 * 
	 * @og.rev 6.8.1.5 (2017/09/08) PMD refactoring. It is a good practice to call super() in a constructor
	 */
	public Editor_DATALIST_R() {
		super();					// 6.8.1.5 (2017/09/08) PMD refactoring. It is a good practice to call super() in a constructor

		selection	= null;
		seqFlag		= false;
		useSLabel	= "auto";		// 6.2.0.0 (2015/02/27) SLABEL 対応
		addKeyLabel	= null;			// 6.2.0.0 (2015/02/27) キー:ラベル形式
	}

	/**
	 * コンストラクター。
	 * 
	 * @param clm DBColumnオブジェクト
	 */
	private Editor_DATALIST_R( final DBColumn clm ) {
		super( clm );

		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );
		seqFlag		= "SEQ".equals( clm.getEditorParam() );
		useSLabel	= clm.getUseSLabel() ;		// 6.2.0.0 (2015/02/27) SLABEL 対応
		addKeyLabel = clm.getAddKeyLabel();		// 6.2.0.0 (2015/02/27) キー:ラベル形式

		final String disabled = clm.isWritable() ? null : "disabled";

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		final Attributes selAttri = new Attributes()
				.set( "disabled" ,disabled )
				.set( clm.getEditorAttributes() );			// #addAttributes( Attributes ) の代替え

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		selTagBuffer.add( XHTMLTag.selectAttri( selAttri ) )
					.add( selAttri.get( "optionAttributes" ) );				// 6.0.4.0 (2014/11/28)

		// 6.0.4.0 (2014/11/28) selection は、Column から取得するのではなく、Factory で作成する。
		selection = SelectionFactory.newSelection( "MENU",clm.getCodeData(),addKeyLabel );	// 6.2.0.0 (2015/02/27)
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 * 
	 * @param	clm DBColumnオブジェクト
	 * 
	 * @return CellEditorオブジェクト
	 */
	@Override
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_DATALIST_R( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 * 
	 * @param value 入力値
	 * 
	 * @return データの編集用文字列
	 */
	@Override
	public String getValue( final String value ) {
		final String newValue = StringFormat.getValue( value );				// 6.4.5.3 (2016/05/13) コロン区切りの先頭だけ

		// input タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String intag = new TagBuffer( "input" )
						.add( "name"	, name )
						.add( "id"		, name , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
						.add( "list"	, name + ".sel" )
						.add( "value"	, newValue )
						.add( "size"	, size1 )
						.add( tagBuffer.makeTag() )
						.makeTag();

		final boolean useSlbl = "true".equalsIgnoreCase( useSLabel );		// 6.2.0.0 (2015/02/27)

		// datalist タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String dltag = getOption(
					new TagBuffer( "datalist" )
						.add( "id"	, name + ".sel" )	// inputタグとの関係付けるためのキーワード
					, value
					, false								// キャッシュは使用しない。(つまり、null は返らない)
					, useSlbl							// 6.2.0.0 (2015/02/27) SLABEL 対応
				).makeTag() ;

		// DIV1 の display:none は、datalist の optionのBODY部が、HTML5 以外では表示されてしまうのを防ぐため。
		return intag + CR + DIV1 + dltag + DIV2 + CR;
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し、
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @param	row		行番号
	 * @param	value	入力値
	 * 
	 * @return	データ表示/編集用の文字列
	 */
	@Override
	public String getValue( final int row, final String value ) {
		final String name2 = name + HybsSystem.JOINT_STRING + row;

		// Selection オブジェクトのキャッシュ機能 (true:使用可能)
		final boolean useSelCache = value != null && value.indexOf( ':' ) < 0;

		final String listId = useSelCache ? name : name2; // キャッシュを使用する場合は、共通の name を使う。

		final String newValue = StringFormat.getValue( value );				// 6.4.5.3 (2016/05/13) コロン区切りの先頭だけ

		// input タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String intag = new TagBuffer( "input" )
						.add( "name"	, name2 )
						.add( "id"		, name2 , isNull( attributes.get( "id" ) ) )	// 4.3.7.2 (2009/06/15)
						.add( "list"	, listId + ".sel" )
						.add( "value"	, newValue )
						.add( "size"	, size2 )
						.add( tagBuffer.makeTag() )
						.makeTag( row,newValue );				// 6.4.5.3 (2016/05/13)

		final boolean useSlbl = "auto".equalsIgnoreCase( useSLabel ) || "true".equalsIgnoreCase( useSLabel );		// 6.2.0.0 (2015/02/27)

		// datalist タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final TagBuffer dltag = getOption(
					new TagBuffer( "datalist" )
						.add( "id"	, listId + ".sel" )			// inputタグとの関係付けるためのキーワード
					, value
					, useSelCache
					, useSlbl									// 6.2.0.0 (2015/02/27) SLABEL 対応
				);

		// キャッシュが効くと、getOption の戻り値は、null になる。
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		// 反転注意
		return dltag == null
					? intag + CR
//					: intag + CR + DIV1 + dltag.makeTag( row,value ) + DIV2 + CR ;
					: intag + CR + DIV1 + dltag.makeTag( row,newValue ) + DIV2 + CR ;		// 6.4.5.3 (2016/05/13)
	}

	/**
	 * コードリソースから取得した、選択肢(オプション)をTagBuffer に反映します。
	 * 
	 * 第３引数は、Selection オブジェクトのキャッシュ機能を使用するかどうかを指定します。
	 * true で、使用する事を前提に、チェックを行います。
	 * 
	 * @param	buf			タグ文字列のバッファー
	 * @param	value		選択されている値
	 * @param	useSelCache	Selection オブジェクトのキャッシュ
	 * @param   useSlbl ラベル(短)をベースとしたオプション表示を行うかどうか。
	 * 
	 * @return	buf	オプションタグ
	 */
//	private TagBuffer getOption( final TagBuffer buf, final String value, final boolean useSelCache ) {
	private TagBuffer getOption( final TagBuffer buf,final String value,final boolean useSelCache,final boolean useSlbl ) {

		if( useSelCache ) {
			if( selection == bkSel ) { return null; }
			bkSel = selection;
		}

//		buf.setBody( selection.getOption( value, seqFlag ) );

//		return buf;
		return buf.addBody( selection.getOption( value, false, useSlbl ) );	// 6.2.0.0 (2015/02/27) SLABEL 対応
	}
}
