/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.mail;

import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.HybsEntry ;
import org.opengion.fukurou.util.Argument ;
// import org.opengion.fukurou.util.FileString ;
import org.opengion.fukurou.util.FileUtil ;							// 6.4.5.2 (2016/05/06)
import static org.opengion.fukurou.system.HybsConst.CR;				// 6.3.8.0 (2015/09/11)

import java.util.Map;
import java.util.LinkedHashMap;

import java.util.logging.Logger;
import java.util.logging.Level;

/**
 * MailTransfer は、SMTPプロトコルによるメール送信プログラムです。
 *
 * 引数には、host,from,to,cc,bcc,replyTo,subject,message,charset,file,debug,help が指定できます。
 * 引数のプロパテイのキー部は、大文字・小文字が厳格に適用されますので、正確に記述願います。
 *
 * Usage: java org.opengion.fukurou.fukurou.mail.MailTransfer
 *        -host=メールサーバー(必須)
 *        -from=送信元アドレス(必須)
 *        -to=送信先アドレスをCSV形式で並べる(必須)
 *        -port=送信サーバーのポートを指定(初期値:-1)
 *        -useSSL=SSL接続するかどうか[true:する/false:しない]を指定(初期値:false:しない)
 *        -auth=認証方式(POP_BEFORE_SMTP , SMTP_AUTH のみ対応)
 *        -authPort=認証ポート(初期値:-1)
 *        -user=メールを取得するログインユーザー
 *        -passwd=メールを取得するログインパスワード
 *        -cc=関係者アドレスをCSV形式で並べる
 *        -bcc=隠し送信先アドレスをCSV形式で並べる
 *        -replyTo=返信先アドレスをCSV形式で並べる
 *        -subject=タイトル
 *        -message=本文を格納しているファイルのアドレス
 *        -msgEncode=本文を格納しているファイルの文字コード(初期値:java.nio.charset.Charset#defaultCharset())
 *        -charset=メール送信時のキャラクタセット [ISO-2022-JP / Windows-31J]
 *        -fileXX=添付ファイルのファイルパス( XX は複数登録時の添え字 )
 *        -debug=メイラーのデバッグモードを指定します。[false / true]
 *        -help=使用方法を出力して、終了します。
 *        -{&#064;XXXX}=YYYY  メッセージ本文の {&#064;XXXX} 文字列を、YYYY 文字列に変換します。
 *            {&#064;ARG.XXX} 予約文字変換 上記引数を割り当てます。
 *            {&#064;DATE.XXX} 予約文字変換 SimpleDateFormat 形式の文字を変換します。(日付、時刻等)
 *            {&#064;ENV.XXX}  予約文字変換 システムプロパティーの文字を変換します。(java -Dkey=value オプション)
 *
 * E-Mail で日本語を送信する場合、ISO-2022-JP(JISコード)化して、7bit で
 * エンコードして送信する必要がありますが、Windows系の特殊文字や、unicodeと
 * 文字のマッピングが異なる文字などが、文字化けします。
 * 対応方法としては、
 * １．Windows-31J + 8bit 送信
 * ２．ISO-2022-JP に独自変換 + 7bit 送信
 * の方法があります。
 * 今回、この２つの方法について、charset で指定できます。
 * 初期値は、ISO-2022-JP です。
 *
 * ※ 6.2.4.1 (2015/05/22) SMTP_AUTH 対応
 *
 * ※ 6.3.8.0 (2015/09/11)
 *    useSSL属性=true に設定すると、protocolに、smtps を使用します。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class MailTransfer {
	// 6.3.9.1 (2015/11/27) The Logger variable declaration does not contain the static and final modifiers(PMD)
//	private static Logger logger = Logger.getLogger( "org.opengion.fukurou.fukurou.mail.MailTransfer" );			// 4.3.3.5 (2008/11/08)
//	private static final Logger LOGGER = Logger.getLogger( "org.opengion.fukurou.fukurou.mail.MailTransfer" );		// 6.4.1.1 (2016/01/16) logger → LOGGER refactoring
	private static final Logger LOGGER = Logger.getLogger( MailTransfer.class.getName() );							// 6.8.1.0 (2017/07/14) ｸﾗｽ名を動的作成に変更

	/** メール送信時のデフォルトキャラクタセット 「={@value}」  */
	public static final String CHARSET = "ISO-2022-JP" ;

	/** メイラーの初期デバッグモード 「={@value}」  */
	public static final boolean DEBUG_MODE = false ;

	private Argument argment;									// 6.3.8.0 (2015/09/11)

	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> MUST_PROPARTY   ;	// 必須パラメータ					// 6.4.1.1 (2016/01/16) mustProparty   → MUST_PROPARTY   refactoring
	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> USABLE_PROPARTY ;	// 整合性チェックパラメータ			// 6.4.1.1 (2016/01/16) usableProparty → USABLE_PROPARTY refactoring

	static {
		MUST_PROPARTY = new LinkedHashMap<>();
		MUST_PROPARTY.put( "host",	"メールサーバー(必須)" );
		MUST_PROPARTY.put( "from",	"送信元アドレス(必須)" );
		MUST_PROPARTY.put( "to",		"送信先アドレスをCSV形式で並べる(必須)" );

		USABLE_PROPARTY = new LinkedHashMap<>();
		USABLE_PROPARTY.put( "port"		, "送信サーバーのポート" );					// 6.0.3.0 (2014/11/13) SMTP認証
		USABLE_PROPARTY.put( "useSSL"	, "SSL接続するかどうかを指定(初期値:false:しない)" );			// 6.3.8.0 (2015/09/11)
		USABLE_PROPARTY.put( "auth"		, "認証方式(POP_BEFORE_SMTP , SMTP_AUTHのみ対応)" );			// 6.2.4.1 (2015/05/22)
		USABLE_PROPARTY.put( "authPort"	, "認証ポート" );								// 6.0.3.0 (2014/11/13) SMTP認証
		USABLE_PROPARTY.put( "user"		, "認証ユーザー" );							// 6.0.3.0 (2014/11/13) SMTP認証
		USABLE_PROPARTY.put( "passwd"	, "認証パスワード" );							// 6.0.3.0 (2014/11/13) SMTP認証
		USABLE_PROPARTY.put( "cc"		, "関係者アドレスをCSV形式で並べる" );
		USABLE_PROPARTY.put( "bcc"		, "隠し送信先アドレスをCSV形式で並べる" );
		USABLE_PROPARTY.put( "replyTo"	, "返信先アドレスをCSV形式で並べる" );
		USABLE_PROPARTY.put( "subject"	, "タイトル" );
		USABLE_PROPARTY.put( "message"	, "本文を格納しているファイルのアドレス" );
		USABLE_PROPARTY.put( "msgEncode"	, "本文を格納しているファイルの文字コード [Windows-31J / UTF-8]" );
		USABLE_PROPARTY.put( "charset"	, "メール送信時のキャラクタセット [ISO-2022-JP / Windows-31J]" );
		USABLE_PROPARTY.put( "file"		, "添付ファイルのファイルパス( XX は複数登録時の添え字 )" );
		USABLE_PROPARTY.put( "{@"		, "{@XXXX}=YYYY  メッセージ本文の {@XXXX} 文字列を、YYYY 文字列に変換します。" );
		USABLE_PROPARTY.put( "{@ARG."	, "{@ARG.XXX} 予約文字変換 上記引数を割り当てます。" );
		USABLE_PROPARTY.put( "{@DATE."	, "{@DATE.XXX} 予約文字変換 SimpleDateFormat 形式の文字を変換します。(日付、時刻等)" );
		USABLE_PROPARTY.put( "{@ENV."	, "{@ENV.XXX} 予約文字変換 システムプロパティーの文字を変換します。(java -Dkey=value オプション)" );
		USABLE_PROPARTY.put( "debug"		, "メイラーのデバッグモードを指定します。[false / true]" );
		USABLE_PROPARTY.put( "help"		, "使用方法を出力して、終了します。" );
	}

	/**
	 * トランスファーを開始します。
	 *
	 * @og.rev 4.3.3.5 (2008/11/08) Argument オブジェクトへの引数を util → mail に訂正します。
	 * @og.rev 6.0.3.0 (2014/11/13) SMTP認証
	 * @og.rev 6.3.8.0 (2015/09/11) SSL接続するかどうかを指定するパラメータを追加します。
	 * @og.rev 6.3.8.0 (2015/09/11) message で指定されたファイルの
	 * @og.rev 6.4.5.1 (2016/04/28) FileStringのコンストラクター変更
	 * @og.rev 6.4.5.2 (2016/05/06) fukurou.util.FileString から、fukurou.util.FileUtil に移動。
	 *
	 * @param	args	引数配列(可変長引数)
	 */
	public void start( final String... args ) {

		// パラメータの解析、取得
		LOGGER.fine( "パラメータの解析、取得" );
//		final Argument argment = new Argument( "org.opengion.fukurou.mail.MailTransfer" );		// 4.3.3.5 (2008/11/08)
		argment = new Argument( "org.opengion.fukurou.mail.MailTransfer" );		// 4.3.3.5 (2008/11/08)
		argment.setMustProparty( MUST_PROPARTY );
		argment.setUsableProparty( USABLE_PROPARTY );

		argment.setArgument( args );

		// help パラメータが true に指定された場合の処理。
		if( argment.getProparty( "help",false ) ) {
			System.out.println( argment.toString() );
			return;
		}

		final String host		= argment.getProparty( "host" );
		final String chset		= argment.getProparty( "charset", CHARSET );

		final boolean useSSL	= argment.getProparty( "useSSL"	,false );		// 6.3.8.0 (2015/09/11) SSL接続

		final String port		= argment.getProparty( "port" );				// 6.0.3.0 (2014/11/13) SMTP認証
		final String auth		= argment.getProparty( "auth" );				// 6.0.3.0 (2014/11/13) SMTP認証
		final String authPort	= argment.getProparty( "authPort" );			// 6.0.3.0 (2014/11/13) SMTP認証
		final String user		= argment.getProparty( "user" );				// 6.0.3.0 (2014/11/13) SMTP認証
		final String passwd		= argment.getProparty( "passwd" );				// 6.0.3.0 (2014/11/13) SMTP認証

//		final MailTX mail = new MailTX( host,chset,port,auth,authPort,user,passwd );		// 6.0.3.0 (2014/11/13) SMTP認証
		final MailTX mail = new MailTX( host,chset,port,auth,authPort,user,passwd,useSSL );	// 6.3.8.0 (2015/09/11) SSL接続
		LOGGER.fine( "org.opengion.fukurou.mail.MailTX の設定" );

		mail.setFrom(							argment.getProparty( "from" ) );
		mail.setTo(		 StringUtil.csv2Array(	argment.getProparty( "to" ) ) );
		mail.setCc(		 StringUtil.csv2Array(	argment.getProparty( "cc" ) ) );
		mail.setBcc(	 StringUtil.csv2Array(	argment.getProparty( "bcc" ) ) );
		mail.setReplyTo( StringUtil.csv2Array(	argment.getProparty( "replyTo" ) ) );
		mail.setSubject( 						argment.getProparty( "subject" ) );
		mail.setDebug(							argment.getProparty( "debug", DEBUG_MODE ) );

		// message は、本文を格納しているファイル名が指定されています。
		final String msgFile   = argment.getProparty( "message" ) ;
		LOGGER.fine( "本文を格納しているファイルの取得。message=" + msgFile );
		if( msgFile != null ) {
			final String msgEncode = argment.getProparty( "msgEncode" ) ;
			// 6.4.5.1 (2016/04/28) FileStringのコンストラクター変更
//			final FileString fs = new FileString();
//			fs.setFilename( msgFile );
//			fs.setEncode( msgEncode );									// 6.3.8.0 (2015/09/11)
//			final FileString fs = new FileString( msgFile , msgEncode );
//			String message = fs.getValue();
			String message = FileUtil.getValue( msgFile , msgEncode );	// 6.4.5.2 (2016/05/06)

			// {@XXX} 変換は、Argument クラスの機能を使う。
			message = argment.changeParam( message );
			mail.setMessage( message );
		}

		// 添付ファイルは、fileXX 形式で、複数指定されています。
		LOGGER.fine( "添付ファイルを取り込みます。" );
		final HybsEntry[] entry = argment.getEntrys( "file" );
		String[] files = new String[entry.length];
		for( int i=0; i<files.length; i++ ) {
			files[i] = entry[i].getValue() ;
		}
		mail.setFilename( files );

		// 送信
		LOGGER.fine( "送信処理を行います。" );
		mail.sendmail();
	}

	/**
	 * main メソッドです。
	 *
	 * @param	args	引数配列
	 */
	public static void main ( final String[] args ) {
		final MailTransfer transfer = new MailTransfer();
		try {
			LOGGER.info( "メール送信処理を開始します  ---------------------------------------------" );
			transfer.start( args );
			LOGGER.info( "正常に終了しました。" );
		}
		catch( final Throwable th ) {
//			final String errMsg = "メール送信中に例外が発生しました。 " + th.getMessage() ;
			final String errMsg = "メール送信中に例外が発生しました。 "
							+ CR + transfer.argment
							+ CR + th.getMessage() ;
			LOGGER.log( Level.SEVERE,errMsg, th );
//			throw new OgRuntimeException( errMsg,th );	// 6.3.8.0 (2015/09/11) 削除します。
		}
	}
}
