/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.geom.Rectangle2D;
import java.awt.Color;

import org.jfree.chart.renderer.category.BarRenderer3D;
import org.jfree.chart.renderer.category.CategoryItemRendererState;
import org.jfree.chart.axis.CategoryAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.data.category.CategoryDataset;

/**
 * HybsBarRenderer は、org.jfree.chart.renderer.category.BarRenderer を
 * 拡張したカスタマイズクラスです。
 * これは、描画に対して、予め制限を設けて、処理速度の向上を図っています。
 *
 * @og.rev 6.0.2.2 (2014/10/03) 新規作成
 *
 * @version  0.9.0	2001/05/05
 * @author	 Kazuhiko Hasegawa
 * @since	 JDK1.1,
 */
public class HybsBarRenderer3D extends BarRenderer3D implements HybsDrawItem {
	private static final long serialVersionUID = 602220141003L ;

	private Color[]		categoryColor			;		// 6.0.2.1 (2014/09/26) categoryカラー配列
	private final int hsCode = Long.valueOf( System.nanoTime() ).hashCode() ;	// 5.1.9.0 (2010/08/01) equals,hashCode

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public HybsBarRenderer3D() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * itemLabelVisible 時に、最後の値のみ表示するかどうか[true:有効/false:無効]を指定します。
	 *
	 * これは、itemLabelVisible 属性に、"last" という設定値を指定した場合は、
	 * 最後のみラベル表示します。
	 * このメソッドでは、true が指定された場合は、"last" 属性が有効になったと
	 * 判断します。
	 * (独自メソッド。HybsDrawItem より継承)
	 *
	 * @og.rev 4.1.2.0 (2008/03/12) 新規追加
	 * @og.rev 6.0.2.2 (2014/10/03) このクラスでは使用していません。
	 *
	 * @param	flag	最後の値のみ表示するかどうか[true:有効/false:無効]
	 */
	@Override
	public void setItemLabelLastVisible( final boolean flag ) {
		// 現時点では、何もしません。
	}

	/**
	 * categoryカラー配列を設定します。
	 *
	 * これは、HybsJDBCCategoryDataset クラスで、カテゴリカラーを指定した場合に、
	 * そこから取り出した値をセットすることで、Hybs***Renderer に設定して使います。
	 * Hybs***Renderer 側では、このカラー配列を使用して、getItemPaint(int,int) を
	 * オーバーライドして使います。
	 * (独自メソッド。HybsDrawItem より継承)
	 *
	 * @og.rev 6.0.2.1 (2014/09/26) 新規追加
	 *
	 * @param	cateColor	categoryカラー配列(可変長引数)
	 */
	@Override
	public void setCategoryColor( final Color... cateColor ) {
		// 6.0.2.5 (2014/10/31) refactoring
		if( cateColor != null ) { categoryColor = cateColor.clone(); }		// 6.1.1.0 (2015/01/17) 可変長引数でもnullは来る。
	}

	/**
	 * カテゴリ違いのColorオブジェクトを返します。
	 *
	 * Returns the paint used to color data items as they are drawn.
	 * <p>
	 * The default implementation passes control to the
	 * <code>lookupSeriesPaint()</code> method. You can override this method
	 * if you require different behaviour.
	 *
	 * @param row  the row (or series) index (zero-based).
	 * @param column  the column (or category) index (zero-based).
	 *
	 * @return カテゴリ違いのColorオブジェクト
	 */
	@Override
	public Paint getItemPaint( final int row, final int column ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return categoryColor == null ? super.getItemPaint( row,column ) : categoryColor[column];

//		if( categoryColor == null ) { return super.getItemPaint( row,column ); }
//		return categoryColor[column];
	}

	/**
	 * drawItem と同等の機能を持った、メソッドです。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 * @og.rev 4.1.2.0 (2008/03/12) ラベルのアンダーライン時にItemLavelを表示しない
	 * @og.rev 6.0.2.2 (2014/10/03) このクラスでは高速化せず、親クラスの drawItem で処理しています。
	 *
	 * @param g2			Graphics2Dオブジェクト
	 * @param state			CategoryItemRendererStateオブジェクト
	 * @param dataArea		Rectangle2Dオブジェクト
	 * @param plot			CategoryPlotオブジェクト
	 * @param domainAxis	CategoryAxisオブジェクト
	 * @param rangeAxis		ValueAxisオブジェクト
	 * @param dataset		CategoryDatasetオブジェクト
	 * @param serNo			シリアル番号
	 */
	@Override
	public void drawItem2( final Graphics2D g2, final CategoryItemRendererState state,
			final Rectangle2D dataArea, final CategoryPlot plot, final CategoryAxis domainAxis,
			final ValueAxis rangeAxis, final CategoryDataset dataset, final int serNo ) {

		final int clmCount = dataset.getColumnCount();
		final int rowCount = dataset.getRowCount();
		final int passCount = getPassCount();

		for( int pass=0; pass<passCount; pass++ ) {
			for( int column=0; column<clmCount; column++ ) {
				for( int row=0; row<rowCount; row++ ) {
					if( row == serNo ) { continue; }	// Mis Add 2007/07/23
						drawItem(g2, state, dataArea, plot,
							domainAxis, rangeAxis, dataset,
							row, column, pass);
				}
				// 指定のシリーズと一致する場合は、最後に描画する。 Mis Add 2007/07/23
				if( serNo >= 0 ) {
						drawItem(g2, state, dataArea, plot,
							domainAxis, rangeAxis, dataset,
							serNo, column, pass);
				}
			}
		}
	}

	/**
	 * この文字列と指定されたオブジェクトを比較します。
	 *
	 * 親クラスで、equals メソッドが実装されているため、警告がでます。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) findbug対応
	 * @og.rev 5.1.9.0 (2010/08/01) findbug対応
	 *
	 * @param	object	比較するオブジェクト
	 *
	 * @return	Objectが等しい場合は true、そうでない場合は false
	 */
	@Override
	public boolean equals( final Object object ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return super.equals( object ) && hsCode == ((HybsBarRenderer3D)object).hsCode;

//		if( super.equals( object ) ) {
//			return hsCode == ((HybsBarRenderer3D)object).hsCode;
//		}
//		return false;
	}

	/**
	 * このオブジェクトのハッシュコードを取得します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) findbug対応
	 * @og.rev 5.1.9.0 (2010/08/01) findbug対応
	 *
	 * @return	ハッシュコード
	 */
	@Override
	public int hashCode() { return hsCode ; }
}
