/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.servlet;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.mail.internet.MimeUtility;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
	// import javax.servlet.http.HttpSession;				// 2017/10/06 ADD bluemixのストレージに保存する場合

import org.opengion.fukurou.security.HybsCryptography;
import org.opengion.fukurou.system.Closer;
import org.opengion.fukurou.util.KanaFilter;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import static org.opengion.fukurou.system.HybsConst.FS;		// 6.1.0.0 (2014/12/26) refactoring
	// import org.opengion.hayabusa.io.StorageAPI;				// 5.9.25.0 (2017/10/06) クラウドストレージ対応
	// import org.opengion.hayabusa.io.StorageAPIFactory;		// 5.9.25.0 (2017/10/06) クラウドストレージ対応

// import javax.servlet.annotation.WebServlet;				// 6.3.4.0 (2015/08/01)

/**
 * サーバー管理ファイルをダウンロードする場合に使用する、サーブレットです。
 *
 * 引数(URL)に指定のファイルをサーバーからクライアントにダウンロードさせます。
 * file には、サーバーファイルの物理アドレスを指定します。相対パスを使用する場合は、
 * コンテキストルート(通常、Tomcatでは、G:\webapps\dbdef2\ など)からのパスと判断します。
 * name には、クライアントに送信するファイル名を指定します。ファイル名を指定しない場合は、
 * サーバーの物理ファイルのファイル名が代わりに使用されます。
 * 日本語ファイル名は、すべて UTF-8化して処理します。指定するファイルに日本語が含まれる
 * 場合は、URLエンコードを行ってください。
 * 基本的にはContent-disposition属性として"attachment"が指定されます。
 * 但し、引数に inline=true を指定することで、Content-disposition属性に"inline"が指定されます。
 * また、システムリソースのUSE_FILEDOWNLOAD_CHECKKEYをtrueに指定することで、簡易的なチェックを
 * 行うことができます。
 * 具体的には、これを有効にすると、file属性の値から計算されるMD5チェックサムと、"key"という
 * パラメーターに指定された値が一致した場合のみダウンロードが許可され、keyが指定されていない、
 * または値が異なる場合はダウンロードエラーとなります。
 *
 * 一般的なサーブレットと同様に、デプロイメント・ディスクリプタ WEB-INF/web.xml に、
 * servlet 要素と そのマッピング(servlet-mapping)を定義する必要があります。
 *
 *     &lt;servlet&gt;
 *         &lt;servlet-name&gt;fileDownload&lt;/servlet-name&gt;
 *         &lt;servlet-class&gt;org.opengion.hayabusa.servlet.FileDownload&lt;/servlet-class&gt;
 *     &lt;/servlet&gt;
 *
 *     &lt;servlet-mapping&gt;
 *         &lt;servlet-name&gt;fileDownload&lt;/servlet-name&gt;
 *         &lt;url-pattern&gt;/jsp/fileDownload&lt;/url-pattern&gt;
 *     &lt;/servlet-mapping&gt;
 *
 * 一般には、http://:ポート/システムID/jsp/fileDownload?file=サーバー物理ファイル&amp;name=ファイル名
 * 形式のURL でアクセスします。
 * 
 * 5.9.25.0 (2017/10/06)
 * クラウド上のPaaSでオブジェクトストレージを利用する際は以下のシステムリソースを設定してください。
 * CLOUD_STORAGE,CLOUD_STORAGE_CONTAINER
 * plugin/cloud内のクラスを利用してファイルアップロード(FileUploadタグ)、ダウンロード(FileDownloadサーブレット)をAPI経由で行います。
 * プラグインが利用するjarファイルの配置は必要です。
 * 
 * 5.8.1.0 (2014/11/07)
 * forwardでアクセスする場合はファイル名の文字コード変換が不要なため、useStringConvert=falseの
 * 引数を与えてください。（falseとしない場合は日本語ファイル名等でエラーが発生します）
 *
 * @og.rev 3.8.1.1 (2005/11/21) 新規追加
 * @og.rev 5.9.25.0 (2017/10/06) クラウド対応
 * @og.group その他機能
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
// @WebServlet(name="org.opengion.hayabusa.servlet.FileDownload",urlPatterns={"/jsp/fileDownload"})
public class FileDownload extends HttpServlet {
	private static final long serialVersionUID = 539020110901L ;

	// 拡張子contentType対応テーブル
	private static final String CONTENT_TYPE_TABLE[][] = {
		{"jpg", "image/pjpeg"	},
		{"gif", "image/gif"		},
		{"txt", "text/plain"	},
		// OpenDocument追加
		{"xls", "application/vnd.ms-excel"},
		{"odp", "application/vnd.oasis.opendocument.presentation"}, // 4.3.5.5 (2008/03/08)
		{"ods", "application/vnd.oasis.opendocument.spreadsheet"}, // 4.3.5.5 (2008/03/08)
		{"odt", "application/vnd.oasis.opendocument.text"} // 4.3.5.5 (2008/03/08)
	};
	private static final int EXTENTION	 = 0;
	private static final int CONTENT_TYPE= 1;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public FileDownload() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * GET メソッドが呼ばれたときに実行します。
	 *
	 * 処理は、doPost へ振りなおしています。
	 *
	 * @param	request	HttpServletRequestオブジェクト
	 * @param	response	HttpServletResponseオブジェクト
	 *
	 * @og.rev 3.8.1.2 (2005/12/19) 半角カナ-全角カナ変換機能の追加
	 *
	 * @throws ServletException サーブレット関係のエラーが発生した場合、throw されます。
	 * @throws IOException 入出力エラーが発生したとき
	 */
	@Override
	public void doGet( final HttpServletRequest request, final HttpServletResponse response )
							throws ServletException, IOException {
		doPost( request,response );
	}

	/**
	 * POST メソッドが呼ばれたときに実行します。
	 *
	 * file 引数の サーバー物理ファイルを、クライアントにストリーム化して返します。
	 * name 引数があれば、その名前のファイル名でクライアントがファイルセーブできるように
	 * します。name 引数がなければ、そのまま物理ファイル名が使用されます。
	 * サーバー物理ファイル名が、相対パスの場合、コンテキストルートに対する相対パスになります。
	 * (例：G:\webapps\dbdef2\ など)
	 *
	 * @og.rev 5.3.2.0 (2011/02/01) 日本語ファイル名が正しく処理できないバグを修正
	 * @og.rev 5.3.4.0 (2011/04/01) IEでファイルが正しくダウンロードできないバグを修正
	 * @og.rev 5.3.5.0 (2011/05/01) ファイルダウンロードチェックキー対応
	 * @og.rev 5.3.6.0 (2011/06/01) ファイルダウンロードはattachmentに変更(ダウンロードダイアログを出す)
	 * @og.rev 5.3.8.0 (2011/08/01) ファイル名指定でIEの場合、URLエンコードすると途中で切れるため(IE7のバグ)、Shift_JIS(WIndows-31J)で直接指定する。
	 * @og.rev 5.3.9.0 (2011/09/01) 引数にinline=trueを指定することで、インライン表示が出来るように対応
	 * @og.rev 5.7.1.2 (2013/12/20) 日本語ファイルのIE11対応（UA変更）,msg ⇒ errMsg 変更
	 * @og.rev 5.8.1.0 (2014/11/07) forward時の文字コード変換不要対応
	 * @og.rev 5.9.25.0 (2017/10/06) ｸﾗｳﾄﾞｽﾄﾚｰｼﾞからﾀﾞｳﾝﾛｰﾄﾞ処理を追加対応
	 *
	 * @param	request	HttpServletRequestオブジェクト
	 * @param	response	HttpServletResponseオブジェクト
	 *
	 * @throws ServletException サーブレット関係のエラーが発生した場合、throw されます。
	 * @throws IOException 入出力エラーが発生したとき
	 */
	@Override
	public void doPost( final HttpServletRequest request, final HttpServletResponse response )
							throws ServletException, IOException {

//		// 3.8.1.2 (2005/12/19) 半角カナ-全角カナ変換機能の追加
//		final boolean hanzenFlag = HybsSystem.sysBool( "USE_FILEDOWNLOAD_HAN_ZEN" );

//		String reqFilename = request.getParameter( "file" );
//		String newFilename = request.getParameter( "name" );

//		// 5.3.9.0 (2011/09/01) 引数にinline=trueを指定することで、インライン表示が出来るように対応
//		final boolean inline = StringUtil.nval( request.getParameter( "inline" ), false );
//		final String dipositionType = inline ? "inline" : "attachment";

		// 2017/10/06 ADD
		// ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ指定
		final String storage = HybsSystem.sys( "CLOUD_STORAGE");
		// ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ指定ﾌﾗｸﾞ
		final boolean cloudFlag = storage != null && storage.length() > 0;
//		boolean cloudFlag = false;
//		if(storage != null && storage.length() > 0){
//			cloudFlag = true;
//		}

		// 5.8.1.0 (2014/11/07) エンコード変換対応
		final boolean useStrCnv = StringUtil.nval( request.getParameter( "useStringConvert" ), true );

		// クライアント側の文字エンコーディングをUTF-8に変換
		// 5.8.1.0 (2014/11/07) 条件追加
		String reqFilename = request.getParameter( "file" );		// 6.4.1.1 (2016/01/16) PMD refactoring.
		if( useStrCnv ){
			reqFilename = new String( reqFilename.getBytes("ISO-8859-1"), "UTF-8" );
		}

		// 2017/10/06 ADD reqFilenameの保存
		final String cloudFilename = reqFilename;

		// 5.3.5.0 (2011/05/01) ファイルダウンロードチェックキー対応
		final boolean useCheck = HybsSystem.sysBool( "USE_FILEDOWNLOAD_CHECKKEY" );
		if( useCheck ) {
			final String checkKey = request.getParameter( "key" );
			if( checkKey == null || !checkKey.equals( HybsCryptography.getMD5( reqFilename ) ) ) {
				final String errMsg = "アクセスが拒否されました。(URLチェック)";
				throw new HybsSystemException( errMsg );	// 5.7.1.2 (2013/12/20) msg ⇒ errMsg 変更
			}
		}

		// 相対パスを絶対パスに変換。ファイルセパレータも正規化されています。
		reqFilename = HybsSystem.url2dir( reqFilename );

		// 拡張子からcontentTypeを獲得
		final String contentType = getContentType( reqFilename );
		// contentTypeを出力
		response.setContentType( contentType );

		// 表示ファイル名の指定
		String newFilename = request.getParameter( "name" );		// 6.4.1.1 (2016/01/16) PMD refactoring.
		if( newFilename == null || newFilename.isEmpty() ) {
			newFilename = getFileName( reqFilename );
		}
		else if( useStrCnv ){		// 5.8.1.0 (2014/11/07) 条件追加
			newFilename = new String( newFilename.getBytes("ISO-8859-1"), "UTF-8" );
		}

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		// 3.8.1.2 (2005/12/19) 半角カナを全角カナに置き換えます。ファイルダイアログの文字化け仮対応
//		if( hanzenFlag ) {
		if( HybsSystem.sysBool( "USE_FILEDOWNLOAD_HAN_ZEN" ) ) {
			newFilename = KanaFilter.han2zen( newFilename );
		}

		// 5.7.1.2 (2013/12/20) 条件を反転させた上でIE11対応を行う
		final String reqHeader = request.getHeader( "User-Agent" );
		if( reqHeader.indexOf( "MSIE" ) >= 0 || reqHeader.indexOf( "Trident" ) >= 0 ) {
			newFilename = new String( newFilename.getBytes("Windows-31J"), "ISO-8859-1" );
		}
		else {
			newFilename = MimeUtility.encodeWord( newFilename, "UTF-8", "B" );
		}

		// 5.3.9.0 (2011/09/01) 引数にinline=trueを指定することで、インライン表示が出来るように対応
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final boolean inline = StringUtil.nval( request.getParameter( "inline" ), false );
		final String dipositionType = inline ? "inline" : "attachment";

		// ファイル名の送信( attachment部分をinlineに変更すればインライン表示 )
		// 5.3.9.0 (2011/09/01) 引数にinline=trueを指定することで、インライン表示が出来るように対応
		response.setHeader( "Content-disposition", dipositionType + "; filename=\"" + newFilename + "\"" );

		// 5.3.4.0 (2011/04/01) IEでファイルが正しくダウンロードできないバグを修正
		response.setHeader( "Cache-Control", "public" );

		// ファイル内容の出力
		FileInputStream     fin = null;
		ServletOutputStream out = null;
		// 2017/10/06 MODIFY bluemixのストレージ利用の処理を追加
		InputStream is = null;
		try {
//			fin = new FileInputStream( reqFilename );
			// 2017/10/06 ADD bluemixのストレージに保存する場合の処理を追加
	//		if(cloudFlag){
	//			HttpSession hsession = request.getSession(true);
	//			StorageAPI storageApi = StorageAPIFactory.newStorageAPI(storage, HybsSystem.sys("CLOUD_STORAGE_CONTAINER"), hsession);
	//			is = storageApi.get(cloudFilename, hsession);
	//		}else{
	//		// 標準のファイル保存
				fin = new FileInputStream( reqFilename );
				is = fin;
	//		}
			out = response.getOutputStream();

			// ファイル読み込み用バッファ
			final byte buffer[]  = new byte[4096];
			int size;
	//		while((size = fin.read(buffer))!=-1) {
			while((size = is.read(buffer))!=-1) {
				out.write(buffer,0, size);
				out.flush();
			}
		}
		finally {
			Closer.ioClose(is);		// 2017/10/06 ADD
			Closer.ioClose( fin );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
			Closer.ioClose( out );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
		}
	}

	/**
	 * アドレス名から拡張子を取り出します。
	 *
	 * アドレス名の後ろから、"." 以降を拡張子として切り取ります。
	 * 拡張子が存在しない場合(指定のファイル名に "." が含まれない場合)は
	 * ゼロ文字列("")を返します。
	 *
	 * @param	fileAddress	アドレス名
	 *
	 * @return	拡張子
	 * @og.rtnNotNull
	 */
	private String getExtention( final String fileAddress ) {
		final int idx = fileAddress.lastIndexOf( '.' );

		return idx >= 0 ? fileAddress.substring( idx+1 ) : "";		// 6.1.1.0 (2015/01/17) refactoring

	}

	/**
	 * アドレス名からファイル名を取り出します。
	 *
	 * アドレス名の後ろから、ファイルセパレータ以降をファイル名として切り取ります。
	 * ファイルセパレータが存在しない場合はアドレス名をそのまま返します。
	 * ここでは、OS毎に異なるファイルセパレータを統一後に処理してください。
	 *
	 * @param	fileAddress	アドレス名
	 *
	 * @return	ファイル名
	 */
	private String getFileName( final String fileAddress ) {
		final int idx = fileAddress.lastIndexOf( FS );

		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		// 条件変更(反転)注意
		return idx < 0 ? fileAddress : fileAddress.substring( idx+1 );

//		if( idx!=-1 ) { return fileAddress.substring( idx+1 ); }
//		return fileAddress;
	}

	/**
	 * アドレス名から対応するコンテンツタイプを取り出します。
	 *
	 * アドレス名から、ファイル拡張子を取り出し、対応するコンテンツタイプを返します。
	 * コンテンツタイプは、CONTENT_TYPE_TABLE 配列に定義している中から検索して返します。
	 * 存在しない場合は、"application/octet-stream" を返します。
	 *
	 * @param	fileAddress	アドレス名
	 *
	 * @return	コンテンツタイプ
	 */
	private String getContentType( final String fileAddress ) {
		final String extention = getExtention( fileAddress );
		for( int j=0; j<CONTENT_TYPE_TABLE.length; j++ ) {
			if( CONTENT_TYPE_TABLE[j][EXTENTION].equalsIgnoreCase( extention ) ) {
				return CONTENT_TYPE_TABLE[j][CONTENT_TYPE];
			}
		}
		return "application/octet-stream";
	}
}
