/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.fukurou.system.DateSet;						// 6.4.2.0 (2016/01/29)
// import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBConstValue;

/**
 * データベース登録時に、固定値カラムの設定を行う情報を管理するクラスです。
 * このクラスは、GrowingMIS Ver6用の管理クラスです。
 * 具体的には、
 *     作成日時    DYSET
 *     作成ＰＧ    PGSET
 *     作成親ＰＧ  PGPSET
 *     作成社員    USRSET
 *     更新日時    DYUPD
 *     更新ＰＧ    PGUPD
 *     更新親ＰＧ  PGPUPD
 *     更新社員    USRUPD
 * などの情報を、INSERT、UPDATE の状況に応じて提供します。
 * INSERT には、すべての情報(XXUPD)にも値をセットします。
 * これは、UPDATE された日付や時刻で、一覧検索できるようにする為で、
 * null にしておくと、IS NULL で問い合わせる必要があります。
 * INSERT データだけ取出す場合は、WHERE DYSET = DYUPD とします。
 * また、 物理SEQは、SEQxxxx(xxxxはテーブル名)形式のカラム名とし、
 * 値は、SEQxxxxS01.NEXTVAL とします。
 *
 * コンストラクターでは、テーブル名、ユーザーID,画面ID を受け取ります。
 * 登録時刻は、このオブジェクトが生成された時間を通常は使用します。
 * テーブル名を渡すのは、ワークなど、テーブル単位に処理するカラムが
 * 異なるので、それへの対応です。
 * 処理が簡素化できるように、ルールを規定すべきです。
 *
 * @og.rev 5.5.1.8 (2012/04/17) 新規追加
 * @og.group データ登録
 *
 * @version  5.5
 * @author   Kazuhiko Hasegawa
 * @since    JDK7.0,
 */
public class DBConstValue_GJ implements DBConstValue {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	private static final String[] MSTR_INSKEYS = {"DYSET","PGSET","PGPSET","USRSET","DYUPD","PGUPD","PGPUPD","USRUPD" };

	private static final String[] MSTR_UPDKEYS = { "DYUPD","PGUPD","PGPUPD","USRUPD" };

	private static final String[] ZERO = new String[0] ;

	private String[] insKeys = ZERO;
	private String[] insVals = ZERO;

	private String[] updKeys = ZERO;
	private String[] updVals = ZERO;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public DBConstValue_GJ() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * 初期化情報を設定します。
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) DateSet.getDate( String ) を利用するように修正します。
	 *
	 * @param	table	テーブル名
	 * @param	user 	ユーザーID
	 * @param	gamen	画面ID
	 */
	public void init( final String table, final String user, final String gamen ) {
		if( isUseClm( table ) ) {
			final String userID  = "'" + user + "'";
			final String gamenID = "'" + gamen + "'";
//			final String timestamp = "'" + HybsSystem.getDate( "yyyyMMddHHmmss" ) + "'" ;
			final String timestamp = "'" + DateSet.getDate( "yyyyMMddHHmmss" ) + "'" ;		// 6.4.2.0 (2016/01/29)

			insKeys = MSTR_INSKEYS;
			insVals = new String[] {
							 timestamp		// DYSET
							,gamenID		// PGSET
							,gamenID		// PGPSET
							,userID			// USRSET
							,timestamp		// DYUPD
							,gamenID		// PGUPD
							,gamenID		// PGPUPD
							,userID			// USRUPD
			};

			updKeys = MSTR_UPDKEYS;
			updVals = new String[] {
							 timestamp		// DYUPD
							,gamenID		// PGUPD
							,gamenID		// PGPUPD
							,userID			// USRUPD
			};
		}
	}

	/**
	 * インサートする場合のキー配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @return	インサートする場合のキー配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getInsertKeys() {
		return insKeys.clone();
	}

	/**
	 * インサートする場合の値配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @return	インサートする場合のキー配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getInsertVals() {
		return insVals.clone();
	}

	/**
	 * アップデートする場合のキー配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @return	アップデートする場合のキー配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getUpdateKeys() {
		return updKeys.clone();
	}

	/**
	 * アップデートする場合の値配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @return	アップデートする場合のキー配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getUpdateVals() {
		return updVals.clone();
	}

	/**
	 * デリート(論理削除) する場合のキー配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) Returning 'ZERO' may expose an internal array.(PMD)
	 *
	 * @return	デリートする場合のキー配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getLogicalDeleteKeys() {
//		return ZERO;
		return ZERO.clone();		// 6.3.9.0 (2015/11/06) Returning 'ZERO' may expose an internal array.(PMD)
	}

	/**
	 * デリート(論理削除) する場合の値配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) Returning 'ZERO' may expose an internal array.(PMD)
	 *
	 * @return	デリートする場合のキー配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getLogicalDeleteVals() {
//		return ZERO;
		return ZERO.clone();		// 6.3.9.0 (2015/11/06) Returning 'ZERO' may expose an internal array.(PMD)
	}

	/**
	 * カラムを返すかどうかを決めるロジック
	 *
	 * テーブル名が存在していれば、利用可能と判断します。
	 * これは、履歴テーブルなど、この処理を使わない場合に、
	 * テーブル名から判定する場合に使います。現時点では、履歴テーブルは
	 * フレームワーク側で変更するのではなく、トリガ等による自動収集と
	 * しますので、判定対象としていません。
	 *
	 * @param	table	テーブル名
	 *
	 * @return	カラムを返すかどうか(true:カラムを生成/false:カラムは生成しない)
	 */
	private boolean isUseClm( final String table ) {
		final boolean flag = table != null && table.length() > 0 ;

		return flag ;
	}
}
