package org.opengion.hayabusa.taglib;

// import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.io.JsChartData;

import java.util.Set;											// 6.4.3.4 (2016/03/11)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)
import org.opengion.fukurou.util.ToString;
import org.opengion.fukurou.util.ColorMap;						// 6.7.7.0 (2017/03/31)
import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * 設定された値をJsChartDataに設定し、
 * JsChartTagのJsChartDataリストに追加するタグです。
 * 
 * @og.formSample
 * ●形式：&lt;og:jsChartData chartColumn="…" … /&gt;
 * ●body:なし
 * 
 * ●Tag定義：
 * &lt;og:jsChartData
 *      chartColumn     ○【TAG】ﾁｬｰﾄのｶﾗﾑ名を指定します(必須)。
 *      label             【TAG】凡例の値を指定します。
 *      fill              【TAG】線下を塗りつぶすかどうか[true/false]を指定します(初期値:false)。
 *      tension           【TAG】線の伸張を指定します。0で直線になります(初期値:0.4)。
 *      borderColor       【TAG】線の色を指定します。
 *      colorNo           【TAG】線の色(borderColor)をColorMapの色番号で指定します。
 *      borderWidth       【TAG】線の幅を指定します。
 *      backgroundColor   【TAG】ﾃﾞｰﾀの背景色を指定します。
 *      optionAttributes  【TAG】その他ｵﾌﾟｼｮﾝを指定します。
 *      caseKey           【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *      caseVal           【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *      caseNN            【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *      caseNull          【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *      caseIf            【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *  &gt; />
 * 
 * ●使用例
 * &lt;og:jsChart...&gt;
 *     &lt;og:jsChartData
 *         chartColumn ="CLM1"
 *         label       ="ラベル"
 *         fill        ="true"
 *         tension     ="0"
 *         borderColor ="rbga(150,150,150,0.7)"
 *         borderWidth ="2"
 *     /&gt;>
 * &lt/og:jsChart&gt;
 * 
 * @og.group 画面表示
 * 
 * @version	5.9.17.2		2017/02/08
 * @author	T.OTA
 * @since	JDK7.0
 * 
 */
public class JsChartDataTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。{@VALUE} */
	private static final String		VERSION				= "6.7.7.0 (2017/03/31)";
	private static final long		serialVersionUID	= 677020170331L;

//	private static final String[]	TYPE_BOOLEAN		= new String[] { "true", "false" };
	private static final Set<String> TYPE_BOOLEAN = new ArraySet<>( "true", "false" );

	// 変数宣言
	private String					chartColumn			;
	private String					label				;
	private String					fill				= "false";
	private String					tension				= "0.4";
	private String					borderColor			;
	private String					borderWidth			;
	private String					backgroundColor		;
	private String					optionAttributes	;
//	private transient JsChartData	jsData				= new JsChartData();	// 6.7.7.0 (2017/03/31) ローカル変数化

//	/**
//	 * Taglibの開始タグが見つかった時に処理する doStartTag() を オーバーライドします。
//	 * 
//	 * @return 後続処理の指示
//	 */
//	@Override
//	public int doStartTag() {
//		return SKIP_BODY; // Body を評価しない
//	}

	/**
	 * Taglibの終了タグが見つかった時に処理する doEndTag() を オーバーライドします。
	 * 
	 * @og.rev 6.7.6.0 (2017/03/17) タグの使用を決める共通属性の追加
	 * @og.rev 6.7.7.0 (2017/03/31) backgroundColor が未設定の場合は、borderColor を使用します。
	 * 
	 * @return 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();
		if( !useTag() ) { return EVAL_PAGE ; }			// 6.7.6.0 (2017/03/17)

		final JsChartTag jsChartTag = (JsChartTag) findAncestorWithClass( this, JsChartTag.class );

		if( jsChartTag == null ) {
			final String errMsg = "jsChart タグが見つかりませんでした。";
			throw new HybsSystemException( errMsg );
		}

		// 6.7.7.0 (2017/03/31) 各種初期設定を行います。
		if( borderColor == null || borderColor.isEmpty() ) {
			final int size = jsChartTag.getJsChartDataSize();	// 登録順に色番号を設定します。
			final String[] cols = ColorMap.getColorKeys();
			borderColor = cols[size % cols.length];
		}
		if( backgroundColor == null || backgroundColor.isEmpty() ) {
			backgroundColor = borderColor;
		}

		final JsChartData jsData = new JsChartData();	// 6.7.7.0 (2017/03/31) ローカル変数化
		jsData.setChartColumn( chartColumn );
		jsData.setLabel( label );
		jsData.setFill( fill );
		jsData.setTension( tension );
		jsData.setBorderColor( borderColor );
		jsData.setBorderWidth( borderWidth );
		jsData.setBackgroundColor( backgroundColor );
		jsData.setOptionAttributes( optionAttributes );

		jsChartTag.addJsChartData( jsData );

		return EVAL_PAGE;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 * 
	 * @og.rev 6.7.7.0 (2017/03/31) jsDataのローカル変数化。
	 */
	@Override
	protected void release2() {
		super.release2();
//		jsData				= new JsChartData();	// 6.7.7.0 (2017/03/31)
		chartColumn			= null;
		label				= null;
		fill				= "false";
		tension				= "0.4";
		borderColor			= null;
		borderWidth			= null;
		backgroundColor		= null;
		optionAttributes	= null;
	}

	/**
	 * 【TAG】ﾁｬｰﾄのｶﾗﾑ名を指定します(必須)。
	 *
	 * @og.tag
	 * 
	 * @param clm ﾁｬｰﾄのｶﾗﾑ名
	 */
	public void setChartColumn( final String clm ) {
		chartColumn = nval( getRequestParameter( clm ),chartColumn );
	}

	/**
	 * 【TAG】凡例の値を指定します。
	 *
	 * @og.tag
	 * 
	 * @param lbl 凡例
	 */
	public void setLabel( final String lbl ) {
		label = nval( getRequestParameter( lbl ),label );
	}

	/**
	 * 【TAG】線下を塗りつぶすかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * フィル(線より下の塗りつぶし) を設定します。
	 * 
	 * @param fill 塗りつぶすかどうか [true/false]
	 */
	public void setFill( final String fill ) {
		this.fill = nval( getRequestParameter( fill ) , this.fill );

		checkPara( this.fill, TYPE_BOOLEAN, "fill" );
	}

	/**
	 * 【TAG】線の伸張を指定します。0で直線になります(初期値:0.4)。
	 *
	 * @og.tag
	 * 伸張 を設定します。
	 *
	 * @param tension 線の伸張
	 */
	public void setTension( final String tension ) {
		this.tension = nval( getRequestParameter( tension ),this.tension );
	}

	/**
	 * 【TAG】線の色を指定します。
	 *
	 * @og.tag
	 * borderColor = "BLUE" とすると、すべての線の色を指定できます。
	 * 配列で指定すると、データの順番に適用されます。
	 * 例：borderColor = "['#ffaaaa','#ffffaa','#aaffaa','#aaaaff','#aaaaff']"
	 * 
	 * 色の代わりに、ColorMapの色番号を指定した場合でも、borderColor が優先されます。
	 * どちらも指定しない場合は、JsChartTagに登録した順番に色コードで指定されます。
	 * 
	 * @param color 線の色
	 * @see		#setColorNo(String)
	 */
	public void setBorderColor( final String color ) {
		borderColor = nval( getRequestParameter( color ),borderColor );
	}

	/**
	 * 【TAG】線の色(borderColor)をColorMapの色番号で指定します。
	 *
	 * @og.tag
	 * 色の代わりに、ColorMapの色番号を指定した場合でも、borderColor が優先されます。
	 * この引数は、色に変換後、borderColor に設定されます。
	 *
	 * @og.rev 6.7.7.0 (2017/03/31) ColorMapの色番号で指定
	 * 
	 * @param colorNo 線の色の番号
	 * @see		ColorMap#getColorKeys()
	 */
	public void setColorNo( final String colorNo ) {
		if( borderColor == null ) {		// borderColor が、未設定の場合のみ、色番号を利用する。
			final String colNo = nval( getRequestParameter( colorNo ),null );
			if( colNo != null ) {
				try {
					final int no = Integer.parseInt( colNo );
					final String[] cols = ColorMap.getColorKeys();
					borderColor = cols[no % cols.length];
				}
				catch( final NumberFormatException ex ) {
					final String errMsg = "colorNo を数値に変換できません。colorNo=" + colNo ;
					throw new HybsSystemException( errMsg,ex );
				}
			}
		}
	}

	/**
	 * 【TAG】線の幅を指定します。
	 *
	 * @og.tag
	 * 
	 * @param width 線の幅
	 */
	public void setBorderWidth( final String width ) {
		borderWidth = nval( getRequestParameter( width ),borderWidth );
	}

	/**
	 * 【TAG】ﾃﾞｰﾀの背景色を指定します。
	 *
	 * @og.tag
	 * backgroundColor = "BLUE" とすると、すべての背景色を指定できます。
	 * 配列で指定すると、データの順番に適用されます。
	 * 例：backgroundColor = "['#ffaaaa','#ffffaa','#aaffaa','#aaaaff','#aaaaff']"
	 * 
	 * 背景色を指定しない場合、線の色(borderColor)を使用します。
	 * 
	 * @param bgColor 背景色
	 * @see		#setBorderColor(String)
	 */
	public void setBackgroundColor( final String bgColor ) {
		backgroundColor = nval( getRequestParameter( bgColor ),backgroundColor );
	}

	/**
	 * 【TAG】その他ｵﾌﾟｼｮﾝを指定します。
	 *
	 * @og.tag
	 * 
	 * @param attri その他ｵﾌﾟｼｮﾝ
	 */
	public void setOptionAttributes( final String attri ) {
		optionAttributes = nval( getRequestParameter( attri ),optionAttributes );
	}

	/**
	 * ﾊﾟﾗﾒｰﾀﾁｪｯｸ用ﾒｿｯﾄﾞ。
	 * 
	 * @param trg チェック対象
	 * @param set 設定可能なリスト
	 * @param trgStr チェック対象の文字列(ｴﾗｰ表示用)
	 */
//	private void checkPara( String trg, String[] list, String trgStr ) {
	private void checkPara( final String trg, final Set<String> set, final String trgStr ) {
//		if( trg != null && trg.length() > 0 && !check( trg, list ) ) {
		if( trg != null && trg.length() > 0 && !check( trg, set ) ) {
			final StringBuilder errMsg = new StringBuilder( BUFFER_MIDDLE )
				.append( "指定の" ).append( trgStr ).append( "は指定できません。" )
				.append( CR )
				.append( trgStr ).append( "=[" ).append( trg ).append( ']' )
				.append( CR );
			for( final String lst : set ) {
				errMsg.append( " | " )
					.append( lst );
			}
//			for( int i = 0; i < list.length; i++ ) {
//				errMsg.append( " | " );
//				errMsg.append( list[i] );
//			}
			throw new HybsSystemException( errMsg.toString() );
		}
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 * 
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
//		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
		return ToString.title( this.getClass().getName() )
			.println( "VERSIION"			, VERSION			)
			.println( "chartColumn"			, chartColumn		)
			.println( "label"				, label				)
			.println( "fill"				, fill				)
			.println( "tension"				, tension			)
			.println( "borderColor"			, borderColor		)
			.println( "borderWidth"			, borderWidth		)
			.println( "backgroundColor"		, backgroundColor	)
			.println( "optionAttributes"	, optionAttributes	)
			.fixForm().toString();
	}
}
