/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

import java.util.Locale ;

/**
 * 半角小文字の英数字の文字列を扱う為の、カラム属性を定義します。
 *
 * 入力文字列は、trim された後、小文字化(toLowerCase)された後、タイプチェックされます。
 * 半角カタカナは半角小文字に分類されない為、対象外(使用できない)です。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、Byte換算での文字数との比較
 * ・半角小文字チェック「c &lt; 0x20 || c &gt; 0x7e || ( 'A' &lt;= c &amp;&amp; c &lt;= 'Z' )以外」エラー
 * ・文字パラメータの 正規表現チェック
 * ・クロスサイトスクリプティングチェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_XL extends AbstractDBType {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public DBType_XL() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * 半角０文字の固定長でFILL埋めされた文字列を返します。
	 * なお、エラーチェックは行われません。
	 * 実行前に、必ず valueCheck( String value ,int len ) が行われる必要があります。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) エンコード指定に変更します。
	 *
	 * @param   value    ＦＩＬＬ埋めする文字列
	 * @param   sizeX    整数部分の文字列の長さ
	 * @param   sizeY    小数部分の文字列の長さ
	 * @param   encode   固定長で変換する文字エンコード
	 *
	 * @return  ＦＩＬＬ埋めした新しい文字列
	 */
	@Override
	public String valueFill( final String value ,final int sizeX ,final int sizeY,final String encode ) {
		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;

		return StringUtil.stringXFill( value,len );
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと Editor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 前後のスペースを取り除いておく。
	 * @og.rev 3.3.3.1 (2003/07/18) 後ろスペースを取り除く。(StringUtil#rTrim)
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 *
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		return value == null ? null : StringUtil.rTrim( value ).toLowerCase(Locale.JAPAN) ;
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータ(文字パラメータ)を引数に追加
	 * @og.rev 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	小数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ(文字パラメータ)
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 * @og.rtnNotNull
	 */
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
		final ErrorMessage msg = new ErrorMessage();
		if( value == null || value.isEmpty() ) { return msg; }

		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		if( value.length() > len ) {
			// 文字列の長さが指定の長さよりも長いです。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0006", key,value, String.valueOf( value.length() ), String.valueOf( len ) );
		}

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		boolean isError = false;
		for( int i=0; i<value.length(); i++ ) {
			final char ch = value.charAt( i );
			if( ch < 0x20 || ch > 0x7e || ( 'A' <= ch && ch <= 'Z' ) ) {
				buf.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				isError = true;
			}
			else {
				buf.append( ch );
			}
		}
		if( isError ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,buf.toString() );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータ(文字パラメータ)を使用したマッチチェック
		final String check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// クロスサイトスクリプティング対策：'<', '>' は登録させない。
		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
		return xssCheck( key ,value, msg );
	}
}
