/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.system;								// 6.4.2.0 (2016/01/29) package変更 fukurou.util → fukurou.system

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;

/**
 * Logを書き込む為の PrintWriter を管理するクラスです。
 *
 * 実際の Log の書き込みには， LogSender を利用して下さい。
 *
 * @og.rev 6.4.2.0 (2016/01/29) package変更 fukurou.util → fukurou.system
 *
 * @og.group エラー処理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class LogWriter {
	private static PrintWriter writer ;
	private static final Object LOCK = new Object();	// 6.3.9.0 (2015/11/06) synchronizedブロック

	private static String logFileUrl	;				// 4.1.0.1 (2008/01/23)

	/**
	 * デフォルトコンストラクター
	 * private にして，コンストラクターの作成をさせない様にしています。
	 *
	 */
	private LogWriter() {}

	/**
	 * Logファイルの出力先を設定します。
	 *
	 * このファイル名は、日付フォーマット変数を含むことができます。
	 *
	 * @og.rev 4.1.0.1 (2008/01/23) 新規作成
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 * @og.rev 6.4.2.0 (2016/01/29) 出力先変更時は、既存の Writer をclose() してからにする。
	 *
	 * @param   url 出力先
	 * @see		org.opengion.fukurou.system.DateSet#changeString(String)
	 */
	public static void init( final String url ) {
		synchronized( LOCK ) {
			close();
			logFileUrl = url;
		}
	}

	/**
	 * Logを書き出します。
	 *
	 * @og.rev 4.1.0.1 (2008/01/23) 出力時間を出力する。
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
	 * @param   message メッセージ
	 */
	public static void log( final String message ) {
		synchronized( LOCK ) {
			if( writer == null ) { writer = getPrintWriter(); }
		}
		writer.println( "[WriteTime= " + DateSet.getDate( "yyyy/MM/dd HH:mm:ss.SSS" ) + "] " + message );	// 5.5.7.2 (2012/10/09) HybsDateUtil を利用
		writer.flush();
	}

	/**
	 * 例外のスタックトレースをLogWriterのPrintWriterに書き出します。
	 *
	 * @og.rev 4.1.0.1 (2008/01/23) 新規作成
	 * @og.rev 4.3.4.5 (2009/01/08) nullチェック追加
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 * @og.rev 6.4.2.0 (2016/01/29) ex.printStackTrace() を、ThrowUtil#ogStackTrace(Throwable) に置き換え。
	 *
	 * @param   th スタックトレースの取得元Throwableオブジェクト
	 */
	public static void log( final Throwable th ) {
		synchronized( LOCK ) {
			if( writer == null ) { writer = getPrintWriter(); }
		}
		writer.println( ThrowUtil.ogStackTrace( th ) );				// 6.4.2.0 (2016/01/29)
	}

	/**
	 * PrintWriter を close() します。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 */
	public static void close() {
		synchronized( LOCK ) {
			if( writer != null ) { writer.close(); }
			writer = null;
		}
	}

	/**
	 * 追加モードで作成した，PrintWriter を取得します。
	 * PrintWriter は，シングルトーンとして，唯一存在させています。
	 *
	 * @og.rev 4.1.0.1 (2008/01/23) ログファイル出力先を外部から指定する。
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
	 * @return 追加モードで作成したPrintWriter
	 * @og.rtnNotNull
	 */
	private static PrintWriter getPrintWriter() {

		if( logFileUrl == null || logFileUrl.isEmpty() ) {
			return new PrintWriter( System.err );
		}
		else {
			// 日付フォームのファイル名を変換します。
			final DateSet dateSet = new DateSet();
			final String timeFileUrl = dateSet.changeString( logFileUrl );		// 6.3.9.0 (2015/11/06) 日付フォーマットは、作成都度行う事とする。

			try {
				final File logFile = new File( timeFileUrl );
				return new PrintWriter( new BufferedWriter( new FileWriter( logFile, true ) ) );
			}
			catch( final IOException ex ) {
				final String errMsg = "ログライターが作成できません。[" + timeFileUrl + "]";
				throw new OgRuntimeException( errMsg, ex );
			}
		}
	}
}
