/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.Closer ;
import org.opengion.fukurou.util.FileUtil ;
import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.mail.MailTX ;

import java.util.Map ;
import java.util.LinkedHashMap ;
import java.io.PrintWriter ;
import java.io.StringWriter ;

/**
 * Process_Logger は、画面出力、ファイルログ、エラーメールを管理する、
 * ロギング関係の LoggerProcess インターフェースの実装クラスです。
 *
 * MainProcess で使用されるログと、各種 Process で使用されるディスプレイを
 * 管理します。また、エラー発生時の、メール送信機能も、ここで用意します。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * @og.formSample
 *  Process_Logger -logFile=ABC.txt -dispFile=System.out
 *
 *   [ -logFile=ログ出力先指定  ] ： -logFile=[ファイル名/System.out/System.err] (初期値:null)
 *   [ -dispFile=画面出力先指定 ] ： -dispFile=[ファイル名/System.out/System.err](初期値:null)
 *   [ -host=メールサーバ       ] ： -host=メールサーバー
 *   [ -from=送信From           ] ： -from=送信元アドレス
 *   [ -to=受信To               ] ： -to=送信先アドレスをカンマ区切りで並べる
 *   [ -charset=ｷｬﾗｸﾀｾｯﾄ        ] ： -charset=メール送信時のキャラクタセット [ISO-2022-JP / Windows-31J]
 *   [ -subject=タイトル        ] ： -subject=タイトル
 *   [ -message=本文雛形        ] ： -message=本文雛形文章
 *   [ -msgFile=本文雛形ﾌｧｲﾙ    ] ： -msgFile=本文を格納しているファイルのアドレス
 *   [ -{&#064;XXXX}=YYYY       ] ： メッセージ本文の {&#064;XXXX} 文字列を、YYYY 文字列に変換します。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_Logger extends AbstractProcess implements LoggerProcess {

	private String logFile		= null;		// ログ出力先
	private String dispFile		= null;		// 画面出力先

	private PrintWriter logWriter = null;
	private PrintWriter dispWriter = null;

	/** メール送信時のデフォルトキャラクタセット {@value}  */
	public static final String DEFAULT_CHARSET = "ISO-2022-JP" ;
	private String host = "mail.opengion.org";
	private String from = "DUMMY@DUMMY";
	private String to   = null;
	private String subject     = null;			// 5.3.1.0 (2011/03/10)
	private boolean useErrMail = false;

	private static final Map<String,String> mustProparty   ;	// ［プロパティ］必須チェック用 Map
	private static final Map<String,String> usableProparty ;	// ［プロパティ］整合性チェック Map

	static {
		mustProparty = new LinkedHashMap<String,String>();

		usableProparty = new LinkedHashMap<String,String>();
		usableProparty.put( "logFile",		"ログ出力先指定のファイル名を指定します(初期値:null)" +
											CR + "『System.out』,『System.err』は特殊な名称です。" );
		usableProparty.put( "dispFile",		"画面出力先指定のファイル名を指定します(初期値:null)" +
											CR + "『System.out』,『System.err』は特殊な名称です。" );
		usableProparty.put( "host",		"メールサーバー" );
		usableProparty.put( "from",		"送信元アドレス" );
		usableProparty.put( "to",		"送信先アドレスをカンマ区切りで並べる" );
		usableProparty.put( "charset",	"メール送信時のキャラクタセット [ISO-2022-JP / Windows-31J]" );
		usableProparty.put( "subject",	"タイトル" );
		usableProparty.put( "message",	"本文雛形文章" );
		usableProparty.put( "msgFile",	"本文雛形を格納しているファイルのアドレス" );
		usableProparty.put( "{@",		"{@XXXX}=YYYY 汎用文字変換" +
									CR + "メッセージ本文の {@XXXX} 文字列を、YYYY 文字列に変換します。"  );
		usableProparty.put( "{@ARG.",	"{@ARG.XXX} 予約文字変換 上記引数を割り当てます。" );
		usableProparty.put( "{@DATE.",	"{@DATE.XXX} 予約文字変換 の文字を変換します。" +
									CR + "(SimpleDateFormat 形式の日付、時刻等)" );
		usableProparty.put( "{@ENV.",	"{@ENV.XXX} 予約文字変換 ｼｽﾃﾑﾌﾟﾛﾊﾟﾃｨｰの文字を変換します。" +
									CR + "(java -Dkey=value オプションで引き渡します。)" );
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_Logger() {
		super( "org.opengion.fukurou.process.Process_Logger",mustProparty,usableProparty );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) タイトル追加
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		Argument arg = getArgument();

		logFile  = arg.getProparty( "logFile"  );	// ログ出力先
		dispFile = arg.getProparty( "dispFile" );	// 画面出力先

		if( logWriter == null && logFile != null ) {
			logWriter = FileUtil.getLogWriter( logFile );
		}

		if( dispWriter == null && dispFile != null ) {
			dispWriter = FileUtil.getLogWriter( dispFile );
		}

		host = arg.getProparty( "host",host );	// メールサーバー
		from = arg.getProparty( "from",from );	// 送信元アドレス
		to   = arg.getProparty( "to"  ,to   );	// 送信先アドレス
		subject    = arg.getProparty( "subject" );		// 5.3.4.0 (2011/04/01) タイトル
		useErrMail = ( host != null ) && ( from != null ) && ( to != null ) ;
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		if( logWriter != null ) {
			logWriter.flush();
			Closer.ioClose( logWriter );
		}

		if( dispWriter != null ) {
			dispWriter.flush();
			Closer.ioClose( dispWriter );
		}
	}

	/**
	 * ログファイルにメッセージを表示します。
	 *
	 * @param	msg	表示するメッセージ
	 */
	@Override
	public void logging( final String msg ) {
		if( logWriter != null ) {
			logWriter.println( msg ) ;
		}
	}

	/**
	 * ディスプレイにメッセージを表示します。
	 *
	 * @param	msg	表示するメッセージ
	 */
	@Override
	public void println( final String msg ) {
		if( dispWriter != null ) {
			dispWriter.println( msg ) ;
		}
	}

	/**
	 * エラーログにメッセージを表示します。
	 * ここに書き込まれたメッセージは、通常ログと、特殊ログの
	 * 両方に書き込まれます。
	 * 特殊ログとは、メール連絡等のことです。
	 *
	 * @param	msg	表示するメッセージ
	 * @param	th	Throwable例外オブジェクト
	 */
	public void errLog( final String msg,final Throwable th ) {
		String sendMsg = msg;
		if( logWriter != null ) {
			if( th != null ) {
				StringWriter errMsg = new StringWriter();
				errMsg.append( msg ).append( CR );
				th.printStackTrace( new PrintWriter( errMsg ) );
				sendMsg = errMsg.toString();
			}
			logWriter.println( sendMsg ) ;
		}
		println( sendMsg ) ;
		if( useErrMail ) { sendmail( sendMsg ) ; }
	}

	/**
	 * メール送信を行います。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) タイトル追加
	 *
	 * @param	msg	送信するメッセージ
	 */
	private void sendmail( final String msg ) {

		Argument arg = getArgument();

		String charset = arg.getProparty( "charset", DEFAULT_CHARSET );
		MailTX mail = new MailTX( host,charset );
		mail.setFrom( from );
		mail.setTo( StringUtil.csv2Array( to ) );
//		mail.setSubject( arg.getProparty( "subject" ) );
		mail.setSubject( subject );									// 5.3.4.0 (2011/04/01)

		String message = arg.getFileProparty("message","msgFile",false);

		// {@XXX} 変換は、Argument クラスの機能を使う。
		message = arg.changeParam( message );
		message = message + CR + msg ;
		mail.setMessage( message );
		mail.sendmail();
	}

	/**
	 * ログ出力用のPrintWriterを設定します。
	 * 通常は、引数の -logFile=XXXX で指定しますが、直接 PrintWriter を
	 * 渡す必要があるケース(JSPなどで使用するケース)で使用します。
	 * 引数より、こちらの設定のほうが、優先されます。
	 * ※ JspWriter を渡す場合の PrintWriter は、flushing および、close 処理を
	 * 行わない NonFlushPrintWriter を設定してください。
	 *
	 * @param  logWriter	ログ出力用のPrintWriter
	 */
	public void setLoggingWriter( final PrintWriter logWriter ) {
		this.logWriter = logWriter;
	}

	/**
	 * 画面表示用のPrintWriterを設定します。
	 * 通常は、引数の -dispFile=XXXX で指定しますが、直接 PrintWriter を
	 * 渡す必要があるケース(JSPなどで使用するケース)で使用します。
	 * 引数より、こちらの設定のほうが、優先されます。
	 * ※ JspWriter を渡す場合の PrintWriter は、flushing および、close 処理を
	 * 行わない NonFlushPrintWriter を設定してください。
	 *
	 * @param  dispWriter	画面表示用のPrintWriter
	 */
	public void setDisplayWriter( final PrintWriter dispWriter ) {
		this.dispWriter = dispWriter;
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) タイトル追加
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		return "[" + getClass().getName() + "]" + CR
				+ TAB + "Subject      : " + subject + CR
				+ TAB + "Log     File : " + logFile + CR
				+ TAB + "Display File : " + dispFile ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 */
	public String usage() {
		StringBuilder buf = new StringBuilder();

		buf.append( "Process_Logger は、画面出力、ファイルログ、エラーメールを管理する、" 			).append( CR );
		buf.append( "ロギング関係の LoggerProcess インターフェースの実装クラスです。"				).append( CR );
		buf.append( CR );
		buf.append( "MainProcess で使用されるログと、各種 Process で使用されるディスプレイを"		).append( CR );
		buf.append( "管理します。また、エラー発生時の、メール送信機能も、ここで用意します。"		).append( CR );
		buf.append( CR );
		buf.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR );
		buf.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR );
		buf.append( "繋げてください。"																).append( CR );
		buf.append( CR ).append( CR );

		buf.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_Logger().usage() );
	}
}
