/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.common;

import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.util.Cleanable;
import org.opengion.fukurou.util.HybsEntry;
// import org.opengion.fukurou.util.URLConnect;					// 6.9.0.0 (2018/01/31) URLConnect 廃止
import org.opengion.fukurou.util.HttpConnect;					// 6.9.0.0 (2018/01/31) 新規追加
import org.opengion.fukurou.system.LogWriter;
import java.util.Date;
import java.util.Map;
import java.io.IOException;

import java.util.EnumSet;

import javax.servlet.ServletContextListener;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContext;

import javax.servlet.ServletRegistration;
import javax.servlet.FilterRegistration;
import javax.servlet.DispatcherType;

import javax.servlet.annotation.WebListener;				// 6.3.4.0 (2015/08/01)

import org.apache.catalina.ContainerListener;				// 6.3.9.0 (2015/11/06)
import org.apache.catalina.ContainerEvent;					// 6.3.9.0 (2015/11/06)

/**
 * ServletContextListener を実装した、コンテキストの監視オブジェクトです。
 * これは、コンテキスト(Webアプリケーション)の起動/シャットダウンを監視できる。
 *
 * ServletContextListener は、
 *
 *      ConnectionFactory のコネクションプールへのアクセス／開放
 *      ResourceFactory   のリソース情報へのアクセス／開放
 *
 * の作業を行います。
 *
 * このリスナーは、WEB-INF/web.xml で、組み込みます。
 *
 * 【WEB-INF/web.xml】
 *
 *     &lt;listener&gt;
 *         &lt;listener-class&gt;
 *             org.opengion.hayabusa.common.HybsContextListener
 *         &lt;/listener-class&gt;
 *     &lt;/listener&gt;
 *
 * @og.group 初期化
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
@WebListener
public class HybsContextListener implements ServletContextListener , ContainerListener {
	// 4.0.0.0 (2007/10/26) ConnectionFactoryのhayabusa依存を切るために移動してきた
	static {
		final Cleanable clr = new Cleanable() {
			/**
			 * 初期化(クリア)します。
			 * 主に、キャッシュクリアで利用します。
			 */
			public void clear() {
				ConnectionFactory.realClose();
			}
		};
		SystemManager.addCleanable( clr );
	}

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public HybsContextListener() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 *  ServletContextListener インターフェースの実装
	 *
	 * Webアプリケーションがリクエストを処理できる状態になったことを
	 * リスナーに通知する。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) バージョンチェック、HybsSystem初期化追加
	 * @og.rev 3.4.0.0 (2003/09/01) Contextのpathによる、システムパラメータ の切り替え対応
	 * @og.rev 3.4.0.3 (2003/09/10) ServletContext の名称を、仮想パス名とする。
	 * @og.rev 3.5.3.1 (2003/10/31) システムパラメータ ファイルの読み取りタイミングを遅らせます。
	 * @og.rev 4.0.0.0 (2005/01/31) Ver4 のシステムパラメータ情報の取得処理を追加します。
	 * @og.rev 4.1.0.1 (2008/01/23) ログ出力先の設定処理を追加
	 * @og.rev 4.3.4.1 (2008/12/08) ログの環境変数対応
	 * @og.rev 6.3.8.3 (2015/10/03) プログラムによるWebアプリケーションの拡張
	 *
	 * @param  event コンテキストイベント
	 */
	@Override
	public void contextInitialized( final ServletContextEvent event ) {
		final ServletContext context = event.getServletContext();

		final Map<String,String> param = SystemParameter.makeSystemParameter( context );
		HybsSystem.setInitialData( param );			// 4.0.0 (2005/01/31)

		// 4.3.4.1 (2008/12/08) ログの環境変数対応
		LogWriter.init( HybsSystem.url2dir( System.getProperty( "SYS_LOG_URL" ,HybsSystem.sys( "SYS_LOG_URL" ) ) ) );

		// CONTEXT_INITIAL_CALL_URL で登録されたURLを実行します。
		// 処理は、contextInitialized が終了してから実行する必要があります。
		new Thread( new InitialCallURL() ).start();

		System.out.println( "-------" );

		// 6.3.8.3 (2015/10/03) プログラムによるWebアプリケーションの拡張
	//	addRegistration( context );
	}

	/**
	 *  ServletContextListener インターフェースの実装
	 *
	 * Webアプリケーションがシャットダウンされることを
	 * リスナーに通知する。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) キャッシュクリアメソッドを新規追加。
	 * @og.rev 3.3.3.3 (2003/08/06) HybsTimerTaskManager を終了時にキャンセルするロジックを追加。
	 * @og.rev 3.5.2.1 (2003/10/27) リンクエラー対策：永続化セッション(SESSIONS.ser)からオブジェクトを削除しておく。
	 * @og.rev 3.6.0.0 (2004/09/17) CalendarFactory.clear() を追加します。
	 * @og.rev 4.0.0.0 (2005/01/31) コンテキスト名の取り方を変更します。
	 * @og.rev 4.0.0.0 (2005/01/31) Cleanable インターフェースによる初期化処理
	 * @og.rev 4.0.0.0 (2005/01/31) HybsTimerTaskManager は、Cleanable インターフェースによる初期化
	 * @og.rev 4.1.0.0 (2007/12/27) GE12のクリア処理追加
	 * @og.rev 4.3.0.0 (2008/07/18) soffice.binのタスクを削除する処理を追加
	 * @og.rev 5.0.2.0 (2009/11/01) 再編成機能追加
	 *
	 * @param  event コンテキストイベント
	 */
	@Override
	public void contextDestroyed( final ServletContextEvent event ) {
		final String name = HybsSystem.sys( "CONTEXT_NAME" );
		System.out.println( "Context Destroyed [" + name + "]  " + new Date() );

		// 4.1.0.0 (2007/12/26) GE12からSystemParameterで設定したコンテキスト関係の情報
		SystemManager.clearGE12();

		// 4.0.0 (2005/01/31) Cleanable インターフェースによる初期化処理
		SystemManager.allClear( true ) ;

		// 4.3.0.0 (2008/07/18) soffice.binを全てkillします
	//	SystemManager.sofficeKill();

		SystemManager.sessionDestroyed();		// 3.5.2.1 (2003/10/27)

		SystemManager.deleteGUIAccessInfo();	// 5.0.2.0 (2009/11/01)

		System.out.println( "-------" );
	}

	/**
	 *  ContainerListener インターフェースの実装
	 *
	 * セッション固定攻撃対策として、認証の前後でセッションIDが変更されるようになりました。
	 * セッションIDの変更を検知したい場合は、ContainerListenerを実装する必要があります。
	 *
	 * ※ 使い方が分からないので、うまくイベントを拾えていません。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) 新規追加
	 *
	 * @param  event コンテナイベント
	 */
	@Override
	public void containerEvent( final ContainerEvent event ) {
		System.out.println( "【ContainerEvent:" + event.getType() + " : " + event.toString() + "】" );
	}

	/**
	 *  ServletContextListener による、ServletやFilter の動的登録。
	 *
	 * プログラムによるWebアプリケーションの拡張として、Servlet 3.0 より
	 * ServletやFilter を、ServletContextListenerのcontexInitializedメソッド から
	 * 動的に設定できるようになりました。
	 *
	 * @og.rev 6.3.8.3 (2015/10/03) プログラムによるWebアプリケーションの拡張
	 *
	 * @param  context ServletContextオブジェクト
	 */
	private void addRegistration( final ServletContext context ) {

		// web.xml では、filter の定義が先だったので、気持ち、先に設定しておきます。
		// ******* Filter *******

		// AccessStopFilter		(初期:false)
		final FilterRegistration frAS = context.addFilter( "AccessStopFilter", "org.opengion.hayabusa.filter.AccessStopFilter" );
		// frAS.setInitParameter( "startTime", "070000" );
		// frAS.setInitParameter( "stopTime" , "070000" );
		// frAS.setInitParameter( "filename" , "jsp/custom/stopFile.html" );
		frAS.addMappingForUrlPatterns( EnumSet.of(DispatcherType.REQUEST ), true, "/jsp/*" );

		// GZIPFilter			(初期:false)
		final FilterRegistration frGZ = context.addFilter( "GZIPFilter", "org.opengion.hayabusa.filter.GZIPFilter" );
		// frGZ.setInitParameter( "ipAddress", "192.168.,127.0.0.1" );
		// frGZ.setInitParameter( "debug"    , "false" );
		frGZ.addMappingForUrlPatterns( EnumSet.of(DispatcherType.REQUEST ), true, "/jsp/*" );

		// FileFilter			(初期:false)
		final FilterRegistration frFF = context.addFilter( "FileFilter", "org.opengion.hayabusa.filter.FileFilter" );
		// frFF.setInitParameter("saveDir", "filetemp/DIR/");
		frFF.addMappingForUrlPatterns( EnumSet.of(DispatcherType.REQUEST ), true, "/jsp/*" );

		// URLCheckFilter		(初期:false)
		final FilterRegistration frUC = context.addFilter( "URLCheckFilter", "org.opengion.hayabusa.filter.URLCheckFilter" );
		// frUC.setInitParameter( "filename"  , "jsp/custom/refuseAccess.html" );
		// frUC.setInitParameter( "debug"     , "false" );
		// frUC.setInitParameter( "ignoreURL" , "" );
		frUC.addMappingForUrlPatterns( EnumSet.of(DispatcherType.REQUEST ), true, "/jsp/*" );

		// URLHashFilter		(初期:true)
		final FilterRegistration frUH = context.addFilter( "URLHashFilter", "org.opengion.hayabusa.filter.URLHashFilter" );
		// frUH.setInitParameter( "filename", "jsp/custom/refuseAccess.html" );
		// frUH.setInitParameter( "initPage", "/jsp/index.jsp" );
		// frUH.setInitParameter( "debug"    , "false" );
		frUH.addMappingForUrlPatterns( EnumSet.of(DispatcherType.REQUEST ), true, "*.jsp" );

		// ******* Servlet *******

		// FileDownload
		final ServletRegistration srFD = context.addServlet( "fileDownload", "org.opengion.hayabusa.servlet.FileDownload" );
		srFD.addMapping( "/jsp/fileDownload" );

		// HybsAdmin
		final ServletRegistration srAD = context.addServlet( "admin", "org.opengion.hayabusa.servlet.HybsAdmin" );
		srAD.addMapping( "/jsp/admin" );

		// MakeImage
		final ServletRegistration srMI = context.addServlet( "makeImage", "org.opengion.hayabusa.servlet.MakeImage" );
		srMI.addMapping( "/jsp/makeImage" );

		// RemoteControlServlet
		final ServletRegistration srRC = context.addServlet( "remoteControl", "org.opengion.hayabusa.servlet.RemoteControlServlet" );
		srRC.addMapping( "/servlet/remoteControl" );
	}

	/**
	 * CONTEXT_INITIAL_CALL_URL を起動する為のスレッド内部クラスです。
	 *
	 * HybsContextListener が正常終了しないと、Tomcatが起動したことになっていない為、
	 * 通常のJSP処理が出来ません。
	 * ここでは、Tomcat起動時に初期処理URL(CONTEXT_INITIAL_CALL_URL)をコールする為に、
	 * 時間差を利用する為、スレッド化して実行させます。
	 * このスレッドは、２秒間スリープ後に、初期処理URLを呼び出します。
	 *
	 * @og.rev 4.2.2.0 (2008/05/22) 初期URLの接続ユーザーをシステムリソースより取得
	 * @og.rev 6.9.0.0 (2018/01/31) URLConnect 廃止、HttpConnect に置き換えます。
	 *
	 * @og.group ログイン制御
	 *
	 * @version  4.0
	 * @author   Kazuhiko Hasegawa
	 * @since    JDK5.0,
	 */
	private static final class InitialCallURL implements Runnable {
		/**
		 * スレッドの処理開始メソッド。
		 *
		 */
		public void run() {
			try {
				Thread.sleep( 2000 );
			}
			catch( final InterruptedException ex) {
				LogWriter.log( "InterruptedException:" + ex.getMessage() );
			}

			final String      name = HybsSystem.sys( "CONTEXT_NAME" );		// 6.9.0.0 (2018/01/31) 
			final HybsEntry[] urls = HybsSystem.sysEntry( "CONTEXT_INITIAL_CALL_URL" );
			final String  userPass = HybsSystem.sys( "CONTEXT_INITIAL_CALL_USERPASS" );

			boolean isCall = false;
			if( urls.length > 0 ) {
				for( int i=0; i<urls.length; i++ ) {
					final String url = urls[i].getValue();
					if( url == null || url.isEmpty() ) { continue; }
	//				final URLConnect conn = new URLConnect( url,userPass );
					final HttpConnect conn = new HttpConnect( url,userPass );				// 6.9.0.0 (2018/01/31) URLConnect 廃止、HttpConnect に置き換えます。
					try {
	//					conn.connect();
	//					final String msg = conn.getCode() + ":" + conn.getMessage() ;
						conn.readData();													// 6.9.0.0 (2018/01/31) 状態を確認する HttpHead は用意していない。GET で代用
	//					conn.disconnect();
						System.out.println( "    [" + name + "] URL[" + i + "]:" + url );
	//					System.out.println( "           " + msg );
						System.out.println( "           " + conn.getMessage() );			// 6.9.0.0 (2018/01/31)
						isCall = true ;
					}
					catch( final IOException ex ) {
						LogWriter.log( "    [" + name + "] URL[" + i + "]:" + url );
						LogWriter.log( "           " + ex.getMessage() );
					}
				}
			}
			if( isCall ) {
//				System.out.println( "  CONTEXT_INITIAL_CALL_URL" );
				System.out.println( "    [" + name + "] CONTEXT_INITIAL_CALL_URL" );		// 6.9.0.0 (2018/01/31)
				System.out.println( "-------" );
			}
		}
	}
}
