/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.common;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.concurrent.ConcurrentMap;							// 6.4.3.3 (2016/03/04)
import java.util.concurrent.ConcurrentHashMap;						// 6.4.3.1 (2016/02/12) refactoring
import java.util.List;
import java.util.Locale;

import javax.servlet.http.HttpSession;

import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;				// 6.1.0.0 (2014/12/26) refactoring
import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.util.Cleanable;
import org.opengion.fukurou.util.HybsDateUtil;					// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.system.Closer;
import org.opengion.fukurou.system.DateSet;						// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.system.LogWriter;
import org.opengion.fukurou.db.DBSimpleTable;

/**
 * Webアプリケーション全体で使用しているオブジェクト類のトータルの管理クラスです。
 *
 * SystemManager は、
 *
 * 		session オブジェクトの管理とアクセス／開放
 *
 * の作業を行います。
 *
 * 上記のクラス(staticメソッド)へのアクセスは、もちろん直接呼び出して
 * 操作することも可能ですが、サーバーのクリーンシャットダウン時やセッションの
 * 開放時、初期化処理など、ある種の統合的なトリガを受けて、関係するクラスに
 * イベントを伝えるようにすることで、Webアプリケーションサーバーとのやり取りを
 * 一元管理する目的で作成されています。
 *
 * @og.group 初期化
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class SystemManager {
	// 3.1.0.0 (2003/03/20) Hashtable を使用している箇所で、非同期でも構わない箇所を、HashMap に置換え。
	/** 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。  */
	private static final ConcurrentMap<String,UserSummary> USER_SMRY_MAP = new ConcurrentHashMap<>( BUFFER_MIDDLE );		// 6.4.1.1 (2016/01/16) map → USER_SMRY_MAP refactoring

	/** 4.0.0 (2005/01/31) Cleanable インターフェースを実装したオブジェクトを管理します。  */
	private static final List<Cleanable> CLEAR_LIST = new ArrayList<>() ;										// 6.4.1.1 (2016/01/16) clearList → CLEAR_LIST refactoring

	/** 4.3.6.2 (2009/04/15) Context終了時のみclear()される Cleanable ブジェクトを管理します。  */
	private static final List<Cleanable> CNTXT_CLEAR_LIST = new ArrayList<>() ;									// 6.4.1.1 (2016/01/16) contextClearList → CNTXT_CLEAR_LIST refactoring

	// 4.1.0.0 (2008/01/11) GE12クリア用
	// 4.3.6.6 (2009/05/15) ENGINE_INFOは削除しない
	/** エンジン個別(SYSTEM_ID='個別' KBSAKU='0' CONTXT_PATH='自身')パラメータの一括削除のクエリー	{@value}	*/
	private static final String DEL_SYS = "DELETE FROM GE12 WHERE SYSTEM_ID=? AND KBSAKU='0' AND CONTXT_PATH=? AND PARAM_ID != 'ENGINE_INFO'";

	// deleteGUIAccessInfo() メソッドでしか使用しない、定数宣言
	private static final int C_DEL_SYSTEM_ID		= 0;
	private static final int C_DEL_DYSET			= 1;

	/**
	 *  デフォルトコンストラクターをprivateにして、
	 *  オブジェクトの生成をさせないようにする。
	 *
	 */
	private SystemManager() {
	}

	/**
	 * session を記録します。
	 *
	 * 管理者権限で、強制ログアウトさせる場合などに、使用します。
	 * Servlet 2.1 では、HttpSessio#getSessionContext() より取り出した
	 * HttpSessionContextのgetSession(java.lang.String sessionId) で
	 * すべての session を取り出せましたが,Deprecated になりました。
	 * セキュリティー上、好ましくない処理ですので,注意して使用してください。
	 * common\session_init.jsp より登録します
	 *
	 * @og.rev 5.5.9.1 (2012/12/07) セッション作成時に、規定のキーでセッションIDを保存しておく。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap の not null制限のチェック追加
	 *
	 * @param   session Httpセッション
	 */
	public static void addSession( final HttpSession session ) {
		final String sessionID = session.getId();

		final UserSummary userInfo = (UserSummary)session.getAttribute( HybsSystem.USERINFO_KEY );
		if( sessionID != null && userInfo != null ) {
				USER_SMRY_MAP.put( sessionID,userInfo );
			session.setAttribute( HybsSystem.SESSION_KEY, sessionID );		// 5.5.9.1 (2012/12/07) セッションIDを保存
		}
	}

	/**
	 * session を削除します。
	 *
	 * 管理者権限で、強制ログアウトさせる場合などに、使用します。
	 * Servlet 2.1 では、HttpSessio#getSessionContext() より取り出した
	 * HttpSessionContextのgetSession(java.lang.String sessionId) で
	 * すべての session を取り出せましたが,Deprecated になりました。
	 * セキュリティー上、好ましくない処理ですので,注意して使用してください。
	 *
	 * @og.rev 5.5.9.1 (2012/12/07) セッション作成時に登録した規定のキーで userInfo を削除します。
	 * @og.rev 5.6.6.0 (2013/07/05) セッションの Attribute に SESSION_KEY で登録している sessionID も削除します。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 *
	 * @param   session Httpセッション
	 */
	public static void removeSession( final HttpSession session ) {

		final String sessionID = (String)session.getAttribute( HybsSystem.SESSION_KEY );	// 5.5.9.1 (2012/12/07) セッションIDを取り出し

		// 5.6.6.0 (2013/07/05) userInfo の USER_SMRY_MAP からの削除とuserInfo の clear を簡素化。
		if( sessionID != null ) {															// 6.4.3.3 (2016/03/04) ConcurrentHashMap の not null制限
			final UserSummary userInfo = USER_SMRY_MAP.remove( sessionID );
			if( userInfo != null ) { userInfo.clear(); }
		}

		// 5.6.6.0 (2013/07/05) セッションの Attribute に SESSION_KEY で登録している sessionID も削除します。
		session.removeAttribute( HybsSystem.USERINFO_KEY );
		session.removeAttribute( HybsSystem.SESSION_KEY );
	}

	/**
	 * すべてのシステムにログイン中のUserSummary オブジェクトを取得します。
	 *
	 * キーは、UserSummary の Attribute も含めた値が使用できます。
	 * 引数のキーは、内部で大文字に変換されたのち、内部キーとして使用されます。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 内部ロジック大幅変更
	 * @og.rev 5.6.6.0 (2013/07/05) Comparator の作り方を、簡素化します。キーの指定範囲も増やします。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 *
	 * @param   key ソートするキー項目を指定
	 * @param   direction ソートする方向[true:昇順/false:降順]
	 *
	 * @return	ログイン中のオブジェクト
	 */
	public static UserSummary[] getRunningUserSummary( final String key,final boolean direction ) {

		final UserSummary[] users = USER_SMRY_MAP.values().toArray( new UserSummary[USER_SMRY_MAP.size()] );

		if( key != null ) {
			final Comparator<UserSummary> comp = new ATTRI_Comparator( key.toUpperCase( Locale.JAPAN ),direction );
			Arrays.sort( users,comp );
		}

		return users ;
	}

	/**
	 * システムにログイン中の、すべてのセッション数を、取得します。
	 *
	 * ちなみに、不正なデータが存在した場合は、ここでMapから削除しておきます。
	 * ※ ConcurrentHashMap に変更したため、不正なデータ(ここでは、null データ)は、存在しない。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 *
	 * @return ログイン中の有効なすべてのセッション数
	 */
	public static int getRunningCount() {

		return USER_SMRY_MAP.size();
	}

	/**
	 * contextDestroyed 時に、すべてのセッションを、invalidate()します。
	 * 注意：キャッシュで内部管理していたセッションが、すべて無効化されてしまいます。
	 * よって、内部にセッションを管理しなくなったため、invalidate() もできません。
	 * 不具合が出るかもしれません。
	 *
	 * @og.rev 3.5.2.1 (2003/10/27) 新規作成
	 * @og.rev 4.0.0.0 (2005/01/31) セッション ⇒ UserSummary に変更
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap の not null制限のチェック追加
	 *
	 * @see		org.opengion.hayabusa.common.HybsContextListener
	 */
	/* default */ static void sessionDestroyed() {
		// 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。

		// 6.4.3.3 (2016/03/04) ConcurrentHashMap の not null制限のチェック追加

		final int ssCnt = USER_SMRY_MAP.size();

		USER_SMRY_MAP.forEach( (k,v) -> v.clear() );			// v の nullチェックは不要。なぜなら、キーにひも付かないnull値は登録できないため。
		USER_SMRY_MAP.clear();
		System.out.println( "  [" + ssCnt + "] Session Destroyed " );
	}

	/**
	 * 初期化したいオブジェクトを登録します。
	 * オブジェクトは、Cleanable インターフェースを実装しておく必要があります。
	 * 実際に、clear() する場合は、ここで登録した全てのオブジェクトの clear()
	 * メソッドが呼び出されます。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 * @og.rev 4.3.6.2 (2009/04/15) コンテキスト終了時のみのclear()対応
	 *
	 * @param obj インターフェースの実装
	 */
	public static void addCleanable( final Cleanable obj ) {
		addCleanable( obj, false );
	}

	/**
	 * 初期化したいオブジェクトを登録します。
	 * オブジェクトは、Cleanable インターフェースを実装しておく必要があります。
	 * 実際に、clear() する場合は、ここで登録した全てのオブジェクトの clear()
	 * メソッドが呼び出されます。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 * @og.rev 4.3.6.2 (2009/04/15) コンテキスト終了時のみのclear()対応
	 *
	 * @param obj インターフェースの実装
	 * @param flag trueの場合、コンテキスト停止時のみclear()を呼び出す
	 */
	public static void addCleanable( final Cleanable obj, final boolean flag ) {
		if( flag ) {
			synchronized( CNTXT_CLEAR_LIST ) {
				CNTXT_CLEAR_LIST.add( obj );
			}
		}
		else {
			 synchronized( CLEAR_LIST ) {
				CLEAR_LIST.add( obj );
			 }
		}
	}

	/**
	 * addCleanable( final Cleanable ) で登録したすべてのオブジェクトを初期化します。
	 * 処理は、Cleanable インターフェースの clear()メソッドを順次呼び出します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 * @og.rev 4.3.6.2 (2009/04/15) コンテキスト終了時のみのclear()対応
	 *
	 * @param	flag 完全終了時に、true
	 */
	public static void allClear( final boolean flag ) {
		final Cleanable[] clr ;
		synchronized( CLEAR_LIST ) {
			clr = CLEAR_LIST.toArray( new Cleanable[CLEAR_LIST.size()] );
			if( flag ) { CLEAR_LIST.clear() ; }		// contextDestroyed の場合のみ実行
		}
		// 登録の逆順で処理していきます。
		for( int i=clr.length-1; i>=0; i-- ) {
			clr[i].clear();
		}

		// コンテキスト停止時のみclear()
		if( flag ) {
			final Cleanable[] clr2 ;
			synchronized( CNTXT_CLEAR_LIST ) {
				clr2 = CNTXT_CLEAR_LIST.toArray( new Cleanable[CNTXT_CLEAR_LIST.size()] );
				CNTXT_CLEAR_LIST.clear();
			}
			// 登録の逆順で処理していきます。
			for( int i=clr2.length-1; i>=0; i-- ) {
				clr2[i].clear();
			}
		}
	}

	/**
	 * GE12からCONTXT PATHをhost:port/context/で登録している物を削除します。
	 * (web.xmlにTOMCAT_PORTを指定した場合に上記CONTEXT_PATHで登録されます)
	 *
	 * @og.rev 4.1.0.0 (2007/12/26) 新規作成
	 * @og.rev 5.5.4.5 (2012/07/27) 初期起動時のDB接続先は、RESOURCE_DBID とする。
	 * @og.rev 6.4.2.1 (2016/02/05) try-with-resources 文で記述。
	 *
	 * @see		org.opengion.hayabusa.common.HybsContextListener
	 */
	/* default */ static void clearGE12() {
		final String HOST_URL 		= HybsSystem.sys( "HOST_URL" );
		final String RESOURCE_DBID	= HybsSystem.sys( "RESOURCE_DBID" );	// 5.5.4.5 (2012/07/27) 初期起動時のDB接続先
		if( HOST_URL != null && !"**".equals( HOST_URL ) ) {
			Connection connection = null;
			try {
				connection = ConnectionFactory.connection( RESOURCE_DBID, null );	// 5.5.4.5 (2012/07/27) 初期起動時のDB接続先は、RESOURCE_DBID とする。
				try( final PreparedStatement pstmt = connection.prepareStatement( DEL_SYS ) ) {
					pstmt.setString( 1, HybsSystem.sys( "SYSTEM_ID" ) );
					pstmt.setString( 2, HOST_URL );
					final int delCnt = pstmt.executeUpdate();
					connection.commit();
					System.out.println( HOST_URL + " DELETE FROM GE12[" + delCnt + "]" );
				}
			}
			catch( final HybsSystemException e) {
				LogWriter.log( e );
			}
			catch( final SQLException e) {
				Closer.rollback( connection );
				LogWriter.log( e );
			}
			finally {
				ConnectionFactory.close( connection, null );
			}
		}
	}

	/**
	 * アクセス統計テーブル(GE15)の再編成を行います。
	 * データの保存期間については、システムリソースのACCESS_TOKEI_ALIVE_DAYSで指定します。
	 * データの作成された日時を基準として、上記の期間よりも古いデータは、物理削除されます。
	 * ACCESS_TOKEI_ALIVE_DAYSが指定されていない場合、データの削除は行われません。
	 *
	 * @og.rev 5.0.2.0 (2009/11/01) 新規作成
	 * @og.rev 5.5.5.1 (2012/08/07) リソース系DBID 付け忘れ対策
	 * @og.rev 6.4.2.0 (2016/01/29) HybsDateUtil.getDatePlus() と、DateSet.getDate( String ) を利用するように修正します。
	 *
	 * @see		org.opengion.hayabusa.common.HybsContextListener
	 */
	/* default */ static void deleteGUIAccessInfo() {
		final String aliveDays = HybsSystem.sys( "ACCESS_TOKEI_ALIVE_DAYS" );
		if( aliveDays == null || aliveDays.isEmpty() ) {
			return;
		}
		final String delBaseDate = HybsDateUtil.getDatePlus( DateSet.getDate( "yyyyMMdd" ), -1 * Integer.parseInt( aliveDays ) );	// 6.4.2.0 (2016/01/29)

		final String[] names = new String[] { "SYSTEM_ID","DYSET" };
		String[] values = new String[names.length];
		values[C_DEL_SYSTEM_ID		] = HybsSystem.sys( "SYSTEM_ID" );
		values[C_DEL_DYSET			] = delBaseDate + "000000";

		final String RESOURCE_DBID	= HybsSystem.sys( "RESOURCE_DBID" );	// 5.5.5.1 (2012/08/07) リソース系DBID 付け忘れ対応
		final DBSimpleTable dbTable = new DBSimpleTable( names );
		dbTable.setApplicationInfo( null );
		dbTable.setConnectionID( RESOURCE_DBID );	// 5.5.5.1 (2012/08/07)
		dbTable.setTable( "GE15" );
		dbTable.setWhere( "SYSTEM_ID = [SYSTEM_ID] and DYSET <= [DYSET]" );

		boolean okFlag = false;
		try {
			dbTable.startDelete();
			dbTable.execute( values );
			okFlag = true;
		}
		catch( final SQLException ex) {
			LogWriter.log( "  アクセス統計テーブル削除時にエラーが発生しました" );
			LogWriter.log( ex.getMessage() );
		}
		finally {
			final int cnt = dbTable.close( okFlag );
			System.out.println();
			System.out.println( "  アクセス統計テーブルから、[" + cnt + "]件、削除しました。" );
		}
	}

	/**
	 * UserSummary の Attribute で比較する Comparator 内部クラスの定義。
	 *
	 * key が、Attribute のキーになりますが、使用するのは、大文字化してからです。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) 新規追加
	 */
	private static final class ATTRI_Comparator implements Comparator<UserSummary>, Serializable {
		private static final long serialVersionUID = 566020130705L ;		// 5.6.6.0 (2013/07/05)
		private final String  key  ;
		private final boolean direct ;

		/**
		 * ソートの方向を引数にとるコンストラクタ。
		 *
		 * @og.rev 5.6.6.0 (2013/07/05) 新規追加
		 *
		 * @param	key			キー
		 * @param	direction	ソートの方向[true:昇順/false:降順]
		 */
		public ATTRI_Comparator( final String key,final boolean direction ) {
			this.key = key;
			direct   = direction;
		}

		/**
		 * getAttribute 比較メソッド
		 * インタフェース Comparable の 実装です。
		 *
		 * キーとして、getAttribute( String ) の取得結果を使用する為、null もあり得ます。その場合、equals 整合性は取れませんが、
		 * 処理としては、正常に動作するようにしておきます。つまり、null はもっとも小さい値とし、比較対象がともに null の
		 * 場合は、同じと判断します。
		 *
		 * @og.rev 5.6.6.0 (2013/07/05) 新規追加
		 *
		 * @param o1 比較対象の最初のオブジェクト
		 * @param o2 比較対象の 2 番目のオブジェクト
		 * @return	最初の引数が 2 番目の引数より小さい場合は負の整数、両方が等しい場合は 0、最初の引数が 2 番目の引数より大きい場合は正の整数
		 */
		public int compare( final UserSummary o1, final UserSummary o2 ) {
			final String key1 = o1.getAttribute( key );
			final String key2 = o2.getAttribute( key );

			int rtn ;
			if( key1 == null && key2 == null )	{ rtn =  0; }
			else if( key1 == null )				{ rtn = -1; }
			else if( key2 == null )				{ rtn =  1; }
			else								{ rtn = key1.compareTo( key2 ) ; }

			return direct ? rtn : -rtn;			// マイナス 0 が気になるが、まあ、良しとする。
		}
	}
}
