/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import java.util.Map;
import java.util.Set;											// 6.4.3.4 (2016/03/11)

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBColumnConfig;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.CodeData;
import org.opengion.hayabusa.resource.LabelData;
import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * 検索結果のカラムに対して様々な属性を付加するタグです。
 *
 * このタグでは、カラムリソースの属性を再設定できます。検索結果(DBTableModel)
 * には、カラムオブジェクト(DBColumn)を持っています。このオブジェクトは、
 * カラムリソースで設定した値を持っています。その個々の値を再設定することで、
 * カラムオブジェクトの属性をその都度設定しなおすことができます。
 * 最もよく使用するのは、must 属性の設定と、DBMENU エディターの設定です。
 * must属性は、予め必須項目として指定することが出来ない為、画面毎に再設定が
 * 必要です。ただし、ここでのmust属性は、カラムの表示色を変更するだけで、
 * 実際のnullチェックは、columnCheck タグのnullCheck属性で指定します。
 * DBMENU エディターは、データベースの属性から選択リストを作成する機能ですが、
 * これも、マスタ登録時は、テキストフィールドで入力し、使用時は、選択するケース
 * の様に、画面毎に異なる形式で使用したい場合があるためです。
 * このタグでは、カラムオブジェクトの属性以外に、HTMLの属性を付加する事も
 * できます。
 *
 *    6.7.6.0 (2017/03/17) [strictCheck属性]は、カラムＩＤの存在チェックを行うかどうかを指定します(初期値:true)
 *      true    カラムＩＤがDBTableModel に存在しない場合は、エラーになる。
 *      false   カラムＩＤがDBTableModel に存在しない場合は、無視する。
 *
 * @og.formSample
 * ●形式：&lt;og:ColumnEditor column=&quot;･･･&quot; ･･･ &gt; ･･･ &lt;/og:ColumnEditor &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:columnEditor
 *       column             【TAG】カラム名をセットします
 *       columns            【TAG】カラム名をCSV形式で複数セットします
 *       label              【TAG】ラベルリソースに定義されたラベルＩＤをセットします
 *       className          【TAG】カラムのクラス名(VARCHAR2,NUMBER など)をセットします。既存値:設定値対応
 *       size               【HTML】カラムの入力枠サイズをセットします(大きさだけで入力文字制限ではありません)
 *       viewSize           【TAG】カラムの表示サイズ(横幅)を文字数でセットします。
 *       maxlength          【HTML】カラムの入力文字制限数をセットします
 *       writable           【TAG】カラムが書き込み可能かどうか[true/false]をセットします(初期値:null)
 *       renderer           【TAG】カラムの表示用レンデラー({@og.doc03Link renderer Renderer_****})をセットします。既存値:設定値対応
 *       editor             【TAG】カラムの編集用エディター({@og.doc03Link editor Editor_****})をセットします。既存値:設定値対応
 *       dbType             【TAG】ＤＢタイプ({@og.doc03Link dbType DBType_****})をセットします。既存値:設定値対応
 *       codeName           【TAG】コードセレクトのユニークキー(コード名称)をセットします
 *       codeList           【TAG】コードセレクトのサブセットとなるコードをCSV形式で指定します
 *       codeGroup          【TAG】コードセレクトのサブセットとなるグループを指定します
 *       codeKeyVal         【TAG】新しいコードセレクトを作成する為の「キー:ラベル キー:ラベル」の文字列を指定します
 *       defaultVal         【TAG】初期値をセットします
 *       param              【TAG】パラメータをセットします
 *       roles              【TAG】ロールをセットします
 *       must               【TAG】必須入力を表す色に変えるかどうか[true/false]を指定します(初期値:false)
 *       mustAny            【TAG】選択必須入力(どれかひとつ必須)を表す色[true/mustAny/その他]を指定します(初期値:無指定)
 *       optionAttributes   【TAG】JavaScript などの HTML基本タグ以外の属性を、そのままタグとして使用します
 *       addNoValue         【TAG】セレクトメニューの値なしのOptionの有無[true/false]を指定します(初期値:falseと同等)
 *       addKeyLabel        【TAG】セレクトメニューの場合、キー:ラベル形式で表示するかどうか[true/false/null]を指定します(初期値:null)
 *       writeKeyLabel      【TAG】セレクトメニューの場合、キーとラベルを分けて出力するかどうか[true/false]を指定します(初期値:false)
 *       useLabelMap        【TAG】SQL文を指定してラベル情報を動的に読み込むかどうか[true/false]指定します(初期値:false)
 *       useSLabel          【TAG】MENU,DBMENUでSLABELを利用するかどうか[auto/true/false]を指定します(初期値:null)
 *       noDisplayVal       【TAG】指定の値を非表示にします(レンデラーのみ有効)(初期値:null)
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/application]を指定します(初期値:session)
 *       tableId            【TAG】(通常は使いません)sessionから所得する DBTableModelオブジェクトの ID(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])
 *       dbid               【TAG】(通常は使いません)データベースへ接続する場合のID(初期値:null)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       lang               【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       style              【HTML】この要素に対して適用させるスタイルシート(style)を設定します
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       onClick            【HTML】JavaScriptのイベント onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onChange           【HTML】JavaScriptのイベント onChange を設定します(例:onChange="renew('query.jsp','QUERY');")
 *       onBlur             【HTML】JavaScriptのイベント onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus            【HTML】JavaScriptのイベント onFocus を設定します
 *       ondblClick         【HTML】JavaScriptのイベント ondblClick を設定します
 *       onMouseDown        【HTML】JavaScriptのイベント onMouseDown を設定します
 *       onMouseUp          【HTML】JavaScriptのイベント onMouseUp を設定します
 *       onMouseMove        【HTML】JavaScriptのイベント onMouseMove を設定します
 *       onMouseOut         【HTML】JavaScriptのイベント onMouseOut を設定します
 *       onMouseOver        【HTML】JavaScriptのイベント onMouseOver を設定します
 *       onSelect           【HTML】JavaScriptのイベント onSelect を設定します
 *       onKeydown          【HTML】JavaScriptのイベント onKeydown を設定します
 *       onKeypress         【HTML】JavaScriptのイベント onKeypress を設定します
 *       onKeyup            【HTML】JavaScriptのイベント onKeyup を設定します
 *       autocomplete       【HTML5】入力候補を提示して入力内容を自動補完する[on/off]（初期値はon）。
 *       autofocus          【HTML5】指定した入力欄にカーソルが当たって自動的にフォーカスされます。
 *       pattern            【HTML5】正規表現で入力値のパターンを指定します。
 *       placeholder        【HTML5】入力欄に初期表示する内容を指定します。
 *       required           【HTML5】入力必須を指定します
 *       list               【HTML5】ユーザーに入力候補として提案するデータリストの要素のid属性の値を指定します。
 *       min                【HTML5】入力欄で入力できる最小値を指定します。
 *       max                【HTML5】入力欄で入力できる最大値を指定します。
 *       step               【HTML5】入力欄で刻むステップ値を指定する。
 *       type               【HTML5】HTML5追加分[text/search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
 *       eventColumn        【TAG】イベントカラム(親カラム)のセット
 *       eventValue         【TAG】イベントValueのＳＱＬ分
 *       eventURL           【TAG】eventColumn利用時に呼ばれるJSPのURL
 *       eventCallback      【TAG】eventColumn実行後のcallbak関数指定を指定します
 *       noResource         【TAG】ResourceManager を使用しないかどうか[true/false]を指定します(初期値:false)
 *       strictCheck        【TAG】(通常は使いません)カラムＩＤの存在チェックを行うかどうか[true/false]を指定します(初期値:true)
 *       stringOutput       【TAG】レンデラ利用のExcel出力時にセルタイプを文字列にするかどうか(初期値:true)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:columnEditor&gt;
 *
 * ●使用例：
 *     ・必須表示の場合
 *         &lt;og:columnEditor column=&quot;NOSYN&quot; must=&quot;true&quot; /&gt;
 *
 *     ・DBからのセレクトメニューの場合
 *         &lt;og:columnEditor column=&quot;CDBK&quot; editor=&quot;DBMENU&quot;&gt;
 *                 select CDBK,CDBK||&#39;:&#39;||NMBK from DB02 order by 1
 *         &lt;/og:columnEditor&gt;
 *
 *     ・親子関係のプルダウンを作る場合
 *       ({&#064;xxx.value}で同じ行のxxxカラムで選択した値を取得します)
 *         &lt;og:columnEditor column=&quot;CDBK&quot; editor=&quot;DBMENU&quot;
 *              eventColumn="OYA"&gt;
 *                 select CDBK,CDBK||&#39;:&#39;||NMBK from DB02
 *                   where CLM = {&#064;OYA.value} order by 1
 *         &lt;/og:columnEditor&gt;
 *
 *
 * @og.rev 3.5.1.0 (2003/10/03) 親クラスをCommonTagSupportからHTMLTagSupportに変更。
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ColumnEditorTag extends HTMLTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.8.1.4 (2017/08/25)" ;
	private static final long serialVersionUID = 681420170825L ;

	// 5.7.1.0 (2013/12/06) HTML5 で新たに追加された、type を追加
	// 6.3.4.0 (2015/08/01) String から String配列 に置き換え

	// 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	private static final Set<String> TYPE_SET = new ArraySet<>( "text","search","tel","url","email","datetime","date","month","week","time","datetime-local","number","range","color" );

	private static final String DEFAULT_ADD_NO_VALUE= "true";	// 3.7.0.5 (2005/04/11)

	private transient	LabelData				labelData	;

	/** 6.4.3.1 (2016/02/12) 取得元が ConcurrentHashMap で返す。  */
	private transient	Map<String,LabelData>	labelMap	;	// 6.3.9.0 (2015/11/06) transient 追加

	private String		tableId			= HybsSystem.TBL_MDL_KEY;
	private String[]	columns			;		// 3.7.0.5 (2005/04/11)
	private boolean		allColumns		;		// 3.8.8.5 (2007/03/09)
	private String		className		;
	private String		fieldSize		;		// 6.2.0.0 (2015/02/27)
	private String		viewSize		;		// 6.2.0.0 (2015/02/27) viewSize 属性の追加
	private String		maxlength		;		// 3.5.5.5 (2004/04/23)
	private String		writable		;
	private String		renderer		;
	private String		editor			;
	private String		dbType			;
	private String		codeName		;		// 3.5.5.7 (2004/05/10)
	private String		codeList		;		// 5.1.9.0 (2010/08/01) 新規追加(Codeリソースのサブセット)
	private String		codeGroup		;		// 5.1.9.0 (2010/08/01) 新規追加(Codeリソースのサブセット)
	private String		codeKeyVal		;		// 6.3.1.1 (2015/07/10) 新規追加(新しいコードセレクトを作成する為の KEY:VAL 文字列)
	private String		defaultVal		;
	private String		parameter		;
	private String		dbid			;
	private String		addNoValue		;		// 3.5.5.9 (2004/06/07)
	private String		addKeyLabel		;		// 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか
	private boolean		writeKeyLabel	;		// 6.2.3.0 (2015/05/01)

	private String 	eventColumn			;		// 4.3.6.0 (2009/04/01) イベントカラム
	private String	eventValue			;		// 6.3.4.0 (2015/08/01) eventValue 追加
	private String	rawParameter		;		// 4.3.6.0 (2009/04/01) 生パラメータ
	private String	eventURL			;		// 4.3.6.0 (2009/04/01) イベントURL

	// 4.3.4.0 (2008/12/01) ラベルデータの動的取得に対応します。
	private boolean		useLabelMap		;		// 4.3.4.0 (2008/12/01)
	private String		useSLabel		;		// 5.6.2.3 (2013/03/22) MENU,DBMENUでSLABELを利用するかどうか
	private String		noDisplayVal	;		// 5.6.2.3 (2013/03/22) 非表示文字の設定

	// 5.7.3.0 (2014/02/07) ResourceManager を使用しないかどうか[true/false]を指定
	private boolean		noResource		;
	private boolean		strictCheck		= true;	// 6.7.6.0 (2017/03/17) カラムＩＤの存在チェックを行うかどうかを指定します。

	private boolean		stringOutput	= HybsSystem.sysBool( "USE_STRING_EXCEL_OUTPUT" );		// 5.7.6.3 (2014/05/23)

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ColumnEditorTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig#getColumnConfig() メソッド廃止
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 * @og.rev 3.5.1.0 (2003/10/03) 親クラスをCommonTagSupportからHTMLTagSupportに変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 入力欄の大きさを指定する viewSize 属性を追加。
	 * @og.rev 3.5.5.7 (2004/05/10) CodeSelection を、codeName を元に構築できるように設定します。
	 * @og.rev 3.5.5.9 (2004/06/07) addNoValue が、設定された場合のみ、config に設定します
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			if( "DBMENU".equalsIgnoreCase(editor) ) {
				className = "VARCHAR2";
				renderer  = "DBMENU";
				editor    = "DBMENU";

				if( addNoValue == null ) {
					addNoValue = DEFAULT_ADD_NO_VALUE ;
				}
			}

			return EVAL_BODY_BUFFERED ;
		}
		return SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 2.3.1.3 (2003/02/08) ラベル、サイズのみでも変更できるように修正
	 * @og.rev 3.0.0.1 (2003/02/14) DBMENU作成時に、ラベルの最後のスペースを除去する。
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig#getColumnConfig() メソッド廃止
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 * @og.rev 3.5.0.0 (2003/09/17) キー、ラベルともに、""(ダブルコーテーション)で囲うように変更。
	 * @og.rev 3.5.5.7 (2004/05/10) このタグでSQLを発行するのではなく、DBMENU の DBColumnを作成する。
	 * @og.rev 3.5.5.9 (2004/06/07) addNoValue が null の場合は、初期値を設定します。
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 * @og.rev 3.8.6.3 (2006/11/30) SQL 文の前後のスペースを取り除きます。
	 * @og.rev 4.3.4.0 (2008/12/01) ラベルデータの動的取得に対応します。
	 * @og.rev 4.3.6.0 (2009/04/01) イベントカラム対応
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.5.4.0 (2012/07/02) 予約語対応
	 * @og.rev 6.3.1.1 (2015/07/10) BodyString,BodyRawStringは、CommonTagSupport で、trim() します。
	 * @og.rev 6.4.0.5 (2016/01/09) useLabelMap="true" 時のSQL文の実行は、dbid を使用して行う。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		final String sql = getBodyString();
		if( sql != null && sql.length() > 0 ) {
			if( useLabelMap )	{
				// 6.4.0.5 (2016/01/09) useLabelMap="true" 時のSQL文の実行は、dbid を使用して行う。
				labelMap = getResource().getLabelMap( sql.trim() , dbid );		// 4.3.4.0 (2008/12/01)
			}
			else				{
				if( eventColumn != null && eventColumn.length() >0 ){  // 4.3.6.0 (2009/04/01)
					rawParameter = getReservedParameter( getBodyRawString() );		// 6.3.1.1 (2015/07/10)
				}
				parameter = sql.trim() ;
			}
		}

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 全カラム が指定されたことにする allColumns 属性を追加
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	 * @og.rev 4.3.4.0 (2008/12/01) カラム指定のワイルドカード対応
	 * @og.rev 5.2.1.0 (2010/10/01) must , mustAny 属性の処理を、HTMLTagSupport に移します。
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 * @og.rev 6.7.6.0 (2017/03/17) strictCheck 追加。
	 * @og.rev 6.8.1.4 (2017/08/25) must="false"で、config.setEditorAttributes で、属性を追加するようになった為の取り消しの対応。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( !useTag() ) { return EVAL_PAGE; }

		final DBTableModel table = (DBTableModel)getObject( tableId );
		if( table == null ) { return EVAL_PAGE; }

		if( columns != null ) {
			// 3.8.8.5 (2007/03/09) 全カラム が指定
			final String mustType = getMustType();	// 5.2.1.0 (2010/10/01)
			if( allColumns ) {
				final int len = table.getColumnCount();
				for( int i=0; i<len; i++ ) {

					// 4.3.4.0 (2008/12/01)
					final DBColumn dbColumn = table.getDBColumn( i );
					for( int j=0; j<columns.length; j++ ) {
						final String prefix = columns[j].replace( "*", "" );
						if( dbColumn.getName().indexOf( prefix ) == 0 ) { // 前方一致のみ対応
							// 6.8.1.4 (2017/08/25) must="false"で、config.setEditorAttributes で、属性を追加するようになった為の取り消しの対応。
							final DBColumnConfig config = getModifyConfig( dbColumn );

							// 4.1.2.1 (2008/03/13)
							if( mustType != null ) {
								table.addMustType( i,mustType );
								// 6.8.1.4 (2017/08/25) must="false" で、must の取り消し
								if( "false".equalsIgnoreCase( mustType ) ) {
									config.removeEditorAttributes( "class" , "must" );
								}
							}

							table.setDBColumn( i,new DBColumn( config ) );

							break;
						}
					}
				}
			}
			else {
				for( int i=0; i<columns.length; i++ ) {
					final int clmNo = table.getColumnNo( columns[i],strictCheck );		// 6.7.6.0 (2017/03/17)
					if( clmNo < 0 ) { continue; }										// 6.7.6.0 (2017/03/17) 

					final DBColumn dbColumn = table.getDBColumn( clmNo );

					final DBColumnConfig config = getModifyConfig( dbColumn );
					// 6.8.1.4 (2017/08/25) must="false"で、config.setEditorAttributes で、属性を追加するようになった為の取り消しの対応。

					// 4.1.2.1 (2008/03/13)
					if( mustType != null ) {
						table.addMustType( clmNo,mustType );
						// 6.8.1.4 (2017/08/25) must="false" で、must の取り消し
						if( "false".equalsIgnoreCase( mustType ) ) {
							config.removeEditorAttributes( "class" , "must" );
						}
					}

					table.setDBColumn( clmNo,new DBColumn( config ) );

				}
			}
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグ文字列を作成します。
	 *
	 * ColumnEditor としては、HTMLTagSupport を継承しているため、
	 * このメソッドを定義する必要がありますが、内部的には使用していません。
	 * 各種属性は、入力する必要がある為、継承しているだけです。
	 * このクラスでは、このメソッドは、使われておりません。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) 新規追加
	 *
	 * @return  タグ文字列(常に null を返す)
	 */
	@Override
	protected String makeTag() {
		return null;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 *
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.1.0 (2003/10/03) 親クラスをCommonTagSupportからHTMLTagSupportに変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 入力欄の大きさを指定する viewSize 属性を追加。
	 * @og.rev 3.5.5.7 (2004/05/10) codeName 属性を追加。
	 * @og.rev 3.5.5.9 (2004/06/07) addNoValue 属性を修正。
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 全カラム が指定されたことにする allColumns 属性を追加
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	 * @og.rev 4.3.4.0 (2008/12/01) ラベルデータの動的取得に対応します。
	 * @og.rev 5.1.9.0 (2010/08/01) codeList , codeGroup 属性を追加。(Codeリソースのサブセット)
	 * @og.rev 5.2.1.0 (2010/10/01) must , mustAny 属性の処理を、HTMLTagSupport に移します。
	 * @og.rev 5.6.2.3 (2013/03/22) useSLabel 属性を追加。(MENU,DBMENUでSLABELを利用するかどうか)
	 * @og.rev 5.6.2.3 (2013/03/22) 非表示文字の設定。noDisplayVal 属性を追加。
	 * @og.rev 5.7.3.0 (2014/02/07) noResource 属性を追加。ResourceManager を使用しないかどうか
	 * @og.rev 5.7.6.3 (2014/05/23) stringOutput追加
	 * @og.rev 6.0.4.0 (2014/11/28) addKeyLabel 属性を追加
	 * @og.rev 6.2.0.0 (2015/02/27) viewSize 属性の追加
	 * @og.rev 6.2.3.0 (2015/05/01) キー,ラベルで分離して出力するかどうか。writeKeyLabel 属性を追加
	 * @og.rev 6.3.1.1 (2015/07/10) codeKeyVal 属性を追加
	 * @og.rev 6.3.4.0 (2015/08/01) eventValue 追加
	 * @og.rev 6.7.6.0 (2017/03/17) strictCheck 追加。
	 */
	@Override
	protected void release2() {
		super.release2();
		tableId			= HybsSystem.TBL_MDL_KEY;
		columns			= null;		// 3.7.0.5 (2005/04/11)
		allColumns		= false;	// 3.8.8.5 (2007/03/09)
		labelData		= null;		// 4.0.0 (2005/01/31)
		className		= null;
		fieldSize		= null;		// 6.2.0.0 (2015/02/27)
		viewSize		= null;		// 6.2.0.0 (2015/02/27)
		maxlength		= null;		// 3.5.5.5 (2004/04/23)
		writable		= null;
		renderer		= null;
		editor			= null;
		dbType			= null;
		codeName		= null;		// 3.5.5.7 (2004/05/10)
		codeList		= null;		// 5.1.9.0 (2010/08/01) 新規追加(Codeリソースのサブセット)
		codeGroup		= null;		// 5.1.9.0 (2010/08/01) 新規追加(Codeリソースのサブセット)
		codeKeyVal		= null;		// 6.3.1.1 (2015/07/10) 新規追加(新しいコードセレクトを作成する為の KEY:VAL 文字列)
		defaultVal		= null;
		parameter		= null;
		dbid			= null;
		addNoValue		= null;		// 3.5.5.9 (2004/06/07)
		addKeyLabel		= null;		// 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか
		writeKeyLabel	= false;	// 6.2.3.0 (2015/05/01) キー,ラベルで分離して出力するかどうか
		useLabelMap		= false;	// 4.3.4.0 (2008/12/01)
		useSLabel		= null;		// 5.6.2.3 (2013/03/22) MENU,DBMENUでSLABELを利用するかどうか
		noDisplayVal	= null;		// 5.6.2.3 (2013/03/22) 非表示文字の設定
		labelMap		= null;		// 4.3.4.0 (2008/12/01)
		eventColumn		= null; 	// 4.3.6.0 (2009/04/01)
		eventValue  	= null;  	// 6.3.4.0 (2015/08/01)
		rawParameter	= null;		// 4.3.6.0 (2009/04/01)
		eventURL		= null;		// 4.3.6.0 (2009/04/01)
		noResource		= false;	// 5.7.3.0 (2014/02/07) noResource 属性を追加。
		strictCheck		= true;		// 6.7.6.0 (2017/03/17)
		stringOutput	= HybsSystem.sysBool( "USE_STRING_EXCEL_OUTPUT" );		// 5.7.6.3 (2014/05/23)
	}

	/**
	 * カスタマイズされた DBColumnConfig を作成します。
	 *
	 * DBColumnConfig は、DBColumn より取得します。この DBColumnConfig に、
	 * 属性で指定された値をセットすることで、カスタマイズを行います。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 新規追加
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	 * @og.rev 4.3.4.0 (2008/12/01) ラベルデータの動的取得に対応します。
	 * @og.rev 4.3.6.0 (2009/04/01) evnetColumn対応
	 * @og.rev 5.1.8.0 (2010/07/01) メソッド名変更(setDefValue ⇒ setDefault)
	 * @og.rev 5.1.9.0 (2010/08/01) codeList ,codeGroup 属性を追加(Codeリソースのサブセット)
	 * @og.rev 5.6.2.3 (2013/03/22) useSLabel 属性を追加。(MENU,DBMENUでSLABELを利用するかどうか)
	 * @og.rev 5.6.2.3 (2013/03/22) 非表示文字の設定。noDisplayVal 属性を追加。
	 * @og.rev 5.7.3.0 (2014/02/07) noResource 属性を追加。ResourceManager を使用しないかどうか
	 * @og.rev 5.7.6.2 (2014/05/16) IEのHTML5機能が無効の場合の処理
	 * @og.rev 5.7.6.3 (2014/05/23) stringOutput追加
	 * @og.rev 6.0.4.0 (2014/11/28) addKeyLabel 属性を追加
	 * @og.rev 6.2.0.0 (2015/02/27) viewSize 属性の追加
	 * @og.rev 6.2.3.0 (2015/05/01) maxlength 変数は、0 の場合は、無制限になります。
	 * @og.rev 6.3.1.0 (2015/06/28) 個人リソースのUSE_ADD_KEY_LABELで初期値設定
	 * @og.rev 6.3.1.1 (2015/07/10) codeKeyVal 属性を追加
	 * @og.rev 6.3.4.0 (2015/08/01) eventValue 追加
	 * @og.rev 5.9.18.1 (2017/03/24) DATALIST_R用のIEのHTML5機能が無効の場合の処理を追加
	 *
	 * @param	dbColumn	カラムオブジェクト
	 *
	 * @return  カスタマイズされた DBColumnConfig
	 */
	private DBColumnConfig getModifyConfig( final DBColumn dbColumn ) {
		// 5.7.3.0 (2014/02/07) noResource 属性を追加
		if( noResource ) {
			labelData = new LabelData( dbColumn.getName() );
			renderer  = "LABEL";
		}

		final DBColumnConfig config = dbColumn.getConfig();

		// 5.7.6.2 (2014/05/16) IEのHTML5機能が無効の場合の処理
		final String ieHTML5 = (String)getSessionAttribute( HybsSystem.IE_HTML5_KEY );
		if( "FALSE".equalsIgnoreCase( ieHTML5 ) ) {
			if( "DATALIST".equalsIgnoreCase( editor ) ||
				"DATALIST".equalsIgnoreCase( config.getEditor() ) ) {
					editor = "INDBMENU";
			}
			// 5.9.18.1 (2017/03/24) IEのHTML5機能が無効の場合の処理(DATALIST_R用)
			if( "DATALIST_R".equalsIgnoreCase( editor ) ||
				"DATALIST_R".equalsIgnoreCase( config.getEditor() )){
					editor = "INMENU";
			}
		}

		// 6.3.1.0 (2015/06/28) 個人リソースのUSE_ADD_KEY_LABELで初期値設定
		if( addKeyLabel	 == null ) { addKeyLabel = nval( sys( "USE_ADD_KEY_LABEL" ) , null ); }		// 4:個人設定可

		final String lang = get( "lang" );	// 4.0.0 (2005/01/31) super クラスで受けます。

		if( lang		!= null ) { config.setLang(		 lang		); }
		if( labelData	!= null ) { config.setLabelData( labelData	); }		// 4.0.0 (2005/01/31)
		if( className	!= null ) { config.setClassName( className	); }
		if( fieldSize	!= null ) { config.setFieldSize( fieldSize	); }		// 6.2.0.0 (2015/02/27) メソッド名変更
		if( viewSize	!= null ) { config.setViewLength( viewSize	); }		// 6.2.0.0 (2015/02/27) 新規追加
		if( maxlength	!= null ) { config.setMaxlength( maxlength	); }		// 3.5.5.5 (2004/04/23)
		if( writable	!= null ) { config.setWritable( writable	); }
		if( renderer	!= null ) { config.setRenderer( renderer	); }
		if( editor		!= null ) { config.setEditor( editor 		); }
		if( dbType		!= null ) { config.setDbType( dbType 		); }
		if( defaultVal	!= null ) { config.setDefault( defaultVal	); }		// 5.1.8.0 (2010/07/01)
		if( parameter	!= null ) { config.setParameter( parameter	); }
		if( dbid		!= null ) { config.setDbid( dbid			); }
		if( eventColumn != null ) { config.setEventColumn( eventColumn ); }		// 4.3.6.0 (2009/04/01)
		if( eventValue  != null ) { config.setEventValue(  eventValue  ); }		// 6.3.4.0 (2015/08/01)
		if( eventURL	!= null ) { config.setEventURL( eventURL	   ); }		// 4.3.6.0 (2009/04/01)
		if( rawParameter != null ){ config.setRawParameter( rawParameter ); }	// 4.3.6.0 (2009/04/01)
		if( useSLabel	 != null ) { config.setUseSLabel( useSLabel  ); }		// 5.6.2.3 (2013/03/22)
		if( noDisplayVal != null ) { config.setNoDisplayVal( noDisplayVal ); }	// 5.6.2.3 (2013/03/22)
		if( addKeyLabel	 != null ) { config.setAddKeyLabel( addKeyLabel	); }	// 6.0.4.0 (2014/11/28)

		if( writeKeyLabel ) { config.setWriteKeyLabel( writeKeyLabel ); }		// 6.2.3.0 (2015/05/01)

		// 4.0.0 (2005/11/30) ロールを指定
		final String roles = get( "roles" );
		if( roles != null ) { config.setRoles(     roles      ); }

		// 4.0.0 (2005/01/31)
		if( codeName != null ) {
			final CodeData codeData = getResource().getCodeData( codeName ) ;
			config.setCodeData( codeData );
		}

		// codeName で指定のリソースでも、サブセットが作成できます。
		// 5.1.9.0 (2010/08/01) codeList 属性を追加(Codeリソースのサブセット)
		if( codeList != null ) {
			final CodeData codeData = config.getCodeData() ;
			if( codeData != null ) {
				config.setCodeData( codeData.subsetList( codeList ) );
			}
		}

		// 5.1.9.0 (2010/08/01) codeGroup 属性を追加(Codeリソースのサブセット)
		if( codeGroup != null ) {
			final CodeData codeData = config.getCodeData() ;
			if( codeData != null && codeData.useGroup() ) {
				config.setCodeData( codeData.subsetGroup( codeGroup ) );
			}
		}

		// 6.3.1.1 (2015/07/10) 新規追加(新しいコードセレクトを作成する為の KEY:VAL 文字列)
		if( codeKeyVal != null ) {
			config.setParameter( codeKeyVal );
			config.setRenderer( "KVMENU" );
			config.setEditor(   "KVMENU" );
		}

		// 3.5.5.9 (2004/06/07)
		// 注意：addNoValue は、互換性の関係で、値が指定された場合のみ、
		// config に設定します。
		// editor="DBMENU" が、指定された場合は、addNoValue が指定されていない場合のみ、
		// 従来との互換性の関係より、このクラスのデフォルト(true)を、config に設定します。
		if( addNoValue != null ) {
			// Boolean.parseBoolean は、引数がnullの場合は、falseを返すが、値が設定されている時のみセットするので、nullチェックしている。
			config.setAddNoValue( Boolean.parseBoolean( addNoValue ) );	// 6.1.0.0 (2014/12/26) refactoring
		}

		// 3.5.1.0 (2003/10/03) 親クラスをCommonTagSupportからHTMLTagSupportに変更。
		config.setEditorAttributes( getAttributes() );

		// 4.3.4.0 (2008/12/01) ラベルデータの動的取得に対応します。
		if( useLabelMap && labelMap != null ) {
			final LabelData ldata = labelMap.get( dbColumn.getName() );
			if( ldata != null ) {
				config.setLabelData( ldata );
			}
		}

		// 5.7.6.3 (2014/05/23)
		config.setStringOutput( stringOutput );

		return config ;
	}

	/**
	 * 【TAG】カラム名をセットします。
	 *
	 * @og.tag
	 * カラム名をセットします。
	 * columns と同時にはセットできません。
	 * 値に"*"を指定することで、全カラムを対象とすることができます。
	 * また、"PN*,CDK*"等のように前方一致でのワイルドカード指定も可能です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 全カラム が指定されたことにする allColumns 属性を追加
	 *
	 * @param	clm ネーム
	 */
	public void setColumn( final String clm ) {
		if( columns != null ) {
			final String errMsg = "columns との同時セットは出来ません。: " + CR
						+ "column=[" + clm + "] , "
						+ "columns=[" + StringUtil.array2csv( columns ) + "]";
			throw new HybsSystemException( errMsg );
		}
		final String column = getRequestParameter( clm );
		if( column != null && column.length() > 0 ) {
			columns = new String[] { column };
			allColumns = column.indexOf( '*' ) >= 0 ;	// 3.8.8.5 (2007/03/09)
		}
	}

	/**
	 * 【TAG】カラム名をCSV形式で複数セットします。
	 *
	 * @og.tag
	 * カラム名をセットします。
	 * column と同時にはセットできません。
	 * 分解方法は、通常のパラメータ取得後に、CSV分解します。
	 * 値に"*"を指定することで、全カラムを対象とすることができます。
	 * また、"PN*,CDK*"等のように前方一致でのワイルドカード指定も可能です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 通常のパラメータ取得後に、CSV分解に戻します。
	 * @og.rev 3.8.8.5 (2007/03/09) 全カラム が指定されたことにする allColumns 属性を追加
	 *
	 * @param	clms カラム名(CSV形式)
	 */
	public void setColumns( final String clms ) {
		if( columns != null ) {
			final String errMsg = "column との同時セットは出来ません。: " + CR
						+ "column=[" + StringUtil.array2csv( columns ) + "] , "
						+ "columns=[" + clms + "]" ;
			throw new HybsSystemException( errMsg );
		}
		final String column = nval( getRequestParameter( clms ),null );
		if( column != null ) {
			allColumns = column.indexOf( '*' ) >= 0 ;		// 3.8.8.5 (2007/03/09)
			columns = StringUtil.csv2Array( column );
			if( columns.length == 0 ) { columns = null; }
		}
	}

	/**
	 * 【TAG】ラベルリソースに定義されたラベルＩＤをセットします。
	 *
	 * @og.tag
	 * ラベルをセットします。
	 *
	 * @og.rev 2.3.1.2 (2003/01/28) ラベルリソースからラベル名を取得するように修正
	 * @og.rev 4.0.0.0 (2005/01/31) キーより、ラベルデータオブジェクトを取得します。
	 *
	 * @param	lbl ラベル
	 */
	public void setLabel( final String lbl ) {
		labelData = getResource().getLabelData( getRequestParameter( lbl ) ) ;	// 4.0.0 (2005/01/31)
	}

	/**
	 * 【TAG】カラムのクラス名(VARCHAR2,NUMBER など)をセットします。
	 *
	 * @og.tag
	 * カラムのクラス名をセットします。
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	cls クラス
	 */
	public void setClassName ( final String cls ) {
		className = nval( getRequestParameter( cls ),className );
	}

	/**
	 * 【HTML】最大入力文字数を指定します(入力可能文字数で、バイト数ではありません)。
	 *
	 * @og.tag
	 * カラムの入力文字制限数をセットします。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) maxlength 変数は、0 の場合は、無制限になります。
	 *
	 * @param	len 入力文字制限数
	 */
	public void setMaxlength( final String len ) {
		maxlength = nval( getRequestParameter( len ),maxlength );

		// 6.2.3.0 (2015/05/01) maxlength 変数は、0 の場合は、無制限になります。
		if( "0".equals( maxlength ) ) { set( "NO_MAXLEN", "true" ); }
	}

	/**
	 * 【TAG】カラムの入力枠サイズ(横幅)を文字数でセットします(入力文字制限ではありません)。
	 *
	 * @og.tag
	 * カラムのサイズをセットします。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 入力欄の大きさを指定する viewSize 属性を追加。
	 *
	 * @param	siz サイズ
	 */
	public void setSize( final String siz ) {
		fieldSize = nval( getRequestParameter( siz ),null );		// 6.2.0.0 (2015/02/27)
	}

	/**
	 * 【TAG】カラムの表示サイズ(横幅)を文字数でセットします。
	 *
	 * @og.tag
	 * 入力枠ではなく、表示する時の文字数制限に使われます。
	 * maxLength は、DB上の桁数(バイト数)ですが、viewSize は、文字数です。
	 * SLABEL の簡易バイト数計算(全角2Byte)でもなく、文字数計算になります。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) viewSize 属性の追加
	 *
	 * @param   len 表示サイズ(横幅)
	 */
	public void setViewSize( final String len ) {
		viewSize = nval( getRequestParameter( len ),null );
	}

	/**
	 * 【TAG】カラムが書き込み可能かどうか[true:可能/false:不可]を設定します(初期値:null)。
	 *
	 * @og.tag
	 * カラムの書き込み可能を設定します。
	 * 初期値は、null:無指定 です。
	 *
	 * @param	flag 書き込み許可 [true:許可/false:不可]
	 */
	public void setWritable ( final String flag ) {
		writable = nval( getRequestParameter( flag ),writable );
	}

	/**
	 * 【TAG】カラムの表示用レンデラー({@og.doc03Link renderer Renderer_****})をセットします。
	 *
	 * @og.tag
	 * これは、plugin.column 以下の Renderer_**** クラスの **** を与えます。
	 * これらは、CellRenderer インターフェースを継承したサブクラスです。
	 * 属性クラス定義の {@link org.opengion.hayabusa.db.CellRenderer Renderer} を参照願います。
	 * {@og.doc03Link renderer Renderer_**** クラス}
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	rdr 表示用レンデラー (Renderer_**** の ****)
	 * @see		org.opengion.hayabusa.db.CellRenderer
	 */
	public void setRenderer ( final String rdr ) {
		renderer = nval( getRequestParameter( rdr ),renderer );
	}

	/**
	 * 【TAG】カラムの編集用エディター({@og.doc03Link editor Editor_****})をセットします。
	 *
	 * @og.tag
	 * これは、plugin.column 以下の Editor_**** クラスの **** を
	 * 与えます。これらは、CellEditor インターフェースを継承したサブクラスです。
	 * 属性クラス定義の {@link org.opengion.hayabusa.db.CellEditor Editor} を参照願います。
	 * {@og.doc03Link editor Editor_**** クラス}
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	ed 編集用エディター (Editor_**** の ****)
	 * @see		org.opengion.hayabusa.db.CellEditor
	 */
	public void setEditor ( final String ed ) {
		editor = nval( getRequestParameter( ed ),editor );
	}

	/**
	 * 【TAG】ＤＢタイプ({@og.doc03Link dbType DBType_****})をセットします。
	 *
	 * @og.tag
	 * これは、org.opengion.hayabusa.db.column 以下の DBType_**** クラスの **** を
	 * 与えます。これらは、DBType インターフェースを継承したサブクラスです。
	 * 属性クラス定義の {@link org.opengion.hayabusa.db.DBType DBType} を参照願います。
	 * {@og.doc03Link dbType DBType_**** クラス}
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	dt パラメータ
	 * @see		org.opengion.hayabusa.db.DBType  DBType_**** の ****
	 */
	public void setDbType( final String dt ) {
		dbType = nval( getRequestParameter( dt ),dbType );
	}

	/**
	 * 【TAG】コードセレクトのユニークキー(コード名称)をセットします。
	 *
	 * @og.tag
	 * このキーを元に、CodeSelectionオブジェクトを構築します。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10)  新規作成
	 *
	 * @param	name	メニューのユニークキー(コード名称)
	 */
	public void setCodeName( final String name ) {
		codeName = nval( getRequestParameter( name ),codeName );
	}

	/**
	 * 【TAG】コードセレクトのサブセットとなるコード(キー)をCSV形式で指定します。
	 *
	 * @og.tag
	 * この値に限定された、コードリソースを、既存の、CodeData の
	 * サブセットとして作成します。
	 * codeName で指定されたリソースでも、サブセットを作成する事ができます。
	 * ここで指定するのは、コード(キー)だけです。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01)  新規作成
	 *
	 * @param	list	コードセレクトのサブセットとなるコード(キー)
	 */
	public void setCodeList( final String list ) {
		codeList = nval( getRequestParameter( list ),codeList );
	}

	/**
	 * 【TAG】コードセレクトのサブセットとなるグループを指定します。
	 *
	 * @og.tag
	 * この値に限定された、コードリソースを、既存の、CodeData の
	 * サブセットとして作成します。
	 * codeName で指定されたリソースでも、サブセットを作成する事ができます。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01)  新規作成
	 *
	 * @param	group	コードセレクトのサブセットとなるグループ
	 */
	public void setCodeGroup( final String group ) {
		codeGroup = nval( getRequestParameter( group ),codeGroup );
	}

	/**
	 * 【TAG】新しいコードセレクトを作成する為の「キー:ラベル キー:ラベル」の文字列を指定します。
	 *
	 * @og.tag
	 * 既存の、CodeData には存在しない、新しいコードリソースを作成する為の、文字列を指定します。
	 * 文字列は、「キー:ラベル キー:ラベル」形式で、スペースで分解後、":" でキーとラベルに分離します。
	 * スペース分解後の文字列に、":" が含まれていない場合は、キーをラベルとして扱います。
	 * また、ラベル部分は、ラベルリソースを使用して、変換を行います。
	 * 内部的には、CodeData を作成しません。DBColumnオブジェクト内で、直接、Selection_KEYVAL を生成します。
	 * codeName 、codeList、codeGroup などが指定された場合は、そちらが優先されます。
	 * 「キー:ラベル キー:ラベル」で、ラベル にスペースを含ませる場合は、ダブルクォーテーションで囲ってください。
	 * 「"キー:ラベル" "キー:ラベル"」という感じです。
	 *
	 * ※ editor,renderer を、"KVMENU" にして、param に、「キー:ラベル キー:ラベル」
	 *    の文字列を指定する方法の、簡易版です。
	 *
	 * @og.rev 6.3.1.1 (2015/07/10) codeKeyVal 属性を追加
	 *
	 * @param	keyval	KEY:VAL文字列
	 */
	public void setCodeKeyVal( final String keyval ) {
		codeKeyVal = nval( getRequestParameter( keyval ),codeKeyVal );
	}

	/**
	 * 【TAG】パラメータをセットします。
	 *
	 * @og.tag
	 * ここで指定したパラメータは、renderer、editor、dbType すべてのパラメータに
	 * セットされます。
	 * パラメータをセットします。
	 *
	 * @og.rev 5.6.4.3 (2013/05/24) 新規追加
	 *
	 * @param	prm パラメータ
	 */
	public void setParam( final String prm ) {
		parameter = nval( getRequestParameter( prm ),parameter );
		rawParameter = nval( prm, parameter );
	}

	/**
	 * 【TAG】value属性に値がセットされていないとき使用する、初期値を指定します。
	 *
	 * @og.tag
	 * value属性に値がセットされていないときに、この初期値を使用します。
	 *
	 * @param	dv 初期値
	 */
	public void setDefaultVal( final String dv ) {
		defaultVal = nval( getRequestParameter( dv ),defaultVal );
	}

	/**
	 * 【TAG】セレクトメニューの値なしのOptionの有無[true/false]を指定します(初期値:falseと同等)。
	 *
	 * @og.tag
	 * セレクトメニューの場合、値なしのOptionをSelectに含めるかどうかを指定します。
	 * ここでは、検索結果の登録画面に使用されるケースが多いため、なにも指定されない
	 * 場合は、なにもセットしません。(Coderesourceの指定どおり)
	 * 動作としては、初期値は、含めない("false") と同等です。
	 *
	 * &lt;og:columnEditor column="CDBK" addNoValue="false" /&gt;
	 *
	 * @param	flag  値なしのOptionの有無 [true:含める/それ以外:含めない]
	 */
	public void setAddNoValue( final String flag ) {
		addNoValue = nval( getRequestParameter( flag ),addNoValue );
	}

	/**
	 * 【TAG】セレクトメニューの場合、キー:ラベル形式で表示するかどうか[true/false/null]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * カラムがプルダウンメニュー形式の場合、キー:ラベル形式で表示するか、ラベルだけで表示するかを
	 * 選択できます。
	 * true を指定すると、「強制的に」キー:ラベル形式で表示します。
	 * false の場合は、「強制的に」ラベルのみで表示されます。
	 * 初期値の null の場合、コードリソースや、SQL文に準拠した指定通りの形式で表示します。
	 *
	 * &lt;og:column name="CDC" addKeyLabel="true" /&gt;
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか。新規追加
	 *
	 * @param   flag  キー:ラベル形式表示 [true:キー:ラベル形式/false:ラベルのみ/null:指定通り]
	 */
	public void setAddKeyLabel( final String flag ) {
		addKeyLabel = nval( getRequestParameter( flag ),addKeyLabel );
	}

	/**
	 * 【TAG】セレクトメニューの場合、キーとラベルを分けて出力するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * カラムがプルダウンメニュー形式の場合、ファイル出力時に、キーとラベルを分離して出力する事ができます。
	 * この属性と、addKeyLabel とは、微妙に関連していて、addKeyLabel="true" のみの場合は、
	 * キー:ラベル形式で、出力されますが、writeKeyLabel="true" も指定されている場合は、こちらが優先されます。
	 * なお、この指定が有効なのは、セレクトメニューのカラムのみです。
	 * 初期値は、false です。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) キー／ラベルで分離して出力するかどうか。
	 *
	 * @param   flag  キー／ラベル分離出力するかどうか [true:分離出力/false:通常]
	 */
	public void setWriteKeyLabel( final String flag ) {
		writeKeyLabel = nval( getRequestParameter( flag ),writeKeyLabel );
	}

	/**
	 * 【TAG】SQL文を指定してラベル情報を動的に読み込むかどうか[true/false]指定します(初期値:false)。
	 *
	 * @og.tag
	 * trueを指定すると、body部分に記述されたSQL文よりラベル情報を生成して、
	 * 動的にラベル表示を切り替えます。
	 *
	 * ワイルドカードを使用した、カラムの複数指定と組み合わせることで、外部から設定
	 * されたラベル情報で一括で変更することができます。
	 *
	 * 発行するクエリでは、第1カラムにラベルキーを、第2カラムにラベル名称を設定します。
	 * 第3カラムが存在する場合は、名称(短)として使用されます。(必須ではありません)
	 *
	 * ここで作成されたラベル情報は、カラムエディターを適用したDBTableModelのみで
	 * 有効であり、内部的にもキャッシュされません。
	 *
	 * 初期値は、false(動的読込しない)です。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) 新規作成
	 *
	 * @param	flag 動的読込 [true:する/false:しない]
	 */
	public void setUseLabelMap( final String flag ) {
		useLabelMap = nval( getRequestParameter( flag ),useLabelMap );
	}

	/**
	 * 【TAG】MENU,DBMENUでSLABELを利用するかどうか[auto/true/false]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * 通常はセレクトメニューは一覧とそれ以外で短縮ラベルの利用が自動で切り替わります。
	 * 強制的にSLABELを利用させたい場合にこの属性をtrue/falseにセットして下さい。
	 * auto/true/false以外を指定した場合はfalse扱いとします。
	 * ここでの初期値は、値の変更を行わないということで、null です。
	 *
	 * @og.rev 5.6.2.3 (2013/03/22) columnタグに実装されているので移植
	 *
	 * @param prm SLABEL利用 [auto:自動/true:利用する/false:利用しない]
	 */
	public void setUseSLabel( final String prm ) {
		useSLabel = nval( getRequestParameter( prm ),useSLabel );
		if( useSLabel != null &&
			!"auto".equalsIgnoreCase( useSLabel ) &&
			!"true".equalsIgnoreCase( useSLabel ) &&
			!"false".equalsIgnoreCase( useSLabel ) ) {
				final String errMsg = "useSLabel は、[auto:自動/true:利用する/false:利用しない] から選んでください。"
								+ "useSLabel=[" + prm + "]" + CR ;
				throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】非表示文字列を指定します(初期値:null)。
	 *
	 * @og.tag
	 * これは、例えば、数字型の場合は、0 がデフォルトなどの場合、
	 * あえて表示したくないなどのケースに使います。
	 * そのような状況が設定されていない場合は、null が返されます。
	 * 初期値は、null です。
	 *
	 * @og.rev 5.6.2.3 (2013/03/22) 新規追加
	 *
	 * @param prm 非表示文字列 [auto:自動/true:利用する/false:利用しない]
	 */
	public void setNoDisplayVal( final String prm ) {
		noDisplayVal = nval( getRequestParameter( prm ),noDisplayVal );
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id テーブルID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId   = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】データベースへ接続する場合のID(初期値:null)。
	 *
	 * @og.tag
	 * データベース接続を使用するエディター、レンデラーを使用する場合に使用する接続先IDを指定します。
	 * カラムオブジェクトは、query処理ではなく、view処理時に適用されるため、データベース
	 * 接続先を指定する必要があります。
	 * 初期値は、null(DEFAULT 接続) です。
	 *
	 * @og.rev 4.0.0.0 (2006/04/02) 新規追加
	 *
	 * @param	id 接続先ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】イベントカラム(親カラム)を設定します。
	 *
	 * @og.tag
	 * 親子関係を持った項目の動的な入れ替えを行う場合に利用します。
	 * このカラムを変化させるトリガとなるカラムの名前をセットします。
	 *
	 * ※詳細はcolumnTagのeventURL属性の説明をご覧下さい。
	 *
	 * なお、columnEditorで、Viewに対して、SQL文の定義(置き換え定義)をする場合でも、
	 * 他カラムの値の参照は、columnタグの場合と同様に、{&#064;XXXX}または$nの変数により行います。
	 * この場合、これらの変数は、"自身と同一行のカラム"を意味します。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 * @og.rev 5.1.7.0 (2010/06/01) コメント大幅修正
	 *
	 * @param	col	親カラム
	 */
	public void setEventColumn( final String col ) {
		eventColumn = nval( getRequestParameter( col ), eventColumn );
	}

	/**
	 * 【TAG】eventColumn使用時の値を、SQL文から求めるためのqueryをセットします。
	 *
	 * @og.tag
	 * eventColumn利用時は{&#064;XXXX}はリクエストパラメータではなく
	 * xxxカラムの実際の値が利用されます。
	 *
	 * @og.rev 6.3.4.0 (2015/08/01) eventValue 追加
	 * @param	val	パラメータ
	 */
	public void setEventValue( final String val ) {
		eventValue = nval( getReservedParameter( val ), eventValue );	// ここでは{@XXXX}変換をしない
	}

	/**
	 * 【TAG】イベントカラム指定時に呼び出すURLを指定します。
	 *
	 * @og.tag
	 * イベントカラム指定時に部品を作成するJSPを指定します。
	 * 初期値はシステムリソースのEVENT_COLUMN_URLです。
	 * (例：eventURL="makeColumn_custom.jsp")
	 *
	 * ※詳細はcolumnTagのeventURL属性の説明をご覧下さい。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @param	url	JSPのURL
	 */
	public void setEventURL( final String url ) {
		eventURL = nval( getRequestParameter( url ), eventURL );
	}

	/**
	 * 【TAG】eventColumn実行後のcallbak関数を指定します。
	 *
	 * @og.tag
	 * eventColumnの動作終了後に実行するCallback関数の指定を行います。
	 * 関数名のみを指定して下さい。
	 * 　ex) sampleCallback()を実行する場合はeventCallback="sampleCallback"
	 * 実行する関数の第一引数には自動的に次の要素が入った配列が渡されます
	 * 　[0] 変更された子カラムのID属性値
	 * 　[1] evnetColumn実行後の子カラムの値
	 * 　[2] eventColumn実行前の子カラムの値
	 *
	 * この属性は、optionAttributesへの設定と同様の動作を行います。
	 *
	 * @og.rev 5.5.4.0 (2012/07/02) 新規追加
	 *
	 * @param   callback Callback関数
	 */
	public void setEventCallback( final String callback ) {
		// optionAttributes扱いで登録します。
		if( callback != null && callback.length() > 0 ){
			add( "optionAttributes", "eventCallback='"+getRequestParameter( callback )+"'" );
		}
	}

	/**
	 * 【HTML5】表示形式を指定します(初期値:text)。
	 *
	 * @og.tag
	 * ColumnTagは、カラムリソースで作成される為、タイプは固定です。
	 * しかし、HTML5 で追加された形式には、有用なタイプも存在する為、
	 * それらへの書き換えをできるようにします。
	 * よって、ここでは、従来型のタイプへの書き換えはできません。
	 * (textだけは例外で戻せるようにします)。
	 *
	 * [search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
	 * が、設定できます。
	 *
	 * HTML5 の機能を有効にするには、ネイティブモードで動作させる必要があります。
	 *   ① USE_IE7_HEADER = "false" に設定する。
	 *   ② USE_HTML5_HEADER = "true" に設定する。
	 *   ③ IEのツール⇒互換表示設定で、互換表示に追加したWebサイトから削除する。
	 *   ④ 同上の設定で、イントラサイトを互換表示で表示するのチェックを外す。
	 * 必要があります。
	 *
	 * &lt;og:input type="text"     /&gt; テキストボックスを表示
	 *
	 * HTML5 で追加されたタイプ
	 * &lt;og:input type="search"			/&gt; 検索テキストの入力欄を作成する
	 * &lt;og:input type="tel"				/&gt; 電話番号の入力欄を作成する
	 * &lt;og:input type="url"				/&gt; URLの入力欄を作成する
	 * &lt;og:input type="email"			/&gt; メールアドレスの入力欄を作成する
	 * &lt;og:input type="datetime"			/&gt; UTC（協定世界時）による日時の入力欄を作成する
	 * &lt;og:input type="date"				/&gt; 日付の入力欄を作成する
	 * &lt;og:input type="month"			/&gt; 月の入力欄を作成する
	 * &lt;og:input type="week"				/&gt; 週の入力欄を作成する
	 * &lt;og:input type="time"				/&gt; 時間の入力欄を作成する
	 * &lt;og:input type="datetime-local"	/&gt; UTC（協定世界時）によらないローカル日時の入力欄を作成する
	 * &lt;og:input type="number"			/&gt; 数値の入力欄を作成する
	 * &lt;og:input type="range"			/&gt; レンジの入力欄を作成する
	 * &lt;og:input type="color"			/&gt; 色の入力欄を作成する
	 *
	 * @og.rev 5.7.1.0 (2013/12/06) HTML5 対応(新規追加)
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @param   type HTML5用 [text/search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
	 */
	public void setType( final String type ) {
		final String tp = getRequestParameter( type );

		if( check( tp , TYPE_SET ) ) {
			set( "type",tp );
		}
		else {
			final String errMsg = "type 属性は、下記の中から選択してください。type=["
						 + tp + "]  in [" + String.join( ", " , TYPE_SET ) + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】ResourceManager を使用しないかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * trueを指定すると、使用しない設定になります。
	 * これは、データベースから検索した値そのものが返される カラムオブジェクトを使用することになります。
	 * 「noResource」なので、意味が反転しているのをご注意ください。
	 *
	 * false(初期値)は、ResourceManager を使用します。
	 *
	 * @og.rev 5.7.3.0 (2014/02/07) 新規作成
	 *
	 * @param	flag ResourceManagerの不使用 [true:不使用/false:使用]
	 */
	public void setNoResource( final String flag ) {
		noResource = nval( getRequestParameter( flag ),noResource );
	}

	/**
	 * 【TAG】(通常は使いません)カラムＩＤの存在チェックを行うかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * true の場合、カラムＩＤがDBTableModel に存在しない場合は、エラーになります。
	 * false の場合、カラムＩＤがDBTableModel に存在しない場合は、無視します。
	 * これは、検索条件によって、設定されるカラムが異なる場合でも、後続処理を
	 * 正常に動作させたい場合に、使用します。
	 * 初期値は true (チェックを行う) です。
	 *
	 * @og.rev 6.7.6.0 (2017/03/17) strictCheck 追加。
	 *
	 * @param  check 存在チェック [true:行う/false:行わない]
	 */
	public void setStrictCheck( final String check ) {
		strictCheck = nval( getRequestParameter( check ),strictCheck );
	}

	/**
	 * 【TAG】レンデラ利用時のExcel出力でセルタイプを文字固定にするか[true/false]を指定します(初期値:システム定数のUSE_STRING_EXCEL_OUTPUT)。
	 *
	 * @og.tag
	 * falseを指定すると、dbTypeに従ったセルタイプとなります。
	 * TableWriter内部でdouble変換をするためエラーに注意してください。
	 * (初期値:システム定数のUSE_STRING_EXCEL_OUTPUT[={@og.value SystemData#USE_STRING_EXCEL_OUTPUT}])。
	 *
	 * @og.rev 5.7.6.3 (2014/05/23) 新規作成
	 *
	 * @param	flag セルタイプの文字固定にするか [true:固定する/false:固定しない]
	 */
	public void setStringOutput( final String flag ) {
		stringOutput = nval( getRequestParameter( flag ),stringOutput );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tableId"		,tableId	)
				.println( "columns"		,columns	)
				.println( "allColumns"	,allColumns	)
				.println( "className"	,className	)
				.println( "size"		,fieldSize	)			// 6.2.0.0 (2015/02/27)
				.println( "viewSize"	,viewSize	)			// 6.2.0.0 (2015/02/27)
				.println( "maxlength"	,maxlength	)
				.println( "writable"	,writable	)
				.println( "renderer"	,renderer	)
				.println( "editor"		,editor		)
				.println( "codeName"	,codeName	)
				.println( "dbType"		,dbType		)
				.println( "defaultVal"	,defaultVal	)
				.println( "parameter"	,parameter	)
				.println( "dbid"		,dbid		)
				.println( "addNoValue"	,addNoValue	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
