/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.servlet;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.servlet.multipart.MultipartParser;
import org.opengion.hayabusa.servlet.multipart.Part;
import org.opengion.hayabusa.servlet.multipart.FilePart;
import org.opengion.hayabusa.servlet.multipart.ParamPart;
import org.opengion.fukurou.util.ZipArchive;				// 6.0.0.0 (2014/04/11) zip 対応

import java.io.File;
import java.io.IOException;
// import java.io.FileNotFoundException;					// 6.9.0.1 (2018/02/05)
import java.util.Map;
import java.util.concurrent.ConcurrentSkipListMap;			// 6.4.3.1 (2016/02/12) refactoring

import java.util.List;
import java.util.ArrayList;
import java.util.Set;
import java.util.Random ;
import java.util.concurrent.atomic.AtomicInteger;			// 5.5.2.6 (2012/05/25) findbugs対応
import javax.servlet.http.HttpServletRequest;

/**
 * ファイルをサーバーにアップロードする場合に使用されるマルチパート処理サーブレットです。
 *
 * 通常のファイルアップロード時の、form で使用する、enctype="multipart/form-data"
 * を指定した場合の、他のリクエスト情報も、取り出すことが可能です。
 *
 * ファイルをアップロード後に、指定のファイル名に変更する機能があります。
 * file 登録ダイアログで指定した name に、"_NEW" という名称を付けたリクエスト値を
 * ファイルのアップロードと同時に送信することで、この名前にファイルを付け替えます。
 * また、アップロード後のファイル名は、name 指定の名称で、取り出せます。
 * クライアントから登録したオリジナルのファイル名は、name に、"_ORG" という名称
 * で取り出すことが可能です。
 *
 * maxPostSize : 最大転送サイズ(Byte)を指定します。 0,またはマイナスで無制限です。
 * useBackup   : ファイルアップロード時に、すでに同名のファイルが存在した場合に、
 *               バックアップ処理(renameTo)するかどうか[true/false]を指定します(初期値:false)
 *
 * ファイルアップロード時に、アップロード先に、同名のファイルが存在した場合は、既存機能は、そのまま
 * 置き換えていましたが、簡易バージョンアップ機能として、useBackup="true" を指定すると、既存のファイルを
 * リネームして、バックアップファイルを作成します。
 * バックアップファイルは、アップロードフォルダを基準として、_backup/ファイル名.拡張子_処理時刻のlong値.拡張子 になります。
 * オリジナルのファイル名（拡張子付）を残したまま、"_処理時刻のlong値" を追加し、さらに、オリジナルの拡張子を追加します。
 * バックアップファイルの形式は指定できません。
 *
 * 5.7.1.2 (2013/12/20) zip 対応
 * filename 属性に、".zip" の拡張子のファイル名を指定した場合は、アップロードされた一連のファイルを
 * ZIP圧縮します。これは、アップロード後の処理になります。
 * ZIP圧縮のオリジナルファイルは、そのまま残ります。
 * なお、ZIPファイルは、useBackup属性を true に設定しても、無関係に、上書きされます。
 *
 * @og.group その他機能
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class MultipartRequest {
	private static AtomicInteger dumyNewFileCnt = new AtomicInteger(1);		// 5.5.2.6 (2012/05/25) findbugs対応

	// 6.3.9.0 (2015/11/06) Variables should start with a lowercase character(PMD)
	private static final String RANDOM_KEY = new Random().nextInt( Integer.MAX_VALUE ) + "_" ;	// 5.6.5.3 (2013/06/28) アップロード時のダミーファイル名をもう少しだけランダムにする。

	/** 6.4.3.1 (2016/02/12) PMD refactoring. TreeMap → ConcurrentSkipListMap に置き換え。  */
	private final Map<String,List<String>> paramMap	= new ConcurrentSkipListMap<>();	// 6.4.3.1 (2016/02/12) ソートします。

	// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
	private final List<UploadedFile> files				= new ArrayList<>();			// 5.7.1.1 (2013/12/13) HTML5対応

	/**
	 * MultipartRequest オブジェクトを構築します。
	 *
	 * 引数として、ファイルアップロード時の保存フォルダ、最大サイズ、エンコード、
	 * 新しいファイル名などを指定できます。新しいファイル名は、アップロードされる
	 * ファイルが一つだけの場合に使用できます。複数のファイルを同時に変更したい
	 * 場合は、アップロードルールにのっとり、リクエストパラメータで指定してください。
	 *
	 * HTML5 では、ファイルアップロード時に、multiple 属性（inputタグのtype="file"）を
	 * 付ける事で、ファイルを複数選択できます。
	 * その場合は、inputのname属性は、一つなので、_NEW による名前の書き換えはできません。
	 *
	 * @og.rev 3.8.1.3A (2006/01/30) 新ファイル名にオリジナルファイル名の拡張子をセットします
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 5.5.2.6 (2012/05/25) findbugs対応。staticフィールドへの書き込みに、AtomicInteger を利用します。
	 * @og.rev 5.6.5.3 (2013/06/28) useBackup引数追加
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
	 * @og.rev 5.7.1.2 (2013/12/20) zip 対応
	 * @og.rev 5.7.4.3 (2014/03/28) zip 対応復活。inputFilename のリクエスト変数処理追加
	 * @og.rev 6.0.2.4 (2014/10/17) useBackup 修正。_PFX(接頭辞) , _SFX(接尾辞) 機能を追加。ファイル名にフォルダ指定可
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 * @og.rev 5.9.25.0 (2017/10/06) ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ利用処理追加
	 * @og.rev 6.9.0.1 (2018/02/05) ﾌｧｲﾙをｾｰﾌﾞするﾃﾞｨﾚｸﾄﾘは、必要な場合のみ、作成します。
	 *
	 * @param	request	HttpServletRequestオブジェクト
	 * @param	saveDirectory	ファイルアップロードがあった場合の保存フォルダ名
	 * @param	maxPostSize		ファイルアップロード時の最大ファイルサイズ(Byte)0,またはマイナスで無制限
	 * @param	encoding		ファイルのエンコード
	 * @param	inputFilename	アップロードされたファイルの新しい名前
	 * @param	useBackup		ファイルアップロード時に、バックアップ処理するかどうか[true/false/rename]を指定
	 * @param	fileURL			クラウドストレージ用のURL
	 * @throws IOException 入出力エラーが発生したとき
	 * @throws IllegalArgumentException ｾｰﾌﾞﾃﾞｨﾚｸﾄﾘ に関係するｴﾗｰ
	 */
	public MultipartRequest(final HttpServletRequest request,
							final String	saveDirectory,
							final int		maxPostSize,
							final String	encoding,
							final String	inputFilename,
							final String	useBackup,								// 6.0.2.4 (2014/10/17) true/false/rename
							final String	fileURL) throws IOException,IllegalArgumentException {			// (2017/10/06) 追加

		if( request == null ) {
			throw new IllegalArgumentException("request cannot be null");
		}

//		// 6.9.0.1 (2018/02/05) ﾌｧｲﾙをｾｰﾌﾞするﾃﾞｨﾚｸﾄﾘは、必要な場合のみ、作成します。
//		if( saveDirectory == null ) {
//			throw new IllegalArgumentException("saveDirectory cannot be null");
//		}
//		// 5.5.2.6 (2012/05/25) 0,またはマイナスで無制限
//		// Save the dir
//		final File dir = new File(saveDirectory);
//
//		// Check saveDirectory is truly a directory
//		if( !dir.isDirectory() ) {
//			throw new IllegalArgumentException("Not a directory: " + saveDirectory);
//		}
//
//		// Check saveDirectory is writable
//		if( !dir.canWrite() ) {
//			throw new IllegalArgumentException("Not writable: " + saveDirectory);
//		}

		// Parse the incoming multipart, storing files in the dir provided,
		// and populate the meta objects which describe what we found
		final MultipartParser parser = new MultipartParser(request, maxPostSize);
		if( encoding != null ) {
			parser.setEncoding(encoding);
		}

		// 2017/10/06 ADD ｼｽﾃﾑﾘｿｰｽにｸﾗｳﾄﾞｽﾄﾚｰｼﾞ利用が登録されている場合は、ｸﾗｳﾄﾞｽﾄﾚｰｼﾞを利用する
		final String storage = HybsSystem.sys( "CLOUD_STORAGE");
		final boolean useStorage = storage != null && storage.length() > 0 ;

		// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
		Part part;
		File dir  = null;
		while( (part = parser.readNextPart()) != null ) {
			final String name = part.getName();
			if( part.isParam() && part instanceof ParamPart ) {
				final ParamPart paramPart = (ParamPart)part;
				final String value = paramPart.getStringValue();
				// 6.4.3.1 (2016/02/12) ConcurrentMap 系は、key,val ともに not null 制限です。
				List<String> existingValues = paramMap.get(name);
				if( existingValues == null ) {
					existingValues = new ArrayList<>();
					paramMap.put(name, existingValues);
				}
				existingValues.add(value);
			}
			else if( part.isFile() && part instanceof FilePart ) {
				final FilePart filePart = (FilePart)part;
				final String orgName = filePart.getFilename();		// 5.7.1.1 (2013/12/13) 判りやすいように変数名変更
				if( orgName != null ) {
					// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
					// 同一 name で、複数ファイルを扱う必要があります。
					// 3.8.1.2 (2005/12/19) 仮ファイルでセーブする。
					final String uniqKey = RANDOM_KEY + dumyNewFileCnt.getAndIncrement() ;		// 5.6.5.3 (2013/06/28) アップロード時のダミーファイル名をもう少しだけランダムにする。
					filePart.setFilename( uniqKey );
					// 標準のﾌｧｲﾙ書き込み 2017/10/06 DELETE ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ利用判定を追加

					if( useStorage ){
						// ｸﾗｳﾄﾞｽﾄﾚｰｼﾞにｱｯﾌﾟﾛｰﾄﾞ
						filePart.writeToCloud( storage, fileURL, request.getSession(true) );
					}else{
						if( dir == null ) { dir = makeDirs( saveDirectory ); }		// 6.9.0.1 (2018/02/05)
						// 標準のﾌｧｲﾙ書き込み
						filePart.writeTo(dir);
					}

					// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
					files.add( new UploadedFile(
											uniqKey,		// 5.7.1.1 (2013/12/13) 順番変更
											dir.toString(),
											name,			// 5.7.1.1 (2013/12/13) 項目追加
											orgName,
											filePart.getContentType()));

				}
			}
			else {
				final String errMsg = "Partオブジェクトが、ParamPartでもFilePartでもありません。"
							+ " class=[" + part.getClass() + "]";
				throw new OgRuntimeException( errMsg );
			}
		}

		// 5.7.4.3 (2014/03/28) inputFilename は、リクエスト変数が使えるようにします。
		final String filename = getReqParamFileName( inputFilename ) ;

		// 3.5.6.5 (2004/08/09) 登録後にファイルをリネームします。
		// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
		final int size = files.size();

		// 5.7.1.2 (2013/12/20) zip 対応
		// 5.9.25.0 (2017/10/06) FileをString型に変更
		final String[] tgtFiles = new String[size];
		final boolean isZip = filename != null && filename.endsWith( ".zip" ) ;

		for( int i=0; i<size; i++ ) {
			final UploadedFile upFile = files.get(i);
			final String name = upFile.getName();		// 5.7.1.1 (2013/12/13)

			String newName = isZip ? null : filename ;
			String prefix = null ;				// 6.0.2.4 (2014/10/17) _PFX(接頭辞) , _SFX(接尾辞) 機能を追加
			String sufix  = null ;				// 6.0.2.4 (2014/10/17) _PFX(接頭辞) , _SFX(接尾辞) 機能を追加
			if( newName == null && name != null ) {
				final int adrs = name.lastIndexOf( HybsSystem.JOINT_STRING );	// カラム__行番号 の __ の位置
				// 6.0.2.4 (2014/10/17) _PFX(接頭辞) , _SFX(接尾辞) 機能を追加
				if( adrs < 0 ) {
					newName = getParameter( name + "_NEW" );
					prefix  = getParameter( name + "_PFX" );
					sufix   = getParameter( name + "_SFX" );
				}
				else {
					final String name1 = name.substring( 0,adrs );
					final String name2 = name.substring( adrs );
					newName = getParameter( name1 + "_NEW" + name2 );
					prefix  = getParameter( name1 + "_PFX" + name2 );
					sufix   = getParameter( name1 + "_SFX" + name2 );
				}
			}

			// 5.7.1.1 (2013/12/13) UploadedFile 内で処理するように変更
			// 5.9.25.0 (2017/10/06) MODIFY fileURLとsessionを追加
			tgtFiles[i] = upFile.renameTo( newName,prefix,sufix,useBackup,fileURL,request.getSession(true) );
		}
		// 5.7.1.2 (2013/12/20) zip 対応
		// 6.0.0.0 (2014/04/11) 一旦保留にしていましたが、復活します。
		if( isZip && !useStorage ) {
			final File zipFile = new File( saveDirectory,filename );
			// 5.9.25.0 (2017/10/06) tgtFiles が、String型に変更されたため
			final File[] files = new File[tgtFiles.length];
			for( int i=0; i<tgtFiles.length; i++ ) {
				files[i] = new File( tgtFiles[i] );
			}
			ZipArchive.compress( files,zipFile );
		}
	}

	/**
	 * リクエストパラメータの名前配列を取得します。
	 *
	 * @return	リクエストパラメータの名前配列
	 * @og.rtnNotNull
	 */
	public String[] getParameterNames() {
		final Set<String> keyset = paramMap.keySet();
		return keyset.toArray( new String[keyset.size()] );
	}

	/**
	 * ファイルアップロードされたファイル群のファイル配列を取得します。
	 *
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
	 *
	 * @return	アップロードされたファイル群
	 * @og.rtnNotNull
	 */
	public UploadedFile[] getUploadedFile() {
		return files.toArray( new UploadedFile[files.size()] );
	}

	/**
	 * 指定の名前のリクエストパラメータの値を取得します。
	 *
	 * 複数存在する場合は、一番最後の値を返します。
	 *
	 * @param	name	リクエストパラメータ名
	 *
	 * @return	パラメータの値
	 */
	public String getParameter( final String name ) {
		final List<String> values = paramMap.get(name);

		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return values == null || values.isEmpty() ? null : values.get( values.size() - 1 );
	}

	/**
	 * 指定の名前のリクエストパラメータの値を配列型式で取得します。
	 *
	 * @og.rev 5.3.2.0 (2011/02/01) 新規作成
	 * @og.rev 6.3.9.1 (2015/11/27) null ではなく長さが0の配列を返すことを検討する(findbugs)。
	 *
	 * @param	name	リクエストパラメータ名
	 *
	 * @return	パラメータの値配列(存在しない場合は、長さ０の配列を返します)
	 * @og.rtnNotNull
	 */
	public String[] getParameters( final String name ) {
		final List<String> values = paramMap.get(name);
		return values == null || values.isEmpty()
					? new String[0]
					: values.toArray( new String[values.size()] );
	}

	/**
	 * 指定の名前のリクエストパラメータの値を配列(int)型式で取得します。
	 *
	 * @og.rev 5.3.2.0 (2011/02/01) 新規作成
	 * @og.rev 5.3.6.0 (2011/06/01) 配列値が""の場合にNumberFormatExceptionが発生するバグを修正
	 * @og.rev 6.3.9.1 (2015/11/27) null ではなく長さが0の配列を返すことを検討する(findbugs)。
	 *
	 * @param	name	リクエストパラメータ名
	 *
	 * @return	パラメータの値配列(存在しない場合は、長さ０の配列を返します)
	 * @og.rtnNotNull
	 */
	public int[] getIntParameters( final String name ) {
		final List<String> values = paramMap.get(name);

		return values == null || values.isEmpty()
					? new int[0]
					: values.stream()
							.filter( str -> str != null && !str.isEmpty() )		// 条件
							.mapToInt( Integer::parseInt )						// 変換 String → int
							.toArray();											// int[] 配列
	}

	/**
	 * 指定の名前の ファイル名のリクエスト変数処理を行います。
	 *
	 * filename 属性のみ、{&#064;XXXX} のリクエスト変数が使えるようにします。
	 *
	 * @og.rev 5.7.4.3 (2014/03/28) 新規追加
	 *
	 * @param	fname	ファイル名
	 * @return	リクエスト変数を処理したファイル名
	 */
	private String getReqParamFileName( final String fname ) {

		String rtn = fname ;
		if( fname != null ) {
			final StringBuilder filename = new StringBuilder( fname ) ;
			int st = filename.indexOf( "{@" );
			while( st >= 0 ) {
				final int ed = filename.indexOf( "}",st );
				if( ed < 0 ) {
					final String errMsg = "{@XXXX} の対応関係が取れていません。"
								+ " filename=[" + fname + "]";
					throw new OgRuntimeException( errMsg );
				}
				final String key = filename.substring( st+2,ed );		// "}" は切り出し対象外にする。
				final String val = getParameter( key );
				filename.replace( st,ed+1,val );				// "}" を含めて置換したいので、ed+1
				// 次の "{@" を探す。開始は置換文字数が不明なので、st から始める。
				st = filename.indexOf( "{@",st );
			}
			rtn = filename.toString();
		}
		return rtn ;
	}

	/**
	 * 指定のﾃﾞｨﾚｸﾄﾘが無ければ作成します。
	 *
	 * @og.rev 6.9.0.1 (2018/02/05) ﾌｧｲﾙをｾｰﾌﾞするﾃﾞｨﾚｸﾄﾘは、必要な場合のみ、作成します。
	 *
	 * @param	saveDir	ﾃﾞｨﾚｸﾄﾘ名
	 * @return	ｾｰﾌﾞ可能なﾃﾞｨﾚｸﾄﾘ
	 * @throws	IllegalArgumentException	ｾｰﾌﾞﾃﾞｨﾚｸﾄﾘ に関係するｴﾗｰ(無理から)
	 */
	private File makeDirs( final String saveDir ) throws IllegalArgumentException {
		// ｾｰﾌﾞﾃﾞｨﾚｸﾄﾘの名前ﾁｪｯｸ
		if( saveDir == null ) {
			throw new IllegalArgumentException( "saveDir cannot be null" );
		}

		// ｾｰﾌﾞﾃﾞｨﾚｸﾄﾘのｵﾌﾞｼﾞｪｸﾄ
		final File dir = new File( saveDir );

		// ｾｰﾌﾞﾃﾞｨﾚｸﾄﾘ 作成
		if( ! dir.exists() && ! dir.mkdirs() ) {
			throw new IllegalArgumentException( "Not make directory: " + saveDir );
		}

		// ﾃﾞｨﾚｸﾄﾘでなければ、ｴﾗｰ
		if( !dir.isDirectory() ) {
			throw new IllegalArgumentException( "Not a directory: " + saveDir );
		}

		// 書込みできなければ、ｴﾗｰ
		if( !dir.canWrite() ) {
			throw new IllegalArgumentException( "Not writable: " + saveDir );
		}

		return dir;
	}
}
