/*
 * Copyright (c) 2017 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.fileexec;

import java.nio.file.Path;

/**
 * システムのベースフォルダ基準で、各種パスを管理するクラスです。
 *
 *<pre>
 * 本システムでは、ベースフォルダを基準として、各種管理フォルダを規定しています。
 * それらのパスオブジェクトを管理します。
 *
 *</pre>
 *
 * <table border="2" cellspacing="0" cellpadding="2" rules="all" style="font-family: monospace;">
 *  <caption>各種管理フォルダの規定</caption>
 *  <tr><th>フォルダ	</th><th>説明</th></tr>
 *  <tr><td>BASE_PATH	</td><td>基本となるフォルダパス</td></tr>
 *  <tr><td>SUB_PATH	</td><td>ベースパスの下位のサブパス。これがスキャン対象になる。</td></tr>
 *  <tr><td>WORK_PATH	</td><td>中間パス。サブパスのファイルを一旦このワークに移動してから、取込処理を行います</td></tr>
 *  <tr><td>OK_PATH		</td><td>処理が成功した場合のファイルの移動先</td></tr>
 *  <tr><td>NG_PATH		</td><td>処理が失敗した場合のファイルの移</td></tr>
 *</table>
 *
 * @og.rev 7.0.0.0 (2017/07/07) 新規作成
 *
 * @version  7.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.8,
 */
public final class BasePath {
	private static final XLogger LOGGER= XLogger.getLogger( BasePath.class.getName() );		// ログ出力

	/** BASEフォルダの相対パスの処理ﾌｫﾙﾀﾞ(WORK)の初期値 {@value} */
	public static final String DEF_WORK_DIR	= "work" ;
	/** BASEフォルダの相対パスの処理済ﾌｫﾙﾀﾞ(正常)の初期値 {@value} */
	public static final String DEF_OK_DIR	= "bkup_ok" ;
	/** BASEフォルダの相対パスの処理済ﾌｫﾙﾀﾞ(異常)の初期値 {@value} */
	public static final String DEF_NG_DIR	= "bkup_ng" ;

	/** ベースパス  */
	public final Path BASE_PATH	;

	/** サブパス(実際にｽｷｬﾝを行うﾄｯﾌﾟﾊﾟｽ)  */
	public final Path SUB_PATH	;

	/** 処理ﾌｫﾙﾀﾞ(WORK)のパス  */
	public final Path WORK_PATH	;
	/** 処理済ﾌｫﾙﾀﾞ(正常)のパス  */
	public final Path OK_PATH ;
	/** 処理済ﾌｫﾙﾀﾞ(異常)のパス  */
	public final Path NG_PATH	;

	/**
	 * フォルダを指定して、オブジェクトを作成します。
	 *
	 * DIR_BASE,DIR_SUB,DIR_WORK,DIR_BKUP_OK,DIR_BKUP_NG の順にセットされます。
	 * 指定されていない(null)の場合は、初期設定を使用します。
	 *
	 * @param	dirs	フォルダ配列
	 */
	public BasePath( final String... dirs ) {
		BASE_PATH	= FileUtil.readPath( dirs[0] );						// ベースフォルダのチェック

		SUB_PATH	= dirs[1] == null || dirs[1].isEmpty()
							? BASE_PATH
							: FileUtil.readPath( dirs[0] , dirs[1] );

		WORK_PATH	= dirs[2] == null || dirs[2].isEmpty()
							? FileUtil.writePath( dirs[0] , DEF_WORK_DIR )
							: FileUtil.writePath( dirs[2] );

		OK_PATH		= dirs[3] == null || dirs[3].isEmpty()
							? FileUtil.writePath( dirs[0] , DEF_OK_DIR )
							: FileUtil.writePath( dirs[3] );

		NG_PATH		= dirs[4] == null || dirs[4].isEmpty()
							? FileUtil.writePath( dirs[0] , DEF_NG_DIR )
							: FileUtil.writePath( dirs[4] );

		LOGGER.debug( () ->  "[BasePath] BASE_PATH="	+ BASE_PATH
								 + " , SUB_PATH="	+ SUB_PATH
								 + " , WORK_PATH="	+ WORK_PATH
								 + " , OK_PATH="	+ OK_PATH
								 + " , NG_PATH="	+ NG_PATH );
	}
}
