/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import java.util.Date;

import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.fukurou.db.ResultSetValue;		// 6.0.4.0 (2014/11/28)
import org.opengion.hayabusa.common.HybsSystem;		// 6.9.3.0 (2018/03/26)6.9.3.0 (2018/03/26)

import org.jfree.data.gantt.TaskSeriesCollection;
import org.jfree.data.gantt.TaskSeries;
import org.jfree.data.gantt.Task;

/**
 * HybsTaskSeriesCollection は、org.jfree.data.gantt.TaskSeriesCollection を継承したサブクラスで、
 * オブジェクト作成とともに JDBC接続して、TaskSeries データを作成し、セットします。
 * TaskSeriesCollection は、IntervalCategoryDataset, GanttCategoryDataset インターフェースを継承しています。
 *
 * データ形式は、シリーズ名、タスク名、開始日時、終了日時 の順で、シリーズ名でソートしておく必要があります。
 * シリーズ名 は、キーブレイクで、設定する為です。
 * 
 *  select series,task,st(時間),ed(時間) from XX order by series,task,st(時間)
 *
 * @og.rev 5.6.1.0 (2013/02/01) 新規作成
 *
 * @version  0.9.0  2001/05/05
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class HybsTaskSeriesCollection extends TaskSeriesCollection {
	private static final long serialVersionUID = 561020130201L ;

	/** 6.9.3.0 (2018/03/26) データ検索時のフェッチサイズ  {@value} */
	private static final int DB_FETCH_SIZE = HybsSystem.sysInt( "DB_FETCH_SIZE" ) ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public HybsTaskSeriesCollection() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * HybsTaskSeriesCollection オブジェクトの内部に、DB検索結果のデータを設定します(縦持)。
	 *
	 * select series,task,st(時間),ed(時間) from XX order by series,task,st(時間)
	 * シリーズ名 は、キーブレイクで、設定します。
	 * (独自メソッド)
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
	 * @og.rev 6.4.2.1 (2016/02/05) try-with-resources 文で記述。
	 * @og.rev 6.9.3.0 (2018/03/26) データ検索時のフェッチサイズを設定。
	 *
	 * @param conn  the connection.
	 * @param query  the query.
	 * @throws SQLException データベース実行エラーが発生した場合
	 *
	 */
	public void executeQuery( final Connection conn, final String query ) throws SQLException {
		// 6.4.2.1 (2016/02/05) try-with-resources 文
		try( final Statement statement = conn.createStatement();
			final ResultSet resultSet = statement.executeQuery(query) ) {

			statement.setFetchSize( DB_FETCH_SIZE );				// 6.9.3.0 (2018/03/26) データ検索時のフェッチサイズ

			// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
			final ResultSetValue rsv = new ResultSetValue( resultSet );

			final int columnCount = rsv.getColumnCount();						// 6.0.4.0 (2014/11/28)

			if( columnCount < 4 ) {
				final String errMsg = "HybsTaskSeriesCollection.executeQuery() : 実行できません。\n"
							+ "select series,task,st(時間),ed(時間) は、最低必要です。それ以降は無視します。"
							+ " SQL=" + query ;
				throw new SQLException( errMsg );
			}

			String bkSeries = null;		// キーブレイクのための過去のSeries

			TaskSeries taskseries = null;
			while( rsv.next() ) {											// 6.0.4.0 (2014/11/28)
				// first column contains the row key...
				final String seriVal	= rsv.getValue(0);							// シリーズ名
				if( seriVal != null && !seriVal.equals( bkSeries ) ) {
					if( taskseries != null ) { add( taskseries ); }			// キーブレイクでTaskSeriesCollectionにセット
					taskseries = new TaskSeries( seriVal );
					bkSeries = seriVal ;
				}

				// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
				final String taskVal		= rsv.getValue(1);					// タスク名
				final String stDataVal	= rsv.getValue(2);					// st(時間)
				final String edDateVal	= rsv.getValue(3);					// ed(時間)

				final Date stDate = HybsDateUtil.getCalendar( stDataVal ).getTime() ;
				final Date edDate = HybsDateUtil.getCalendar( edDateVal ).getTime() ;

				final Task task = new Task( taskVal, stDate, edDate );

				taskseries.add( task );
			}
			if( taskseries != null ) { add( taskseries ); }		// キーブレイクでTaskSeriesCollectionにセット
		}
	}
}
