/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.model.DataModel;

/**
 * javax.swing.table.TableModel インターフェースを継承したデータベース情報を
 * TableModel情報にマッピングするのに利用します。
 * DataModel<String> インターフェースを継承しています。
 *
 * @og.group テーブル管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface DBTableModel extends DataModel<String> {

	/**
	 * 行指定の書込み許可を与えます。
	 * 具体的には,チェックボックスの表示/非表示を指定します。
	 * これが true の場合は,書込み許可です。(チェックボックスを表示)
	 * false の場合は,書込み不許可(チェックボックスは表示されません。)
	 * 行毎に書込み許可/不許可を指定する場合は,１カラム目に writable
	 * カラムを用意して true/false を指定します。
	 * この writable カラムとの論理積により最終的にチェックボックスの
	 * 表示の ON/OFF が決まります。
	 *
	 * このデフォルト値は、true に設定されています。
	 *
	 */
	boolean DEFAULT_WRITABLE = true;

	/**
	 * 行指定用のチェックボックスに対して初期値を 選択済みにするか、
	 * 非選択済みにするかのデフォルト値を指定します。
	 *
	 * このデフォルト値は、false に設定されています。
	 *
	 */
	boolean DEFAULT_CHECKED = false;

	/**
	 * このオブジェクトを初期化します。
	 * 指定の引数分の内部配列を作成します。
	 * 具体的には,DBColumn の数を指定することになります。
	 *
	 * @param   columnCount カラム数
	 */
	void init( int columnCount ) ;

	/**
	 * このオブジェクトをヘッダー部分をコピーし、データを初期化します。
	 * これは、カラムなどヘッダー系の情報は、元と同じオブジェクトを共有し、
	 * データ部のみ空にした DBTableModel を作成することを意味します。
	 * この際、consistencyKey も複写しますので、整合性は崩れないように、
	 * データ登録を行う必要があります。
	 *
	 * @og.rev 4.0.0.0 (2007/06/28) 新規作成
	 *
	 * @return  DBTableModelオブジェクト
	 */
	DBTableModel newModel();

	/**
	 * カラムのラベル名を返します。
	 * カラムの項目名に対して,見える形の文字列を返します。
	 * 一般には,リソースバンドルと組合せて,各国ロケール毎にラベルを
	 * 切替えます。
	 * セットされた DBColumn#getLabel( int ) の値が返されます。
	 *
	 * @param   column カラム番号
	 *
	 * @return  カラムのラベル名
	 */
	String getColumnLabel( int column ) ;

	/**
	 * row および columnName にあるセルの属性値をStringに変換して返します。
	 *
	 * @param   aRow       値が参照される行
	 * @param   columnName 値が参照されるカラム名
	 *
	 * @return  指定されたセルの値 String
	 * @see #getValue( int , int )
	 */
	String getValue( final int aRow, final String columnName );

	/**
	 * カラム(列)にカラムオブジェクトを割り当てます。
	 * カラムオブジェクトは，ラベルやネームなど，そのカラム情報を
	 * 保持したオブジェクトです。
	 *
	 * @param   dbColumn  カラムオブジェクト
	 * @param   clm       ヘッダーを適応するカラム(列)
	 */
	void setDBColumn( int dbColumn, DBColumn clm ) ;

	/**
	 * カラム(列)のカラムオブジェクトを返します。
	 * カラムオブジェクトは，ラベルやネームなど，そのカラム情報を
	 * 保持したオブジェクトです。
	 *
	 * @param	clm	ヘッダーを適応するカラム(列)
	 *
	 * @return	DBColumnカラムオブジェクト
	 */
	DBColumn getDBColumn( int clm ) ;

	/**
	 * カラムオブジェクト配列を返します。
	 * カラムオブジェクトは，ラベルやネームなど，そのカラム情報を
	 * 保持したオブジェクトです。
	 *
	 * @og.rev 4.0.0.0 (2005/12/31) 新規追加
	 *
	 * @return	カラムオブジェクト配列
	 */
	DBColumn[] getDBColumns() ;

	/**
	 * カラム名をもとに、そのカラム番号を返します。
	 * useThrow が、true の場合は、カラム名が存在しない場合は､ HybsSystemException を
	 * throw します。useThrow が、false の場合は、カラム名が存在しない場合は､ -1 を返します。
	 *
	 * @og.rev 4.0.0.0 (2005/12/31) 新規追加
	 *
	 * @param   columnName   カラム名
	 * @param   useThrow     カラム名が存在しない場合に、Exception を throw するかどうか
	 *
	 * @return  カラム番号
	 */
	int getColumnNo( final String columnName,final boolean useThrow ) ;

	/**
	 * 変更済みフラグを元に戻します。
	 *
	 * 一般には,データベースにテーブルモデルを登録するタイミングで、
	 * 変更済みフラグを元に戻します。
	 *
	 */
	void resetModify() ;

	/**
	 * 変更データを初期値(元の取り込んだ状態)に戻します。
	 *
	 * 変更タイプ(追加/変更/削除)に応じて、処理されます。
	 * 追加時は、追加された行を削除します。
	 * 変更時は、変更された行を元に戻します。
	 * 削除時は、削除フラグを解除します。
	 * それ以外の場合(変更されていない場合)は、なにもしません。
	 *
	 * @param   row  処理を戻す(取り消す)行
	 */
	void resetRow( int row ) ;

	/**
	 * 書込み許可を返します。
	 *
	 * @param   aRow     値が参照される行
	 *
	 * @return  書込み可能(true)／不可能(false)
	 */
	boolean isRowWritable( int aRow ) ;

	/**
	 * 書き込み可能な行(rowWritable == true)のチェックボックスに対して
	 * 初期値を 選択済みか、非選択済みかを返します。
	 *
	 * @param   row      値が参照される行
	 *
	 * @return	初期値チェックON(true)／チェックOFF(false)
	 */
	boolean isRowChecked( int row ) ;

	/**
	 * 検索結果が オーバーフローしたかどうかをチェックします。
	 * Query で検索した場合に、DB_MAX_ROW_COUNT または、Query.setMaxRowCount( int maxRowCount )
	 * で指定された値よりも検索結果が多い場合に、DBTableModel は、先の設定値までの
	 * データを取り込みます。そのときに、オーバーフローフラグを立てておくことで、最大件数を
	 * オーバーしたかどうかを判断します。
	 *
	 * @return   オーバーフロー(true)／正常(false)
	 */
	boolean isOverflow() ;

	/**
	 * 検索されたDBTableModelが登録時に同一かどうかを判断する為の 整合性キーを取得します。
	 *
	 * ここでの整合性は、同一セッション(ユーザー)毎にユニークかどうかで対応します。
	 * 分散環境(複数のセッション間)での整合性は、確保できません。
	 * 整合性キー は、オブジェクト作成時刻としますが、将来変更される可能性があります。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規追加
	 *
	 * @return   整合性キー(オブジェクトの作成時刻)
	 */
	String getConsistencyKey() ;

	// =======================================================================
	//
	// TableModel Interface と共有していたメソッドを、独自に再定義します。
	//
	// =======================================================================

	/**
	 * データテーブル内の列の数を返します。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) インターフェースの見直しにより、追加
	 *
	 * @return  モデルの列数
	 */
	int getColumnCount() ;

	/**
	 * カラム名を取得します。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) インターフェースの見直しにより、追加
	 *
	 * @param   column  最初のカラムは 0、2番目のカラムは 1、などとする。
	 *
	 * @return  カラム名
	 */
	String getColumnName(int column) ;

	// =======================================================================
	//
	// DBTableModelImpl.java で定義されている public メソッド残り全て
	//
	// =======================================================================

	/**
	 * column に対応した 値を登録します。
	 * column には、番号ではなく､ラベルを指定します。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   aRow    値が変更される行
	 * @param   columnName    値が変更されるカラム名
	 * @param   value   新しい値。null も可
	 *
	 */
	void setValue( int aRow, String columnName, String value ) ;

	/**
	 * column および row にあるセルのオブジェクト値を設定します。
	 * value は新しい値です。このメソッドは、tableChanged() 通知を生成します。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.5.3.1 (2003/10/31) インターフェースの見直しにより、private 化する。
	 * @og.rev 4.0.0.0 (2007/05/24) インターフェースの見直しにより、public 化する。
	 *
	 * @param   value   新しい値。null も可
	 * @param   aRow    値が変更される行
	 * @param   aColumn 値が変更される列
	 */
	void setValueAt( String value, int aRow, int aColumn ) ;

	/**
	 * 行を削除します。
	 * 物理削除ではなく、論理削除です。
	 * データを取り込むことは可能です。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   aRow    論理削除される行
	 *
	 */
	void rowDelete( int aRow ) ;

	/**
	 * row にあるセルのオブジェクト値を置き換えて、行を削除します。
	 * 物理削除ではなく、論理削除です。
	 * 値を置き換えたデータを取り込むことが可能です。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   values  新しい配列値。
	 * @param   aRow    論理削除される行
	 *
	 */
	void rowDelete( String[] values, int aRow ) ;

	/**
	 * 行を物理削除します。
	 * メモリ上で編集する場合に使用しますが,一般アプリケーションからの
	 * 使用は、物理削除の為,お勧めいたしません。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   aRow    物理削除される行
	 *
	 */
	void removeValue( int aRow ) ;

	/**
	 * row の下に属性値配列を追加登録します。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   values  属性値配列
	 * @param   aRow    値が参照される行
	 *
	 */
	void addValues( String[] values ,int aRow ) ;

	/**
	 * row の下に属性値配列を追加登録します。
	 * isWritableをfalseにした場合、編集不可能な状態で追加されます。
	 *
	 * @og.rev 4.3.1.0 (2008/09/04) interface に新規登録
	 *
	 * @param   values  属性値配列
	 * @param   aRow    値が参照される行
	 * @param   isWritable 編集不可能な状態で追加するか
	 *
	 */
	void addValues( String[] values ,int aRow, boolean isWritable ) ;

	/**
	 * row あるセルの属性値配列を追加登録します。
	 * これは,初期登録時のみに使用します。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   values  属性値配列
	 */
	void addColumnValues( String[] values ) ;

	/**
	 * row あるセルの属性値配列を追加登録します。
	 * これは,初期登録時のみに使用します。
	 * このメソッドでは、同時に、変更タイプ と、書込み許可を指定できます。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) interface に変更タイプ と、書込み許可を追加
	 *
	 * @param   values   属性値配列
	 * @param   modType  変更タイプ(追加/変更/削除)
	 * @param   rw 書込み可能(true)／不可能(false)
	 */
	void addColumnValues( String[] values , String modType , boolean rw ) ;

	/**
	 * 変更済みフラグを元に戻します。
	 *
	 * 一般には,データベースにテーブルモデルを登録するタイミングで、
	 * 変更済みフラグを元に戻します。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   aRow     値が参照される行
	 */
	void resetModify( int aRow ) ;

	/**
	 * 行が書き込み可能かどうかをセットします。
	 * デフォルト/およびなにも設定しない場合は, DEFAULT_WRITABLE が
	 * 与えられています。
	 * これが true の場合は,書込み許可です。(チェックボックスを表示)
	 * false の場合は,書込み不許可(チェックボックスは表示されません。)
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   aRow     値が参照される行
	 * @param   rw 書込み可能(true)／不可能(false)
	 */
	void setRowWritable( int aRow ,boolean rw ) ;

	/**
	 * 書き込み可能な行(rowWritable == true)のチェックボックスに対して
	 * 初期値を 選択済みにするか、非選択済みにするかを指定します。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   aRow      値が参照される行
	 * @param   rw チェックON(true)／チェックOFF(false)
	 */
	void setRowChecked( int aRow ,boolean rw ) ;

	/**
	 * 行指定の書込み許可を与えます。
	 * 具体的には,チェックボックスの表示/非表示を指定します。
	 * これが true の場合は,書込み許可です。(チェックボックスを表示)
	 * false の場合は,書込み不許可(チェックボックスは表示されません。)
	 * 行毎に書込み許可/不許可を指定する場合は,１カラム目に writable
	 * カラムを用意して true/false を指定します。
	 * この writable カラムとの論理積により最終的にチェックボックスの
	 * 表示の ON/OFF が決まります。
	 * なにも設定しない場合は, ViewForm.DEFAULT_WRITABLE が設定されます。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   rw 書込み可能(true)／不可能(false)
	 */
	void setDefaultRowWritable( boolean rw ) ;

	/**
	 * 書き込み可能な行(rowWritable == true)のチェックボックスに対して
	 * 初期値を 選択済みにするか、非選択済みにするかを指定します。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   rw 選択状態(true)／非選択状態(false)
	 */
	void setDefaultRowChecked( boolean rw ) ;

	/**
	 * 検索結果が オーバーフローしたかどうかを設定します。
	 * Query で検索した場合に、DB_MAX_ROW_COUNT または、Query.setMaxRowCount( int maxRowCount )
	 * で指定された値よりも検索結果が多い場合に、DBTableModel は、先の設定値までの
	 * データを取り込みます。そのときに、オーバーフローフラグを立てておくことで、最大件数を
	 * オーバーしたかどうかを判断します。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) interface に新規登録
	 *
	 * @param   of オーバーフロー(true)／正常(false)
	 */
	void setOverflow( boolean of ) ;

	/**
	 * カラム(列)にmustタイプ値を割り当てます。
	 * この値は、columnCheck 時の nullCheck や mustAnyCheck の
	 * チェック対象カラムとして認識されます。
	 *
	 * @og.rev 4.1.2.1 (2008/03/13) interface に新規登録
	 *
	 * @param   dbColumn  カラムオブジェクト
	 * @param   type      mustタイプ(must,mustAny)
	 */
	void addMustType( int dbColumn, String type ) ;

	/**
	 * mustType="must"時のカラム名を、CSV形式として返します。
	 * この値は、columnCheck 時の nullCheck のチェック対象カラムとして
	 * 認識されます。
	 * カラム名配列は、ソート済みです。
	 *
	 * @og.rev 4.1.2.1 (2008/03/13) interface に新規登録
	 *
	 * @return  mustType="must"時のカラム名配列(ソート済み)
	 */
	String[] getMustArray();

	/**
	 * mustType="mustAny" 他のカラム名を、CSV形式として返します。
	 * この値は、columnCheck 時の mustAnyCheck のチェック対象カラムとして
	 * 認識されます。
	 * カラム名配列は、ソート済みです。
	 *
	 * @og.rev 4.1.2.1 (2008/03/13) interface に新規登録
	 *
	 * @return  mustType="mustAny"時のカラム名配列(ソート済み)
	 */
	String[] getMustAnyArray();
}
