/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.io.AbstractTableWriter;

/**
 * 区切り文字指定(初期値:タブ)ゼロカンマファイルの書き出しクラスです。
 *
 * ラベル，名前，データの出力部のみオーバーライドすれば，各種出力フォーマットに合わせた
 * サブクラスを実現する事が可能です。
 * ゼロカンマファイルとは、EXCELのゼロサプレス対策として、頭ゼロの文字型データを出力する
 * 時に、先頭にカンマ(')を付けて、ゼロが削除(見えなくなる)現象を抑止しているファイルです。
 *
 * このクラスは,可変長タブ区切り文字ファイルの出力機能を実現しています。
 *
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableWriter_Default extends AbstractTableWriter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TableWriter_Default() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * DBTableModel から 各形式のデータを作成して,PrintWriter に書き出します。
	 * このメソッドは、EXCEL 書き出し時に使用します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @see #isExcel()
	 */
	@Override
	public void writeDBTable() {
		final String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * DBTableModel から データを作成して,PrintWriter に書き出します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.5.4.2 (2003/12/15) writer の null チェックを廃止します。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に PrintWriter を受け取るように変更します。
	 *
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	public void writeDBTable( final PrintWriter writer )  {
		if( ! createDBColumn() ) { return ; }

		final String headerSequence = getHeaderSequence();

		writeHeader( writer ) ;
	//	String msg = null;
		for( int i=0; i<headerSequence.length(); i++ ) {
			switch( headerSequence.charAt( i ) ) {
				case 'N':
	//				msg = "N:writeName";
					writeName( table,writer ) ;
					break;
				case 'L':
	//				msg = "L:writeLabel";
					writeLabel( table,writer ) ;
					break;
				case 'S':
	//				msg = "S:writeSize";
					writeSize( table,writer ) ;
					break;
				case 'C':
	//				msg = "C:writeClass";
					writeClass( table,writer ) ;
					break;
				case '-':
	//				msg = "-:separator";
					writeSeparator( table,writer ) ;
					break;
				case 'D':
	//				msg = "D:writeData";
					writeData( table,writer ) ;
					break;
				default:
					break;
			}
		}
	}

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 * このクラスでは，データを ダブルコーテーション(")で囲みます。
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * @og.rev 2.0.0.5 (2002/09/30) 先頭が0 でかつ数字タイプ(S9 or R)でない場合に ' を出力するように修正。
	 * @og.rev 2.3.1.2 (2003/01/28) データ出力時に、改行が余分に出される箇所を修正。
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumn から、getDbType() キーを直接取り出す
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.3.3.1 (2003/07/18) ファイルリード/ライト時に後ろスペースの除去を行います。
	 * @og.rev 3.7.0.2 (2005/02/14) 行番号情報を、出力する(true)/しない(false)を指定
	 * @og.rev 3.8.0.1 (2005/06/17) DBTypeが NVAR の場合は、元のUnicodeに戻します。
	 * @og.rev 5.1.6.0 (2010/05/01) DbType の初期値(dbType)を利用する。
	 * @og.rev 5.2.1.0 (2010/10/01) useRenderer 対応
	 * @og.rev 5.2.2.0 (2010/11/01) 改行、ダブルクオート等の処理
	 * @og.rev 5.3.2.0 (2011/02/01) ","を付加する条件が反転しているバグを修正
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		final int numberOfRows =	table.getRowCount();

		final String separator = getSeparator();
		final boolean useRenderer = isUseRenderer();	// 5.2.1.0 (2010/10/01)

		for( int row=0; row<numberOfRows; row++ ) {
			if( isUseNumber() ) {
				writer.print( row+1 );
				writer.print( separator );
			}

			for( int i=0; i<numberOfColumns; i++ ) {
				if( i != 0 ) { writer.print( separator ); }

				final int clm = clmNo[i];
				if( clm < 0 ) { continue; }			// 6.0.1.2 (2014/08/08) カラム飛ばし

				String val = table.getValue(row,clm);

				// 6.2.3.0 (2015/05/01) keyLabelのデータは、先に書き込みます。
				if( isKeyLbl[i] ) {
					writer.print( StringUtil.rTrim( val ) );
					writer.print( separator );
				}

				if( dbType[i] == NVAR ) {
					val = StringUtil.getReplaceEscape( val );
				}
				// 5.2.1.0 (2010/10/01) useRenderer 対応
				else if( useRenderer ) {
					// 6.0.4.0 (2014/11/28) データ出力用のレンデラー
					val = dbColumn[clm].getWriteValue( val );
				}

				if( StringUtil.startsChar( val , '0' ) && dbType[i] != NUMBER ) {		// 6.2.0.0 (2015/02/27) １文字 String.startsWith
						writer.print( "'" );		// 開始日などの 00000000 を文字列タイプで渡す
				}
				writer.print( quotation2( StringUtil.rTrim( val ) ) );	// 5.2.2.0 (2010/11/01) 改行、ダブルクオート等の処理
			}
			writer.println();
		}
	}
}
