/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.xml;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import java.util.List;
import java.util.ArrayList;

import org.opengion.fukurou.system.HybsConst;							// 6.1.0.0 (2014/12/26) refactoring

/**
 * ノードの基底クラスとなる、OGNode クラスを定義します。
 *
 * OGElement、OGDocument は、この、OGNode クラスを継承します。
 * ただし、OGAttributes は、独立しているため、このクラスは継承していません。
 *
 * 最も一般的なノードは、テキストノードであり、
 *
 * OGNode は、enum OGNodeType で区別される状態を持っています。
 * その内、OGElement と OGDocument は、サブクラスになっています。
 * OGNodeType は、それぞれ、再設定が可能です。
 * 例えば、既存のエレメントやノードに対して、コメントタイプ(Comment)を指定すると、
 * ファイル等への出力時にコメントとして出力されます。
 *
 * 　　List　　　:内部に、OGNode の ArrayList を持つ
 * 　　Text　　　:内部は、文字列の BODY 部分を持つ
 * 　　Comment 　:内部は、文字列であるが、toString() 時には、コメント記号を前後に出力する。
 * 　　Cdata 　　:内部は、TextNodeのArrayList を持つ、toString() 時には、Cdataを前後に出力する。
 * 　　Element 　:タグ名、属性、OGNode の ArrayList の入れ子状態をもつ
 * 　　Document　:トップのElement として、read/write するときに使用。構造は、唯一の OGElement を持つ List タイプ
 *
 * @og.rev 5.1.8.0 (2010/07/01) 新規作成
 * @og.rev 5.6.1.2 (2013/02/22) 構想からやり直し
 *
 * @version  5.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class OGNode {
	/** システムの改行コードを設定します。*/
	protected static final String CR		 = HybsConst.CR;			// 6.1.0.0 (2014/12/26) refactoring
	/** StringBilderなどの初期値を設定します。	{@value} */
	protected static final int BUFFER_MIDDLE = HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

	private final List<OGNode> nodes = new ArrayList<>();	// ノードリスト
	private final String	text		;						// テキストノード用の文字列ノード値
	private OGNodeType		nodeType	;						// List,Text,Comment,Cdata,Element,Document
	private OGNode 			parentNode	;						// 自身の親ノード(ただし、最終セットされたノード)

	/**
	 * デフォルトコンストラクター
	 *
	 * ここでは、NodeType は、List に設定されます。
	 */
	public OGNode() {
		this.text  = null;
		nodeType   = OGNodeType.List;
	}

	/**
	 * テキストノードを構築するためのコンストラクター
	 *
	 * テキストノードは、簡易的に、内部には、ノードリストではなく文字列を持っています。
	 *
	 * @og.rev 5.6.1.2 (2013/02/22) 内部テキストがない場合のタグの終了時にスペースは入れない。
	 *
	 * ここでは、NodeType は、Text に設定されます。
	 * ただし、引数のテキストが null のNodeType は、List に設定されます。
	 *
	 * @param	txt	テキストノードの設定値
	 */
	public OGNode( final String txt ) {
		text = txt ;
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		nodeType = text == null ? OGNodeType.List : OGNodeType.Text;

	}

	/**
	 * テキストノードをノードリストに追加します。
	 *
	 * 内部的にテキストノードを構築して、リストに追加しています。
	 * 戻り値は、StringBuilder#append(String) の様に、連結登録できるように
	 * 自分自身を返しています。
	 * テキストノードに、この処理を行うと、エラーになります。
	 * 一旦、テキストノードとして作成したノードには、ノードを追加できません。
	 *
	 * @param	txt	テキストノードの設定値
	 *
	 * @return	自分自身(this)のノード
	 * @og.rtnNotNull
	 */
	public OGNode addNode( final String txt ) {
		if( txt != null ) {
			if( nodeType == OGNodeType.Text ) {
				// テキストノードにノードは追加できません。
				final String errMsg = "一旦、テキストノードとして作成したノードには、ノードを追加できません。";
				throw new OgRuntimeException( errMsg );
			}

			final OGNode node = new OGNode( txt );
			node.parentNode = this;
			nodes.add( node );
		}
		return this;
	}

	/**
	 * ノードをノードリストに追加します。
	 *
	 * 追加するノードの親として、自分自身を登録します。
	 * なお、同じオブジェクトを、複数の親に追加する場合(ノードリストには追加可能)は、
	 * 親ノードは、最後に登録されたノードのみが設定されます。
	 * テキストノードに、この処理を行うと、エラーになります。
	 * 一旦、テキストノードとして作成したノードには、ノードを追加できません。
	 *
	 * @param	node	ノード
	 *
	 * @return	自分自身(this)のノード
	 * @og.rtnNotNull
	 */
	public OGNode addNode( final OGNode node ) {
		if( node != null ) {
			if( nodeType == OGNodeType.Text ) {
				// テキストノードにノードは追加できません。
				final String errMsg = "一旦、テキストノードとして作成したノードには、ノードを追加できません。";
				throw new OgRuntimeException( errMsg );
			}

			node.parentNode = this;
			nodes.add( node );
		}
		return this;
	}

	/**
	 * ノードリストに追加されている、ノードの個数を返します。
	 *
	 * @return	ノードリストの数
	 */
	public int nodeSize() {
		return nodes.size();
	}

	/**
	 * ノードリストに追加されている、ノードを返します。
	 *
	 * ノードの指定には、配列番号を使用します。
	 * ノードの個数は、事前に、nodeSize() で調べて置いてください。
	 * 当然、テキストノードの場合は、nodeSize()==0 なので、
	 * このメソッドでは取得できません。
	 *
	 * @param	adrs	ノードリストの位置
	 *
	 * @return	指定の配列番号のノード
	 */
	public OGNode getNode( final int adrs ) {
		return nodes.get(adrs);
	}

	/**
	 * ノードリストに、ノードをセットします。
	 *
	 * ノードリストの指定のアドレスに、ノードをセットします。
	 * これは、追加ではなく置換えになります。
	 * ノードの指定には、配列番号を使用します。
	 * ノードの個数は、事前に、nodeSize() で調べて置いてください。
	 *
	 * @param	adrs	ノードリストの位置
	 * @param	node	セットするノード
	 */
	public void setNode( final int adrs , final OGNode node ) {
		nodes.set(adrs,node);
	}

	/**
	 * 自身にセットされている、親ノードを返します。
	 *
	 * 親ノードは、自身のオブジェクトに、一つしか設定できません。
	 * これは、オブジェクトとして、同一ノードを、複数の親ノードに
	 * 追加した場合(これは、ノードリストへの追加なので可能)最後に追加した
	 * 親ノードのみ、保持していることになります。
	 * XML を構築するときは、同一のノードであっても、毎回、作成しなおさないと、
	 * 親ノードを見つけて、何かを行う場合には、おかしな動きをすることになります。
	 * なお、ノードオブジェクト自体が、親ノードから削除されても、自身の
	 * 親ノード情報は保持し続けています。
	 * ある Element から削除したノードを別のElementに追加すると、その時点で、
	 * 親ノードも更新されます。
	 *
	 * @return	親ノード
	 */
	public OGNode getParentNode() {
		return parentNode;
	}

	/**
	 * 自身にセットされている、親ノードの階層数を返します。
	 *
	 * 自身のオブジェクトに設定されている親ノードを順番にさかのぼって、
	 * 何階層あるか返します。
	 * これは、getText(int) の引数に使えます。
	 * 親ノードがひとつもない場合、つまり自身が最上位の場合は、0 が返されます。
	 *
	 * @return	自身の階層
	 */
	public int getParentCount() {
		int para = 0;
		OGNode node = getParentNode();
		while( node != null ) {
			para++ ;
			node = node.getParentNode();
		}
		return para;
	}

	/**
	 * ノードリストから、指定の配列番号の、ノードを削除します。
	 *
	 * ノードの指定には、配列番号を使用します。
	 * ノードの個数は、事前に、nodeSize() で調べて置いてください。
	 *
	 * @param	adrs	ノードリストの位置
	 *
	 * @return	削除されたノード
	 */
	public OGNode removeNode( final int adrs ) {
		return nodes.remove(adrs);
	}

	/**
	 * ノードリストから、すべてのノードを削除します。
	 *
	 * これは、ノードリストをクリアします。
	 *
	 */
	public void clearNode() {
		nodes.clear();
	}

	/**
	 * ノードリストから、指定のノード(orgNode)を新しいノード(newNode)に置き換えます。
	 *
	 * ノードは、それぞれ、ノードが作成された順番で、ユニークな番号を持っています。
	 * その番号を元に、ノードを探し出して、置き換えます。
	 * 通常の、XMLパースから作成されたノードは、すべて一意にユニーク番号が振られますが、
	 * 新しくつったノードを複数のノードと置き換える場合、置き換えられた後のノードは、
	 * オブジェクトそのものが、同一になるため、注意が必要です。
	 *
	 * @param	orgNode	置換元のオリジナルノード
	 * @param	newNode	置換する新しいノード
	 */
	public void changeNode( final OGNode orgNode , final OGNode newNode ) {
		final int size = nodes.size();
		for( int i=0; i<size; i++ ) {
			final OGNode node = nodes.get(i);
			if( node.equals( orgNode ) ) {		// Object.equals なので、オブジェクトそのものの一致判定
				nodes.set( i,newNode );
			}
			else {
				node.changeNode( orgNode,newNode );
			}
		}
	}

	/**
	 * ノードリストから、直下(メンバー)のエレメントのみをリストにして返します。
	 *
	 * ノードリストの第一レベルで、エレメントのみを返します。
	 * 通常は、あるエレメントを、getElementList( String ) 等で検索した後、その子要素を
	 * 取り出す場合に使用します。
	 * 該当するエレメントが、なにも存在しない場合は、空のリストオブジェクトが返されます。
	 *
	 * @return	直下(メンバー)のエレメントのリスト
	 */
	public List<OGElement> getChildElementList() {
		final List<OGElement> eles = new ArrayList<>();

		for( final OGNode node : nodes ) {
			if( node.nodeType == OGNodeType.Element ) {
				eles.add( (OGElement)node );
			}
		}

		return eles;
	}

	/**
	 * ノードリストから、下位の階層に存在するすべてのエレメントをリストにして返します。
	 *
	 * エレメントは、名前を指定して検索します。
	 * 該当するエレメントが、なにも存在しない場合は、空のリストオブジェクトが返されます。
	 *
	 * @param	qName	エレメントの名前
	 *
	 * @return	下位の階層に存在するすべてのエレメントのリスト
	 */
	public List<OGElement> getElementList( final String qName ) {
		final List<OGElement> eles = new ArrayList<>();

		if( qName != null ) {
			for( final OGNode node : nodes ) {
				if( node.nodeType == OGNodeType.Element ) {
					final OGElement ele = (OGElement)node;
					if( qName.equals( ele.getTagName() ) ) {
						eles.add( ele );
					}
					eles.addAll( ele.getElementList( qName ) );
				}
			}
		}

		return eles;
	}

	/**
	 * ノードタイプを設定します。
	 *
	 * ノードタイプとは、List , Text , Comment , Cdata , Element , Document などの
	 * ノードの種別を表す enum タイプです。
	 * 基本的には、オブジェクトの取得時に、ファクトリメソッド経由であれば、自動的に設定
	 * されています。
	 * ここでは、可変設定できます。
	 * 例えば、既存のエレメントやノードに対して、コメントタイプ(Comment)を指定すると、
	 * ファイル等への出力時にコメントとして出力されます。
	 * null を指定すると、なにも処理されません。
	 *
	 * @param	type	enumのOGNodeType
	 * @see OGNodeType
	 */
	public void setNodeType( final OGNodeType type ) {
		if( type != null ) {
			if( type != OGNodeType.Text && nodeType == OGNodeType.Text ) {
				final OGNode node = new OGNode( text );
				node.parentNode = this;
				nodes.add( node );
			}

			nodeType = type ;
		}
	}

	/**
	 * ノードタイプを取得します。
	 *
	 * ノードタイプとは、List , Text , Comment , Cdata , Element , Document などの
	 * ノードの種別を表す enum タイプです。
	 * 基本的には、オブジェクトの取得時に、ファクトリメソッド経由であれば、自動的に設定
	 * されています。
	 *
	 * @return	ノードタイプ
	 * @see OGNodeType
	 */
	public OGNodeType getNodeType() {
		return nodeType;
	}

	/**
	 * ノードリストの文字列を返します。
	 *
	 * これは、タグで言うところのBODY部に書かれた文字列に相当します。
	 * 該当する文字列が、存在しない場合は、空の文字列(ゼロストリング)が返されます。
	 *
	 * @param	cnt		Nodeの階層
	 * @return	ノードリストの文字列(BODY部に書かれた文字列)
	 */
	public String getText( final int cnt ) {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		if( nodeType == OGNodeType.Text ) {
			buf.append( text );
		}
		else {
			for( final OGNode node : nodes ) {
				buf.append( node.getText( cnt ) );
			}
		}

		// 6.4.2.1 (2016/02/05) PMD refactoring. Prefer StringBuffer over += for concatenating strings
		switch( nodeType ) {
			case Comment:	buf.insert( 0,"<!-- "      ).append( "-->"  ); break;
			case Cdata:		buf.insert( 0,"<![CDATA[ " ).append( " ]]>" ); break;
	//		case Document:
	//		case Text:
	//		case DTD:
	//		case List:
			default:		break;
		}

		return buf.toString() ;

	}

	/**
	 * オブジェクトの文字列表現を返します。
	 *
	 * 文字列は、OGNodeType により異なります。
	 * Comment ノードの場合は、コメント記号を、Cdata ノードの場合は、CDATA を
	 * つけて出力します。
	 *
	 * @return	このオブジェクトの文字列表現
	 * @og.rtnNotNull
	 * @see Object#toString()
	 */
	@Override
	public String toString() {
		return getText( -10 );
	}
}
