/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.system.OgRuntimeException ;				// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.system.OgCharacterException ;			// 6.5.0.1 (2016/10/21)
import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.system.Closer ;
import org.opengion.fukurou.system.LogWriter;

import java.util.Map ;
import java.util.LinkedHashMap ;

import java.io.File;
import java.io.BufferedReader;
import java.io.IOException;
import java.nio.charset.CharacterCodingException;					// 6.3.1.0 (2015/06/28)

/**
 * Process_TableReaderは、ファイルから読み取った内容を、LineModel に設定後、
 * 下流に渡す、FirstProcess インターフェースの実装クラスです。
 *
 * DBTableModel 形式のファイルを読み取って、各行を LineModel にセットして、
 * 下流(プロセスチェインのデータは上流から下流に渡されます。)に渡します。
 *
 * columns 属性は、#NAME で列カラムを外部から指定する場合に使用します。
 * この属性とuseNumber属性は独立していますが、一般には、#NAME を指定
 * する場合は、useNumber="true"として、行番号欄は使用しますし、外部から
 * 指定する場合は、useNumber="false"にして先頭から読み取ります。
 * (自動セットではないので、必要に応じて設定してください)
 * useNumber の初期値は、"true" です。
 *
 * ※ 注意
 *  Process_TableReader では、セパレータ文字 で区切って読み込む処理で、前後のｽﾍﾟｰｽを
 *  削除しています。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * @og.formSample
 *  Process_TableReader -infile=INFILE -sep=, -encode=UTF-8 -columns=AA,BB,CC
 *
 *    -infile=入力ファイル名     ：入力ファイル名
 *   [-existCheck=存在確認     ] ：ファイルが存在しない場合エラーにする(初期値:true)
 *   [-sep=セパレータ文字      ] ：区切り文字(初期値:タブ)
 *   [-encode=文字エンコード   ] ：入力ファイルのエンコードタイプ
 *   [-columns=読み取りカラム名] ：入力カラム名(CSV形式)
 *   [-useNumber=[true/false]  ] ：行番号を使用する(true)か使用しない(false)か。
 *   [-display=[false/true]    ] ：結果を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *   [-debug=[false/true]      ] ：デバッグ情報を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_TableReader extends AbstractProcess implements FirstProcess {
	private char		  	separator	= TAB;		// 6.0.2.5 (2014/10/31) TAB を char 化
	private String			infile		;
	private String			encode		;			// 6.3.1.0 (2015/06/28) デバッグ時に使用
	private BufferedReader	reader		;
	private LineModel		model		;
	private String			line		;
	private int[]			clmNos		;			// ファイルのヘッダーのカラム番号
	private boolean			useNumber	= true;		// 5.2.2.0 (2010/11/01) 行番号を使用する(true)か使用しない(false)か
	private boolean			nameNull	;			// ０件データ時 true
	private boolean			display		;			// 表示しない
	private boolean			debug		;			// 5.7.3.0 (2014/02/07) デバッグ情報

	private int				inCount		;
	private int				outCount	;

	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> MUST_PROPARTY   ;		// ［プロパティ］必須チェック用 Map
	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> USABLE_PROPARTY ;		// ［プロパティ］整合性チェック Map

	static {
		MUST_PROPARTY = new LinkedHashMap<>();
		MUST_PROPARTY.put( "infile",	"入力ファイル名 (必須)" );

		USABLE_PROPARTY = new LinkedHashMap<>();
		USABLE_PROPARTY.put( "existCheck",	"ファイルが存在しない場合エラーにする(初期値:true)" );
		USABLE_PROPARTY.put( "sep",			"区切り文字(初期値:タブ)" );
		USABLE_PROPARTY.put( "encode",		"入力ファイルのエンコードタイプ" );
		USABLE_PROPARTY.put( "columns",		"入力カラム名(CSV形式)" );
		USABLE_PROPARTY.put( "useNumber",	"行番号を使用する(true)か使用しない(false)か" );	// 5.2.2.0 (2010/11/01)
		USABLE_PROPARTY.put( "display",		"結果を標準出力に表示する(true)かしない(false)か" +
											CR + " (初期値:false:表示しない)" );
		USABLE_PROPARTY.put( "debug",	"デバッグ情報を標準出力に表示する(true)かしない(false)か" +
											CR + "(初期値:false:表示しない)" );		// 5.7.3.0 (2014/02/07) デバッグ情報
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_TableReader() {
		super( "org.opengion.fukurou.process.Process_TableReader",MUST_PROPARTY,USABLE_PROPARTY );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) useNumber属性の追加
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		final Argument arg = getArgument();

		infile				= arg.getProparty( "infile" );
		encode				= arg.getProparty( "encode"		, System.getProperty( "file.encoding" ) ); // 6.3.1.0 (2015/06/28) デバッグ時に使用
		useNumber			= arg.getProparty( "useNumber"	, useNumber );		// 5.2.2.0 (2010/11/01)
		display				= arg.getProparty( "display"	, display );
		debug				= arg.getProparty( "debug"		, debug );			// 5.7.3.0 (2014/02/07) デバッグ情報

		// 6.0.2.5 (2014/10/31) TAB を char 化
		final String sep = arg.getProparty( "sep",null );
		if( sep != null ) { separator = sep.charAt(0); }

		if( infile == null ) {
			final String errMsg = "ファイル名が指定されていません。" ;
			throw new OgRuntimeException( errMsg );
		}

		final File file = new File( infile );

		if( ! file.exists() ) {
			// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
			final boolean existCheck	= arg.getProparty("existCheck",true);
			if( existCheck ) {
				final String errMsg = "ファイルが存在しません。File=[" + file + "]" ;
				throw new OgRuntimeException( errMsg );
			}
			else {
				nameNull = true; return ;
			}
		}

		if( ! file.isFile() ) {
			final String errMsg = "ファイル名を指定してください。File=[" + file + "]" ;
			throw new OgRuntimeException( errMsg );
		}

		reader = FileUtil.getBufferedReader( file,encode );

		final String[] names ;
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final String  clms	= arg.getProparty("columns" );
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( clms == null ) {
			// 5.2.2.0 (2010/11/01) names の外部指定の処理を先に行う。
			final String[] clmNames = readName( reader );		// ファイルのカラム名配列
			if( clmNames == null || clmNames.length == 0 ) { nameNull = true; return ; }
			names = clmNames;
		}
		else {
			names = StringUtil.csv2Array( clms );	// 指定のカラム名配列
		}

		model = new LineModel();
		model.init( names );

		if( display ) { println( model.nameLine() ); }

		clmNos = new int[names.length];
		for( int i=0; i<names.length; i++ ) {
			final int no = model.getColumnNo( names[i] );
			// 5.2.2.0 (2010/11/01) useNumber="true"の場合は、行番号分を＋１しておく。
			if( no >= 0 ) { clmNos[no] = useNumber ? i+1 : i ; }
		}
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		Closer.ioClose( reader );
		reader = null;
	}

	/**
	 * このデータの処理において、次の処理が出来るかどうかを問い合わせます。
	 * この呼び出し１回毎に、次のデータを取得する準備を行います。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) ""で囲われているデータに改行が入っていた場合の対応
	 * @og.rev 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
	 * @og.rev 6.5.0.1 (2016/10/21) CharacterCodingException は、OgCharacterException に変換する。
	 *
	 * @return	処理できる:true / 処理できない:false
	 */
	public boolean next() {
		if( nameNull ) { return false; }

		boolean flag = false;
		try {
			final StringBuilder buf = new StringBuilder( BUFFER_LARGE );		// 6.1.0.0 (2014/12/26) refactoring
			while((line = reader.readLine()) != null) {
				inCount++ ;
				if( line.isEmpty() || line.charAt(0) == '#' ) { continue; }
				else {
					// 5.2.2.0 (2010/11/01) findbugs 対策(文字列の + 連結と、奇数判定ロジック)
					int quotCount = StringUtil.countChar( line, '"' );
					if( quotCount % 2 != 0 ) {
						String addLine = null;
						buf.setLength(0);							// 6.1.0.0 (2014/12/26) refactoring
						buf.append( line );							// 6.1.0.0 (2014/12/26) refactoring
						while(quotCount % 2 != 0 && (addLine = reader.readLine()) != null) {
							if( addLine.isEmpty() || addLine.charAt(0) == '#' ) { continue; }
							buf.append( CR ).append( addLine );
							quotCount += StringUtil.countChar( addLine, '"' );
						}
						line = buf.toString();
					}
					flag = true;
					break;
				}
			}
		}
		// 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
		catch( final CharacterCodingException ex ) {
			final String errMsg = "文字のエンコード・エラーが発生しました。" + CR
								+	"  ファイルのエンコードが指定のエンコードと異なります。" + CR
								+	" [" + infile + "] , Encode=[" + encode + "]" ;
			throw new OgCharacterException( errMsg,ex );	// 6.5.0.1 (2016/10/21)
		}
		catch( final IOException ex) {
			final String errMsg = "ファイル読込みエラーが発生しました。" + CR
								+	" [" + infile + "] , Encode=[" + encode + "]" ;
			throw new OgRuntimeException( errMsg,ex );
		}
		if( debug ) { println( line ); }			// 5.7.3.0 (2014/02/07) デバッグ情報
		return flag;
	}

	/**
	 * 最初に、 行データである LineModel を作成します
	 * FirstProcess は、次々と処理をチェインしていく最初の行データを
	 * 作成して、後続の ChainProcess クラスに処理データを渡します。
	 *
	 * ファイルより読み込んだ１行のデータを テーブルモデルに
	 * セットするように分割します
	 * なお、読込みは，NAME項目分を読み込みます。データ件数が少ない場合は、
	 * "" をセットしておきます。
	 *
	 * @param	rowNo	処理中の行番号
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel makeLineModel( final int rowNo ) {
		outCount++ ;
		final String[] vals = StringUtil.csv2Array( line ,separator );	// 6.0.2.5 (2014/10/31) TAB を char 化

		final int len = vals.length;
		for( int clmNo=0; clmNo<model.size(); clmNo++ ) {
			final int no = clmNos[clmNo];
			if( len > no ) {
				model.setValue( clmNo,vals[no] );
			}
			else {
				// EXCEL が、終端TABを削除してしまうため、少ない場合は埋める。
				model.setValue( clmNo,"" );
			}
		}
		model.setRowNo( rowNo ) ;

		if( display ) { println( model.dataLine() ); }

		return model;
	}

	/**
	 * BufferedReader より、#NAME 行の項目名情報を読み取ります。
	 * データカラムより前に、項目名情報を示す "#Name" が存在する仮定で取り込みます。
	 * この行は、ファイルの形式に無関係に、TAB で区切られています。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) #NAME 行の区切り文字は、指定の区切り文字を優先して利用する。
	 * @og.rev 6.0.4.0 (2014/11/28) #NAME 判定で、桁数不足のエラーが発生する箇所を修正。
	 * @og.rev 6.3.9.0 (2015/11/06) #NAME 行の区切り文字判定が間違っていたので修正。
	 * @og.rev 6.5.0.1 (2016/10/21) CharacterCodingException は、OgCharacterException に変換する。
	 * @og.rev 6.5.0.1 (2016/10/21) CharacterCodingException は、OgCharacterException に変換する。
	 *
	 * @param 	reader PrintWriterオブジェクト
	 *
	 * @return	カラム名配列(存在しない場合は、サイズ０の配列)
	 * @og.rtnNotNull
	 */
	private String[] readName( final BufferedReader reader ) {
		try {
			// 4.0.0 (2005/01/31) line 変数名変更
			String line1;
			while((line1 = reader.readLine()) != null) {
				inCount++ ;
				if( line1.isEmpty() ) { continue; }
				if( line1.charAt(0) == '#' ) {
					// 6.0.4.0 (2014/11/28) #NAME 判定で、桁数不足のエラーが発生する箇所を修正。
					if( line1.length() >= 5 && "#NAME".equalsIgnoreCase( line1.substring( 0,5 ) ) ) {
						// 6.0.4.0 (2014/11/28) #NAME 行の区切り文字は、指定の区切り文字を優先して利用する。
						final char sep ;
						if( TAB != separator && line1.indexOf( separator ) >= 0 ) {		// 6.3.9.0 (2015/11/06) バグ？
							sep = separator;
						}
						else {
							sep = TAB;
						}
						// 超イレギュラー処理。#NAME をカラム列に入れない(#NAME+区切り文字 の 6文字分、飛ばす)。
						return StringUtil.csv2Array( line1.substring( 6 ) ,sep );
					}
					else  { continue; }
				}
				else {
					final String errMsg = "#NAME が見つかる前にデータが見つかりました。";
					throw new OgRuntimeException( errMsg );
				}
			}
		}
		// 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
		catch( final CharacterCodingException ex ) {
			final String errMsg = "文字のエンコード・エラーが発生しました。" + CR
								+	"  ファイルのエンコードが指定のエンコードと異なります。" + CR
								+	" [" + infile + "] , Encode=[" + encode + "]" ;
			throw new OgCharacterException( errMsg,ex );	// 6.5.0.1 (2016/10/21)
		}
		catch( final IOException ex ) {
			final String errMsg = "ファイル読込みエラーが発生しました。" + CR
								+	" [" + infile + "] , Encode=[" + encode + "]" ;
			throw new OgRuntimeException( errMsg,ex );
		}
		return new String[0];
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		final String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "Input  File  : " + infile	+ CR
				+ TAB + "Input  Count : " + inCount	+ CR
				+ TAB + "Output Count : " + outCount ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) useNumber属性のコメント追加
	 *
	 * @return	このクラスの使用方法
	 * @og.rtnNotNull
	 */
	public String usage() {
		final StringBuilder buf = new StringBuilder( BUFFER_LARGE )
			.append( "Process_TableReaderは、ファイルから読み取った内容を、LineModel に設定後、" 	).append( CR )
			.append( "下流に渡す、FirstProcess インターフェースの実装クラスです。"					).append( CR )
			.append( CR )
			.append( "DBTableModel 形式のファイルを読み取って、各行を LineModel にセットして、"		).append( CR )
			.append( "下流(プロセスチェインのデータは上流から下流に渡されます。)に渡します。"		).append( CR )
			.append( CR )
			.append( "columns 属性は、#NAME で列カラムを外部から指定する場合に使用します。"			).append( CR )
			.append( "この属性とuseNumber属性は独立していますが、一般には、#NAME を指定"			).append( CR )
			.append( "する場合は、useNumber=\"true\"として、行番号欄は使用しますし、外部から"		).append( CR )
			.append( "指定する場合は、useNumber=\"false\"にして先頭から読み取ります。"				).append( CR )
			.append( "(自動セットではないので、必要に応じて設定してください)"						).append( CR )
			.append( "useNumber の初期値は、\"true\" です。"										).append( CR )
			.append( CR )
			.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR )
			.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR )
			.append( "繋げてください。"																).append( CR )
			.append( CR ).append( CR )
			.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_TableReader().usage() );
	}
}
