/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.xml;

/**
 * このクラスは、HybsXMLHandler に設定する Listener インターフェースです。
 * これは、HybsXMLHandler#setTagElementListener( TagElementListener ) することで、
 * ROW 毎に 内部情報を TagElement オブジェクト化し、action( TagElement )
 * が呼び出されます。この Listener を介して、１レコードずつ処理することが
 * 可能です。
 *
 * この Listener を implements したクラスを個別に作成することで、
 * XML処理を、プラグイン可能にします。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface TagElementListener  {

	/**
	 * &lt;ROWSET&gt; タグの一番最初に呼び出されます。
	 * ROWSET の属性である、table 属性と、dbid 属性 を、TagElement の
	 * get メソッドで取得できます。
	 * 取得時のキーは、それぞれ、"TABLE" と "DBID" です。
	 *
	 * @param tag タグエレメント
	 * @see org.opengion.fukurou.xml.TagElement
	 * @see HybsXMLHandler#setTagElementListener( TagElementListener )
	 */
	void actionInit( TagElement tag ) ;

	/**
	 * &lt;ROW&gt; タグの endElement 処理毎に呼び出されます。
	 * この Listener をセットすることにより、行データを取得都度、
	 * TagElement オブジェクトを作成し、このメソッドが呼び出されます。
	 *
	 * @param tag タグエレメント
	 * @see org.opengion.fukurou.xml.TagElement
	 * @see HybsXMLHandler#setTagElementListener( TagElementListener )
	 */
	void actionRow( TagElement tag ) ;

	/**
	 * &lt;EXEC_SQL&gt; タグの endElement 処理毎に呼び出されます。
	 * getBody メソッドを使用して、このタグのBODY部の文字列を取得します。
	 * この Listener をセットすることにより、EXEC_SQL データを取得都度、
	 * TagElement オブジェクトを作成し、このメソッドが呼び出されます。
	 *
	 * @param tag タグエレメント
	 * @see org.opengion.fukurou.xml.TagElement
	 * @see HybsXMLHandler#setTagElementListener( TagElementListener )
	 */
	void actionExecSQL( TagElement tag ) ;

	/**
	 * &lt;MERGE_SQL&gt; タグの endElement 処理時に呼び出されます。
	 * getBody メソッドを使用して、このタグのBODY部の文字列を取得します。
	 * MERGE_SQLタグは、マージ処理したいデータ部よりも上位に記述しておく
	 * 必要がありますが、中間部に記述しても構いません。ただし、１回のみです。
	 * このタグが現れるまでは、INSERT のみ実行されます。このタグ以降は、
	 * 一旦 UPDATE し、結果が ０件の場合は、INSERTする流れになります。
	 * 完全に INSERT のみであるデータを前半に、UPDATE/INSERTを行う
	 * データを後半に、その間に、MERGE_SQL タグを入れることで、無意味な
	 * UPDATE を避けることが可能です。
	 * この Listener をセットすることにより、MERGE_SQL データを取得都度、
	 * TagElement オブジェクトを作成し、このメソッドが呼び出されます。
	 *
	 * @param tag タグエレメント
	 * @see org.opengion.fukurou.xml.TagElement
	 * @see HybsXMLHandler#setTagElementListener( TagElementListener )
	 */
	void actionMergeSQL( TagElement tag ) ;
}
