/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.report;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.report.AbstractCSVPrintPointService;
import org.opengion.fukurou.util.StringUtil;

/**
 * ユニリタ「Report & Form Warehouse」に対応したCSV形式でデータを作成します。
 * 
 * CSVPrintPointService_RFWとの違いは出力ファイル名のみです。
 * CSVはシステムリソースRFW_CSV_OUTPUTDIRで指定した場所に[GRPID]_[YKNO]_[LISTID].csvで出力されます。
 *
 * @og.group 帳票システム
 * 
 * @og.rev 5.9.27.1 (2017/12/08) 新規作成
 *
 * @version  5.9.0.0
 * @author	 Masakazu Takahashi
 * @since    JDK6.0,
 */
public class CSVPrintPointService_RFW2 extends AbstractCSVPrintPointService {

	private static final String CR		= System.getProperty("line.separator");
	private final StringBuilder strCSV	= new StringBuilder();	// CSVはこれに吐く

	private static final String	csvEncode 	= HybsSystem.sys("REPORT_CSV_TEXT_ENCODE");
	
	private static final String RFW_CSV_OUTPUTDIR = HybsSystem.sys("RFW_CSV_OUTPUTDIR");
	
	private static final String RFW_EXCEL_TYPE = StringUtil.nval( HybsSystem.sys("RFW_EXCEL_TYPE"), "XLS" ) ;

	/**
	 * 発行処理
	 * ファイル出力
	 * 
	 *
	 * @return 結果 [true:正常/false:異常]
	 */
	@Override
	public boolean execute(){
		System.out.print( "CSV create ... " );
		BufferedWriter bw = null;
		boolean flg = false;

		try {
			// outdirが\\から開始される場合に、次の\もしくは/までの文字列を出力フォルダに付け足す
			// かつ、outdirからはサーバ名は削除する
			String nasName = "";
			if( outdir != null && outdir.startsWith( "\\\\" ) ){
				int spl = outdir.indexOf( "\\", 2 );
				int spl2 = outdir.indexOf( "/", 2 );
				spl = spl<0 ? outdir.length() : spl;
				spl2 = spl2<0 ? outdir.length() : spl2;
				spl = spl < spl2 ? spl : spl2;
				nasName = "_" + outdir.substring( 2, spl );
				outdir = outdir.substring(spl+1); 
			}
			
			makeheader();
			makebody();
			
			
//			bw = getWriter( RFW_CSV_OUTPUTDIR + File.separator + listid + "_" + ykno + ".csv" ,false,csvEncode);			
			
			// 汎用化も考えたが、予期せぬ出力があると困るのでBIG決め打ち。フォルダ存在しない場合はエラー
			if( dmngrp != null && dmngrp.indexOf( "BIG" ) >= 0 ){ 
				bw = getWriter( RFW_CSV_OUTPUTDIR + nasName +  "_BIG" + File.separator + grpid + "_" + ykno + "_" + listid + ".csv" ,false,csvEncode); 
			}
			else{
				bw = getWriter( RFW_CSV_OUTPUTDIR + nasName + File.separator + grpid + "_" + ykno + "_" + listid + ".csv" ,false,csvEncode); 
			}
			bw.write( strCSV.toString() );
			bw.flush();
			bw.close();

			flg = true;
			
		
			// 先頭が*のデーモングループの場合は約7秒スリープさせる=このスレッドでの連続処理をわざと遅延させる
			// 特殊対応なので決め打ち
			if( dmngrp != null && dmngrp.indexOf( "*" ) == 0 ){
				Thread.sleep(7000);
			}

		}
		catch ( Throwable ex ) {
			errMsg.append( "CSV Print Request Execution Error. " ).append( CR );
			errMsg.append( "==============================" ).append( CR );
			errMsg.append( "SYSTEM_ID=[" ).append( systemId ).append( "] , " );
			errMsg.append( "YKNO=["    ).append( ykno    ).append( "] , " );
			errMsg.append( ex.toString() );
			errMsg.append( CR );
			throw new RuntimeException( errMsg.toString(), ex );
		}
		return flg;
	}

	/**
	 * ヘッダの出力
	 * 
	 */
	private void makeheader(){
		//ヘッダデータを出力する場合はここで指定する。
		strCSV.append( "<rdstart>" ).append( CR );
		
		strCSV.append( "RDSetForm=\"" ).append(modelname).append("\"").append( CR );
		
		//5.9.3.1 (2015/12/16)
		strCSV.append( "RDSetUserName=\"" ).append(systemId).append("\"").append( CR );
		strCSV.append( "RDSetComputer=\"" ).append( listid + "_" + grpid + "_" + ykno ).append("\"").append( CR );
		strCSV.append( "RDSetDocName=\"" ).append(listid).append("\"").append( CR );
		
		String suffix = ""; // 5.9.6.0
		
		// 5.9.6.0 拡張子を自動で付ける対応を入れておく
		// PDFの場合
		if( FGRUN_PDF.equals( fgrun ) ){
			if( outdir != null && outdir.indexOf(".") < 0 ){
				suffix = ".pdf";
			}
			
			strCSV.append( "RDSetOutputMode=PDF" ).append( CR );
			strCSV.append( "RDSetOutputFileName=\"" ).append( outdir ).append( suffix ).append("\"").append( CR );
		}
		// Excel(XLS)
		else if( FGRUN_EXCEL.equals(fgrun) ){
			if( outdir != null && outdir.indexOf(".") < 0 ){
				suffix = ".xls";
			}
			strCSV.append( "RDSetOutputMode=" + RFW_EXCEL_TYPE ).append( CR );
			strCSV.append( "RDSetOutputFileName=\"" ).append( outdir ).append( suffix ).append("\"").append( CR );
		}
		else if( FGRUN_EXCEL2.equals(fgrun) ){
			if( outdir != null && outdir.indexOf(".") < 0 ){
				suffix = ".xlsx";
			}
			strCSV.append( "RDSetOutputMode=XLSX" ).append( CR );
			strCSV.append( "RDSetOutputFileName=\"" ).append( outdir ).append( suffix ).append("\"").append( CR );
		}
		// 印刷
		else{
			strCSV.append( "RDSetOutputMode=SPOOL" ).append( CR );
			strCSV.append( "RDSetOutputPrinter=\"" ).append(prtid).append( "\"" ).append( CR );
		}

		if( option != null && option.length() > 0 ){
			strCSV.append( option ).append( CR ); // 5.9.3.0 (2015/12/04)
		}
		
		strCSV.append( "<rdend>" ).append( CR );
		
		//１行目にカラム名を出力します。クウォートで囲わない。
		// メインテーブルはNULLではない
		for( int clmNo=0; clmNo<table.getColumnCount(); clmNo++ ) {
			// 先頭以外はカンマを付ける
			if( clmNo > 0 ){ strCSV.append( "," ); } 
			strCSV.append( table.getColumnName( clmNo ));
		}
		if( tableH != null){
			for( int clmNo=0; clmNo<tableH.getColumnCount(); clmNo++ ) {
				strCSV.append( "," ); 
				strCSV.append("H_").append( tableH.getColumnName( clmNo ));
			}
		}
		if( tableF != null){
			for( int clmNo=0; clmNo<tableF.getColumnCount(); clmNo++ ) {
				strCSV.append( "," ); 
				strCSV.append("F_").append( tableF.getColumnName( clmNo ));
			}
		}
		strCSV.append( CR );
	}



	/**
	 * 本体の出力を行います
	 * HTMLエスケープされている場合は戻します
	 */
	private void makebody(){

		for( int rowNo=0; rowNo<table.getRowCount(); rowNo++ ) {
			// カラム単位の処理
			for( int clmNo=0; clmNo<table.getColumnCount(); clmNo++ ) {
				// 先頭以外はカンマを付ける
				if( clmNo > 0 ){ strCSV.append( "," ); } 
				// 原則全てダブルクウォートで囲う
				// 5.9.8.2 (2016/05/16) 但し、先頭カラムが制御コードである//EOR//の場合のみ囲わない
				if( clmNo == 0 && "//EOR//".equals( table.getValue( rowNo, clmNo )) ){
					strCSV.append( table.getValue( rowNo, clmNo ) );
				}
				else{
					strCSV.append("\"").append( StringUtil.replace( StringUtil.getReplaceEscape( table.getValue( rowNo, clmNo )) ,"\"","\"\"" ) ).append("\"");
				}
			}
			
			//ヘッダ、フッタは毎行に必ず付加します。
			//例え複数行あったとしても先頭行のみ有効です
			//ヘッダ
			if( tableH != null){
				int rowNoH=0;
				for( int clmNo=0; clmNo<tableH.getColumnCount(); clmNo++ ) {
					// 必ずカンマを付ける
					strCSV.append( "," ); 
					// 全てダブルクウォートで囲う
					strCSV.append("\"").append( StringUtil.replace( StringUtil.getReplaceEscape( tableH.getValue( rowNoH, clmNo )) ,"\"","\"\"" ) ).append("\"");
				}
			}
			
			//フッタ
			if( tableF != null ){
				int rowNoF=0;
				for( int clmNo=0; clmNo<tableF.getColumnCount(); clmNo++ ) {
					// 必ずカンマを付ける
					strCSV.append( "," ); 
					// 全てダブルクウォートで囲う
					strCSV.append("\"").append( StringUtil.replace( StringUtil.getReplaceEscape( tableF.getValue( rowNoF, clmNo )) ,"\"","\"\"" ) ).append("\"");
				}
			}

			strCSV.append( CR );
		}
	}


	/**
	 * ファイル書き込み用のライターを返します。
	 *
	 * @param fileName ファイル名
	 * @param append アベンドするか
	 * @param encode エンコード
	 *
	 * @return ライター
	 */
	private BufferedWriter getWriter( final String fileName, final boolean append, final String encode) {
		File file = new File ( fileName );
		BufferedWriter bw;

		try {
			bw = new BufferedWriter( new OutputStreamWriter( new FileOutputStream( file, append ), encode ) );
		}
		catch ( UnsupportedEncodingException ex ) {
			errMsg.append( "[ERROR] Input File is written by Unsupported Encoding" );
			throw new HybsSystemException( ex );
		}
		catch ( FileNotFoundException ex ) {
			errMsg.append( "[ERROR] File not Found" );
			throw new HybsSystemException( ex );
		}
		return bw;
	}

}
