/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report2;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;

import org.opengion.hayabusa.common.HybsSystemException;

/**
 * OpenOfficeのプロセスを表すクラスです。
 *
 * このクラスでは、TCPによりプロセスに接続を行います。
 * 基本的には、パイプ名による接続({@link SOfficeProcess})を利用すべきですが、
 * x64環境で、64Bit版のJavaを起動した場合、パイプ接続では、UnsatisfiedLinkErrorが発生します。
 * このような場合では、TCP接続を利用することで、上記エラーを回避することができます。
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public final class SOfficeProcessTcp extends SOfficeProcess {

	private static final boolean[] ports 	= new boolean[512];
	private static final Object lock		= new Object();

	private final int initPort;
	private final int thisPort;

	/**
	 * コンストラクタです。
	 *
	 * @param	id			プロセスID
	 * @param	initPort	初期ポート
	 */
	protected SOfficeProcessTcp( final String id, final int initPort ) {
		super( id );

		this.initPort = initPort;
		this.thisPort = getThisPort();
	}

	/**
	 * TCP接続ポート番号を取得します。
	 *
	 * @return TCP接続ポート番号
	 */
	private int getThisPort() {
		try {
			Thread.sleep( 100 ); // 切断後すぐにopenされると、ポートチェックで引っかかるため100msWAIT
		}
		catch( InterruptedException ex ) {
			// ここの Exception は、無視します。
		}

		int port = -1;
		synchronized( lock ) {
			for( int i=0; i<ports.length; i++ ) {
				if( !ports[i] ) {
					if( checkPort( initPort + i ) ) {
//						System.out.println( "port=" + ( initPort + i ) + "を使用済みにマークしました。" );
						ports[i] = true;
//						System.out.println( "port=" + ( initPort + i ) + "を使用します。" );
						port = initPort + i;
						break;
					}
//					else {
//						System.out.println( "port=" + ( initPort + i ) + "は使用中のためスキップします。" );
//					}
				}
			}
		}
		if( port < 0 ) {
			throw new HybsSystemException( "TCP接続ポートを取得することができません" );
		}

		return port;
	}

	/**
	 * 引数のポートが使用中かどうかを調べます。
	 *
	 * @param port ポート番号
	 *
	 * @return	使用中かどうか
	 */
	private boolean checkPort( final int port ) {
		boolean flg = false;
		Socket sk = null;
		try {
			sk = new Socket();
			sk.connect( new InetSocketAddress( "localhost", port ) );
		}
		catch( IOException ex ) {
			flg = true;
		}
		finally {
			try {
//				sk.close();
				if( sk != null ) { sk.close(); }	// 5.5.2.6 (2012/05/25) findbugs対応
			}
			catch( IOException ex ) {
				ex.printStackTrace();
			}
		}
		return flg;
	}

	/**
	 * Pipe名をキーにOpenOfficeのプロセスに接続するための文字列を生成します。
	 * ※TCP接続の場合、キーのPipe名は無視され、内部的に管理されるポート番号一覧より
	 *   接続ポートを取得します。
	 *
	 * @param key Pipe名(無視されます)
	 *
	 * @return 接続文字列
	 */
	@Override
	protected String getConnParam( final String key ) {
		System.out.println( "[INFO]OOo:TCP Connection Start,port=" + thisPort );
//		return "uno:socket,host=localhost,tcpNoDelay=1,port=" + String.valueOf( thisPort ) + ";urp;StarOffice.ComponentContext";
		return "uno:socket,host=localhost,tcpNoDelay=1,port=" + thisPort + ";urp;StarOffice.ComponentContext";
	}

	/**
	 * Pipe名をキーにOpenOfficeのプロセスを生成するためのパラメーター文字列を生成します。
	 * ※TCP接続の場合、キーのPipe名は無視され、内部的に管理されるポート番号一覧より
	 *   接続ポートを取得します。
	 *
	 * @param key Pipe名(無視されます)
	 *
	 * @return プロセス生成パラメーター
	 */
	@Override
	protected String getProcParam( final String key ) {
//		return "-accept=socket,host=localhost,port=" + String.valueOf( thisPort ) + ";urp;";
		return "-accept=socket,host=localhost,port=" + thisPort + ";urp;";
	}

	/**
	 * プロセスを終了します。
	 * また、同時に環境設定用のファイルも削除します。
	 * ここでは、プロセスを終了すると同時に、そのプロセスのポート番号を開放し、
	 * 次に起動されるプロセスで利用できるようにします。
	 */
	@Override
	public void close() {
		super.close();
		synchronized( lock ) {
			ports[thisPort-initPort] = false;
		}
//		System.out.println( "[INFO]OOo:TCP Connection End(Release),port=" + String.valueOf( thisPort ) );
		System.out.println( "[INFO]OOo:TCP Connection End(Release),port=" + thisPort );
	}
}

