/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.db;

import java.util.Locale;

/**
 * 各データベースに対応するenum名を返します。
 * 主に、各データベースにおける関数名の差異を吸収するためのenumです。
 * 本来は、互換性のあるファンクション以外、使用しないようにしましょう。
 * また、無ければ互換性パックなどで、ファンクションを定義してしまうのも
 * 一つの方法です。
 *
 * <table border="1" frame="box" rules="all" >
 *  <caption>各データベースにおける関数</caption>
 *  <tr><th>データベース名 </th><th>連結</th><th>部分文字列</th></tr>
 *  <tr><td>{&#064;DBF.XXX}</td><td>CON </td><td>SUBSTR    </td></tr>
 *  <tr><td>ORACLE         </td><td>||  </td><td>SUBSTR    </td></tr>
 *  <tr><td>HSQL           </td><td>||  </td><td>SUBSTR    </td></tr>
 *  <tr><td>POSTGRES       </td><td>||  </td><td>SUBSTR    </td></tr>
 *  <tr><td>MYSQL          </td><td>||  </td><td>SUBSTR    </td></tr>
 *  <tr><td>SQLSERVER      </td><td>+   </td><td>SUBSTRING </td></tr>
 *  <tr><td>FIREBIRD       </td><td>||  </td><td>SUBSTR    </td></tr>
 *  <tr><td>CACHE          </td><td>||  </td><td>SUBSTRING </td></tr>
 * </table>
 *
 * @og.rev 5.1.4.0 (2010/03/01) 新規作成
 * @og.rev 5.8.5.0 (2015/03/06) CACHE追加
 *
 * @version  5.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public enum DBFunctionName {
	// 引数付きenum定義：ここに、必要な関数が増えるたびに、追加していきます。
	//			  CON	SUBSTR
	  ORACLE	( "||","SUBSTR" )
	, HSQL		( "||","SUBSTR" )
	, POSTGRES	( "||","SUBSTR" )
	, MYSQL		( "||","SUBSTR" )
	, SQLSERVER	( "+" ,"SUBSTRING" )
	, FIREBIRD	( "||","SUBSTR" ) 
	, CACHE		( "||","SUBSTRING" ) ; 

	private final String dbfCON ;
	private final String dbfSUBSTR ;

	/**
	 * コンストラクター(enum の場合は、private宣言される)
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) 新規作成
	 *
	 * @param	con   	第一引数にて指定(CON)
	 * @param	substr	第一引数にて指定(SUBSTR)
	 */
	private DBFunctionName( final String con , final String substr ) {
		dbfCON    = con;
		dbfSUBSTR = substr;
	}

	/**
	 * 共通ファンクションに対応するデータベース個別のファンクション名を返します。
	 *
	 * 現時点では、NAME,CON,SUBSTR のみ使用できます。
	 *
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) 新規作成
	 *
	 * @param   func 共通ファンクション
	 *
	 * @return  ファンクション名
	 */
	public String getFunctionName( final String func ) {
		if( "NAME".equals(   func ) ) { return toString();	}
		if( "CON".equals(    func ) ) { return dbfCON;		}
		if( "SUBSTR".equals( func ) ) { return dbfSUBSTR;	}

		return func;
	}

	/**
	 * シーケンス名よりシーケンスオブジェクトを検索し、次の値を取り出します。
	 * DBに対するシーケンスオブジェクトは予め作成されている必要があります。
	 *
	 * また、MySQLの場合は、シーケンスオブジェクトが実装されていないため、
	 * 内部的には、引数のシーケンス名と同じ名前のテーブルから、Integer型の
	 * "SEQID"という項目名を検索することにより、シーケンスをエミュレートしています。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規追加
	 * @og.rev 5.8.5.0 (2015/03/06) CACHE追加
	 * @og.rev 5.9.31.1 (2018/04/13) DBID追加対応
	 *
	 * @param seqName シーケンス名
	 * @param tran トランザクション
	 *
	 * @return シーケンス番号
	 */
	public int getSequence( final String seqName, final Transaction tran ) {
		return getSequence(seqName, tran, null);
	}
	
	/**
	 * シーケンス名よりシーケンスオブジェクトを検索し、次の値を取り出します。
	 * DBに対するシーケンスオブジェクトは予め作成されている必要があります。
	 *
	 * また、MySQLの場合は、シーケンスオブジェクトが実装されていないため、
	 * 内部的には、引数のシーケンス名と同じ名前のテーブルから、Integer型の
	 * "SEQID"という項目名を検索することにより、シーケンスをエミュレートしています。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規追加
	 * @og.rev 5.8.5.0 (2015/03/06) CACHE追加
	 * @og.rev 5.9.31.1 (2018/04/13) SQL文作成を分離
	 *
	 * @param seqName シーケンス名
	 * @param tran トランザクション
	 * @param DBID DBID
	 *
	 * @return シーケンス番号
	 */
	public int getSequence( final String seqName, final Transaction tran, final String DBID ) {
		String sql = null;
		String[][] rtn = null;
		switch ( this ) {
			case ORACLE:
				sql = "select " + seqName + ".nextval from dual";
				break;
			case HSQL:
				sql = "select next value for " + seqName + " from dual";
				break;
			case POSTGRES:
				sql = "select nextval('" + seqName + "')";
				break;
			case MYSQL:
				sql = "update " + seqName + " set SEQID = last_insert_id(SEQID+1)";
				DBUtil.dbExecute( sql, new String[0], tran );
				sql = "select last_insert_id()";
				break;
			case SQLSERVER:
				throw new RuntimeException( "現在、SQLSERVERではシーケンス機能はサポートされていません。" );
			case FIREBIRD:
				sql = "select gen_id(" + seqName + ", 1) from rdb$database";
				break;
			case CACHE:
				throw new RuntimeException( "現在、CACHEではシーケンス機能はサポートされていません。" );
			default:
				throw new RuntimeException( "現在、このデータベースではシーケンス機能はサポートされていません。" );
		}

		rtn = DBUtil.dbExecute( sql, new String[0], tran, DBID );
		return Integer.valueOf( rtn[0][0] );
	}

	/**
	 * 各データベースに対応するenum名を返します。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) 新規作成
	 * @og.rev 5.8.5.0 (2015/03/06) CACHE追加
	 * @og.rev 5.9.19.0 (2017/04/07) Azure対応。基本的にFunction等はDB種別に依存するはず。
	 *
	 * @param   dbName データベース名
	 *
	 * @return  データベースに対応するenum名
	 */
	public static DBFunctionName getDBName( final String dbName ) {
//		String dbn = dbName.toUpperCase( Locale.JAPAN );
		String dbn = DBUtil.getDBType(dbName).toUpperCase( Locale.JAPAN );

		if( 	 dbn.indexOf( "ORACLE"		) >= 0 ) { return DBFunctionName.ORACLE;	}
		else if( dbn.indexOf( "HSQL"		) >= 0 ) { return DBFunctionName.HSQL;		}
		else if( dbn.indexOf( "POSTGRES"	) >= 0 ) { return DBFunctionName.POSTGRES;	}
		else if( dbn.indexOf( "MYSQL"		) >= 0 ) { return DBFunctionName.MYSQL;		}
		else if( dbn.indexOf( "SQLSERVER"	) >= 0 ) { return DBFunctionName.SQLSERVER;	}
		else if( dbn.indexOf( "FIREBIRD"	) >= 0 ) { return DBFunctionName.FIREBIRD;	}
		else if( dbn.indexOf( "CACHE"		) >= 0 ) { return DBFunctionName.CACHE;	}

		final String errMsg = "初期化時に、指定の dbName キーが存在しません。"
						+ "[" + dbn + "]" ;

		throw new RuntimeException( errMsg );
	}

	/**
	 * 各データベースに対応するファンクション名を返します。
	 *
	 * @og.rev 4.3.8.0 (2009/08/01) SUBSTRを追加
	 * @og.rev 5.1.2.0 (2010/01/01) MySQL対応,SUBSTRB廃止(帳票データの分割の内部処理化に伴う)
	 * @og.rev 5.1.4.0 (2010/03/01) データベース名 ではなく、dbid で判断するように変更
	 * @og.rev 5.7.7.2 (2014/06/20) DBF.NAME 時の処理の簡素化
	 * @og.rev 5.9.19.1 (2017/04/14) DBF.TYPE追加
	 *
	 * @param   func ファンクション名(定義文字)
	 * @param   dbid 接続先ID
	 *
	 * @return  実ファンクション名
	 */
	public static String getFunctionName( final String func ,final String dbid ) {
//		DBFunctionName dbName = DBFunctionName.getDBName( ConnectionFactory.getDBName( dbid ) );

		// 5.7.7.2 (2014/06/20) DBF.NAME 時の処理の簡素化
		String dbName = ConnectionFactory.getDBName( dbid );
		if( "NAME".equals( func ) ) { return dbName; }
		else if( "TYPE".equals( func ) ){ return  DBUtil.getDBType(dbName); }
		else {
			return getDBName( dbName ).getFunctionName( func );
		}

//		return dbName.getFunctionName( func );
	}
}
