/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

/**
 * データのコード情報を取り扱う共通インターフェースです。
 *
 * コードのキーとラベルの情報から、HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したり
 * します。
 *
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface Selection {
	static final String NO_VALUE_OPTION = "<option value=\"\" ></option>" ;	// 3.5.5.7 (2004/05/10)

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このメソッドでは、ラベル(短)が設定されている場合でも、これを使用せずに必ずラベル(長)を使用します。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能を指定する seqFlag を導入
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 *
	 * @return  オプションタグ
	 */
	String getOption( String selectValue,boolean seqFlag ) ;

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このメソッドでは、引数のuseShortLabelがtrueに指定された場合に、ラベル(短)をベースとした
	 * ツールチップ表示を行います。
	 *
	 * @og.rev 5.1.3.0 (2010/02/01) 追加
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 * @param   useShortLabel ラベル(短)をベースとしたオプション表示を行うかどうか。
	 *
	 * @return  オプションタグ
	 */
	String getOption( String selectValue,boolean seqFlag, boolean useShortLabel ) ;

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * getValueLabel( XX ) は、getValueLabel( XX,false ) と同じです。
	 *
	 * @param   selectValue 選択肢の値
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String,boolean )
	 */
	String getValueLabel( String selectValue ) ;

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) を追加
	 *
	 * @param	selectValue	選択肢の値
	 * @param	flag	短縮ラベル使用有無 [true:使用する/false:しない]
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	String getValueLabel( String selectValue,boolean flag ) ;

	/**
	 * 初期値が選択済みの 選択肢(ラジオボタン)を返します。
	 * このラジオボタンは、引数の値を初期値とするラジオボタンタグを返します。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) 新規作成
	 * @og.rev 3.8.6.0 (2006/09/29) useLabel 属性 追加
	 *
	 * @param   name  name属性になるキー文字列
	 * @param   selectValue  選択されている値
	 * @param   useLabel     ラベル表示の有無 [true:有/false:無]
	 *
	 * @return  ラジオボタンタグ
	 */
	String getRadio( String name, String selectValue, boolean useLabel ) ;

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 *
	 * @og.rev 3.8.6.0 (2006/09/29) 新規作成
	 *
	 * @param   selectValue 選択肢の値
	 *
	 * @return  選択肢のラベル
	 */
	String getRadioLabel( String selectValue ) ;

	/**
	 * マルチ・キーセレクトを使用するかどうかを返します。
	 * true：使用する。false:使用しない です。
	 * ただし、実際に使用するかどうかは、HTML出力時に決めることが出来ます。
	 * ここでは、USE_MULTI_KEY_SELECT が true で、USE_SIZE(=20)以上の場合に
	 * true を返します。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 新規作成
	 *
	 * @return  選択リストで、マルチ・キーセレクトを使用するかどうか(true:使用する)
	 */
	boolean useMultiSelect() ;

	/**
	 * オブジェクトのキャッシュが時間切れかどうかを返します。
	 * キャッシュが時間切れ(無効)であれば、true を、有効であれば、
	 * false を返します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 *
	 * @return  キャッシュが時間切れなら true
	 */
	boolean isTimeOver() ;
}
