/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.StringFormat;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.security.HybsCryptography;

/**
 * 動的プルダウンなどで利用されるイベントカラムの各種情報を保持するための
 * 管理クラスです。
 *
 * 内容を置き換えるカラム(子カラム)の名前をキーに、イベントカラム(親カラム)や、
 * イベント発行時の処理URL等を管理します。
 *
 * これらの情報は、ColumnTagやSelectTag、ViewFormTagなどで登録され、その結果を
 * JavaScriptのコードとして出力します。(common/eventColumn.jsp)
 *
 * ここで出力された情報をイベント発行時に、JavaScriptが参照し、処理URLに渡す
 * ことで、動的な項目の入れ替えを実現しています。
 *
 * @og.rev 5.1.7.0 (2010/06/01) 新規追加
 * @og.rev 5.9.0.1 (2015/09/11) アンダースコア対応
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public class DBEventColumn {

	private static final String EVENT_COLUMN_URL = HybsSystem.sys( "JSP" ) + "/" + HybsSystem.sys( "EVENT_COLUMN_URL" );

	final String name;			// 内容を置き換えるカラム(子カラム)
	final String evCol;			// イベントカラム(親カラム・カンマ区切り)
	final String eventUrl;		// イベント発行時の処理URL
	final String renderer;		// 子カラムのレンデラー
	final String editor;		// 子カラムのエディター
	final String rendParam;		// 子カラムの表示パラメーター
	final String editParam;		// 子カラムの編集パラメーター
	// 5.10.4.0 (2018/10/05)
	final boolean flgEncrypt;
	final Pattern trgPattern	= Pattern.compile("(?<=\\{@)[^.]+?(?=\\})");
	HybsCryptography hc			= null;

	/**
	 * 初期情報を含んだ新規オブジェクトを作成します。
	 * このコンストラクタではflgEncryptをfalseとして扱う。
	 *
	 * @og.rev 5.10.4.0 (2018/10/05) 暗号化処理追加
	 *
	 * @param name 内容を置き換えるカラム(子カラム)
	 * @param evCol イベントカラム(親カラム・カンマ区切り)
	 * @param eventUrl イベント発行時の処理URL
	 * @param renderer 子カラムのレンデラー
	 * @param editor 子カラムのエディター
	 * @param rendParam 子カラムの表示パラメーター
	 * @param editParam 子カラムの編集パラメーター
	 */
	public DBEventColumn( final String name, final String evCol, final String eventUrl,
							final String renderer, final String editor, final String rendParam, final String editParam ) {
		this.name		= name;
		this.evCol		= evCol;
		 // 5.1.9.0 (2010/08/01) 動的プルダウン不具合対応
		this.eventUrl	= ( eventUrl != null && eventUrl.length() > 0 ) ? eventUrl : EVENT_COLUMN_URL;
		this.renderer	= renderer;
		this.editor		= editor;
		this.rendParam	= rendParam;
		this.editParam	= editParam;
		// 5.10.4.0 (2018/10/05) 暗号化対応
		this.flgEncrypt = HybsSystem.sysBool("EVENT_SQL_ENCRYPT");
		if(flgEncrypt) {
			this.hc = new HybsCryptography();
		}
	}

	/**
	 * 内容を置き換えるカラム(子カラム)を返します。
	 *
	 * @return 内容を置き換えるカラム(子カラム)
	 */
	public String getName() { return name; }

	/**
	 * イベントカラム(親カラム・カンマ区切り)を返します。
	 *
	 * @return イベントカラム(親カラム・カンマ区切り)
	 */
	public String getEventColumn() { return evCol; }

	/**
	 * イベント発行時の処理URLを返します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) getEventUrl ⇒ getEventURL に変更
	 *
	 * @return イベント発行時の処理URL
	 */
//	public String getEventUrl() { return eventUrl; }
	public String getEventURL() { return eventUrl; }

	/**
	 * 子カラムのレンデラーを返します。
	 *
	 * @return 子カラムのレンデラー
	 */
	public String getRenderer() { return renderer; }

	/**
	 * 子カラムのエディターを返します。
	 *
	 * @return 子カラムのエディター
	 */
	public String getEditor() { return editor; }

	/**
	 * 子カラムの表示パラメーターを返します。
	 * 
	 * @og.rev 5.9.0.1 (2015/09/11) アンダースコア対応
	 * @og.rev 5.10.4.0 (2018/10/05) 暗号化処理追加
	 *
	 * @return 子カラムの表示パラメーター
	 */
	public String getRendParam() {
		StringFormat sf = new StringFormat(
							rendParam
//							,"{@" + evCol.replace( ",", "}:{@" ) + "}"
							,"{@" + evColReplace(evCol).replace( ",", "}:{@" ) + "}"
							,name );
		// 5.10.4.0 (2018/10/05)
//		return sf.format();
		if(flgEncrypt) {
			return hc.encrypt(sf.format());
		}else {
			return sf.format();
		}
	}

	/**
	 * 子カラムの編集パラメーターを返します。
	 * 
	 * @og.rev 5.9.0.1 (2015/09/11) アンダースコア対応
	 * @og.rev 5.10.4.0 (2018/10/05) 暗号化処理追加
	 *
	 * @return 子カラムの編集パラメーター
	 */
	public String getEditParam() {
		StringFormat sf = new StringFormat(
							editParam
//							,"{@" + evCol.replace( ",", "}:{@" ) + "}"
							,"{@" + evColReplace(evCol).replace( ",", "}:{@" ) + "}"
							,name );
		// 5.10.4.0 (2018/10/05)
//		return sf.format();
		if(flgEncrypt) {
			return hc.encrypt(sf.format());
		}else {
			return sf.format();
		}
	}
	
	/**
	 * カンマ区切りのカラム名から先頭のアンダースコアを外します。
	 *
	 * @og.rev 5.9.0.1 (2015/09/11) 新規作成
	 *
	 * @param inStr カンマ区切りイベントカラム
	 * @return 先頭アンダースコアを外したカンマ区切り文字列
	 */
	private String evColReplace( final String inStr ){
		final String[] to;
		if( inStr != null && inStr.indexOf( ',' ) >= 0 ) {
			to = StringUtil.csv2Array( inStr, ',' );
		}
		else {
			to = new String[] { inStr };
		}
		

		for( int i = 0; i < to.length; i++ ) {
			if( to[i].charAt( 0 ) == '_'){
				to[i] = to[i].substring( 1 );
			}
		}
		
		return org.opengion.fukurou.util.StringUtil.array2csv(to);
	}
	
	/**
	 * 文字列から{@XXX}のXXXを抜き出して、CSVに変換する。
	 * 
	 * @og.rev 5.10.4.0 (2018/10/05) 新規作成
	 *
	 * @param trg 変換対象文字列
	 * @return CSV文字列
	 */
	private String getTrgCsv(String trg) {
		if(trg == null || trg.length() == 0) {
			return "";
		}

		Matcher m = trgPattern.matcher(trg);
		StringBuilder rtn = new StringBuilder();
		while(m.find()) {
			if(rtn.length() != 0) {
				rtn.append(",");
			}
			rtn.append(m.group());
		}

		return rtn.toString();
	}

	/**
	 * レンデラーパラメータのキーリストを渡します。
	 * 複合化後の値設定に利用します。
	 *
	 * @og.rev 5.10.4.0 (2018/10/05) 新規作成
	 *
	 * @return レンデラーパラメータのキーリスト
	 */
	public String getRendParamKeys() {
		// 暗号化しない、またはrendParamが空の場合は、空文字を返す
		if(!flgEncrypt ||
				(rendParam == null || rendParam.length() == 0))
			{ return ""; }

		StringFormat sf = new StringFormat(
				rendParam
				,"{@" + evColReplace(evCol).replace( ",", "}:{@" ) + "}"
				,name );

		return getTrgCsv(sf.format());
	}

	/**
	 * エディターパラメータのキーリストを渡します。
	 * 複合化後の値設定に利用します。
	 *
	 * @og.rev 5.10.4.0 (2018/10/05) 新規作成
	 *
	 * @return エディターパラメータのキーリスト
	 */
	public String getEditParamKeys() {
		// 暗号化しない、またはrendParamが空の場合は、空文字を返す
		if(!flgEncrypt ||
				(editParam == null || editParam.length() == 0))
			{ return ""; }

		StringFormat sf = new StringFormat(
				editParam
				,"{@" + evColReplace(evCol).replace( ",", "}:{@" ) + "}"
				,name );

		return getTrgCsv(sf.format());
	}

	/**
	 * 暗号化フラグを渡します。
	 *
	 * @og.rev 5.10.4.0 (2018/10/05) 新規作成
	 *
	 * @return 暗号化フラグ
	 */
	public boolean isFlgEncrypt() {
		return flgEncrypt;
	}

}
